/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.AmazonClientException;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.Statistic;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class S3AFastOutputStream
extends OutputStream {
    private static final Logger LOG = S3AFileSystem.LOG;
    private final String key;
    private final String bucket;
    private final AmazonS3Client client;
    private final int partSize;
    private final int multiPartThreshold;
    private final S3AFileSystem fs;
    private final CannedAccessControlList cannedACL;
    private final ProgressListener progressListener;
    private final ListeningExecutorService executorService;
    private MultiPartUpload multiPartUpload;
    private boolean closed;
    private ByteArrayOutputStream buffer;
    private int bufferLimit;

    public S3AFastOutputStream(AmazonS3Client client, S3AFileSystem fs, String bucket, String key, Progressable progress, CannedAccessControlList cannedACL, long partSize, long multiPartThreshold, ExecutorService threadPoolExecutor) throws IOException {
        this.bucket = bucket;
        this.key = key;
        this.client = client;
        this.fs = fs;
        this.cannedACL = cannedACL;
        if (partSize > Integer.MAX_VALUE) {
            this.partSize = Integer.MAX_VALUE;
            LOG.warn("s3a: MULTIPART_SIZE capped to ~2.14GB (maximum allowed size when using 'FAST_UPLOAD = true')");
        } else {
            this.partSize = (int)partSize;
        }
        if (multiPartThreshold > Integer.MAX_VALUE) {
            this.multiPartThreshold = Integer.MAX_VALUE;
            LOG.warn("s3a: MIN_MULTIPART_THRESHOLD capped to ~2.14GB (maximum allowed size when using 'FAST_UPLOAD = true')");
        } else {
            this.multiPartThreshold = (int)multiPartThreshold;
        }
        this.bufferLimit = this.multiPartThreshold;
        this.closed = false;
        int initialBufferSize = this.fs.getConf().getInt("fs.s3a.fast.buffer.size", 0x100000);
        if (initialBufferSize < 0) {
            LOG.warn("s3a: FAST_BUFFER_SIZE should be a positive number. Using default value");
            initialBufferSize = 0x100000;
        } else if (initialBufferSize > this.bufferLimit) {
            LOG.warn("s3a: automatically adjusting FAST_BUFFER_SIZE to not exceed MIN_MULTIPART_THRESHOLD");
            initialBufferSize = this.bufferLimit;
        }
        this.buffer = new ByteArrayOutputStream(initialBufferSize);
        this.executorService = MoreExecutors.listeningDecorator((ExecutorService)threadPoolExecutor);
        this.multiPartUpload = null;
        this.progressListener = new ProgressableListener(progress);
        LOG.debug("Initialized S3AFastOutputStream for bucket '{}' key '{}'", (Object)bucket, (Object)key);
    }

    @Override
    public synchronized void write(int b) throws IOException {
        this.buffer.write(b);
        if (this.buffer.size() == this.bufferLimit) {
            this.uploadBuffer();
        }
    }

    @Override
    public synchronized void write(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        if (this.buffer.size() + len < this.bufferLimit) {
            this.buffer.write(b, off, len);
        } else {
            int firstPart = this.bufferLimit - this.buffer.size();
            this.buffer.write(b, off, firstPart);
            this.uploadBuffer();
            this.write(b, off + firstPart, len - firstPart);
        }
    }

    private synchronized void uploadBuffer() throws IOException {
        if (this.multiPartUpload == null) {
            this.multiPartUpload = this.initiateMultiPartUpload();
            byte[] allBytes = this.buffer.toByteArray();
            this.buffer = null;
            LOG.debug("Total length of initial buffer: {}", (Object)allBytes.length);
            int processedPos = 0;
            while (this.multiPartThreshold - processedPos >= this.partSize) {
                LOG.debug("Initial buffer: processing from byte {} to byte {}", (Object)processedPos, (Object)(processedPos + this.partSize - 1));
                this.multiPartUpload.uploadPartAsync(new ByteArrayInputStream(allBytes, processedPos, this.partSize), this.partSize);
                processedPos += this.partSize;
            }
            this.bufferLimit = this.partSize;
            this.buffer = new ByteArrayOutputStream(this.bufferLimit);
            this.buffer.write(allBytes, processedPos, this.multiPartThreshold - processedPos);
        } else {
            this.multiPartUpload.uploadPartAsync(new ByteArrayInputStream(this.buffer.toByteArray()), this.partSize);
            this.buffer.reset();
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try {
            if (this.multiPartUpload == null) {
                this.putObject();
            } else {
                int size = this.buffer.size();
                if (size > 0) {
                    this.fs.incrementPutStartStatistics(size);
                    this.multiPartUpload.uploadPartAsync(new ByteArrayInputStream(this.buffer.toByteArray()), size);
                }
                List partETags = this.multiPartUpload.waitForAllPartUploads();
                this.multiPartUpload.complete(partETags);
            }
            this.fs.finishedWrite(this.key);
            LOG.debug("Upload complete for bucket '{}' key '{}'", (Object)this.bucket, (Object)this.key);
        }
        finally {
            this.buffer = null;
            super.close();
        }
    }

    private ObjectMetadata createDefaultMetadata() {
        return this.fs.newObjectMetadata();
    }

    private MultiPartUpload initiateMultiPartUpload() throws IOException {
        InitiateMultipartUploadRequest initiateMPURequest = new InitiateMultipartUploadRequest(this.bucket, this.key, this.createDefaultMetadata());
        initiateMPURequest.setCannedACL(this.cannedACL);
        try {
            return new MultiPartUpload(this.client.initiateMultipartUpload(initiateMPURequest).getUploadId());
        }
        catch (AmazonClientException ace) {
            throw S3AUtils.translateException("initiate MultiPartUpload", this.key, ace);
        }
    }

    private void putObject() throws IOException {
        LOG.debug("Executing regular upload for bucket '{}' key '{}'", (Object)this.bucket, (Object)this.key);
        ObjectMetadata om = this.createDefaultMetadata();
        final int size = this.buffer.size();
        om.setContentLength((long)size);
        final PutObjectRequest putObjectRequest = this.fs.newPutObjectRequest(this.key, om, new ByteArrayInputStream(this.buffer.toByteArray()));
        putObjectRequest.setGeneralProgressListener(this.progressListener);
        ListenableFuture putObjectResult = this.executorService.submit((Callable)new Callable<PutObjectResult>(){

            @Override
            public PutObjectResult call() throws Exception {
                S3AFastOutputStream.this.fs.incrementPutStartStatistics(size);
                return S3AFastOutputStream.this.client.putObject(putObjectRequest);
            }
        });
        try {
            putObjectResult.get();
        }
        catch (InterruptedException ie) {
            LOG.warn("Interrupted object upload: {}", (Object)ie, (Object)ie);
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException ee) {
            throw S3AUtils.extractException("regular upload", this.key, ee);
        }
    }

    private static class ProgressableListener
    implements ProgressListener {
        private final Progressable progress;

        public ProgressableListener(Progressable progress) {
            this.progress = progress;
        }

        public void progressChanged(ProgressEvent progressEvent) {
            if (this.progress != null) {
                this.progress.progress();
            }
        }
    }

    private class MultiPartUpload {
        private final String uploadId;
        private final List<ListenableFuture<PartETag>> partETagsFutures;

        public MultiPartUpload(String uploadId) {
            this.uploadId = uploadId;
            this.partETagsFutures = new ArrayList<ListenableFuture<PartETag>>();
            LOG.debug("Initiated multi-part upload for bucket '{}' key '{}' with id '{}'", new Object[]{S3AFastOutputStream.this.bucket, S3AFastOutputStream.this.key, uploadId});
        }

        private void uploadPartAsync(ByteArrayInputStream inputStream, int partSize) {
            final int currentPartNumber = this.partETagsFutures.size() + 1;
            final UploadPartRequest request = new UploadPartRequest().withBucketName(S3AFastOutputStream.this.bucket).withKey(S3AFastOutputStream.this.key).withUploadId(this.uploadId).withInputStream((InputStream)inputStream).withPartNumber(currentPartNumber).withPartSize((long)partSize);
            request.setGeneralProgressListener(S3AFastOutputStream.this.progressListener);
            ListenableFuture partETagFuture = S3AFastOutputStream.this.executorService.submit((Callable)new Callable<PartETag>(){

                @Override
                public PartETag call() throws Exception {
                    LOG.debug("Uploading part {} for id '{}'", (Object)currentPartNumber, (Object)MultiPartUpload.this.uploadId);
                    return S3AFastOutputStream.this.fs.uploadPart(request).getPartETag();
                }
            });
            this.partETagsFutures.add((ListenableFuture<PartETag>)partETagFuture);
        }

        private List<PartETag> waitForAllPartUploads() throws IOException {
            try {
                return (List)Futures.allAsList(this.partETagsFutures).get();
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted partUpload: {}", (Object)ie, (Object)ie);
                Thread.currentThread().interrupt();
                return null;
            }
            catch (ExecutionException ee) {
                for (ListenableFuture<PartETag> future : this.partETagsFutures) {
                    future.cancel(true);
                }
                this.abort();
                throw S3AUtils.extractException("Multi-part upload with id '" + this.uploadId + "'", S3AFastOutputStream.this.key, ee);
            }
        }

        private void complete(List<PartETag> partETags) throws IOException {
            try {
                LOG.debug("Completing multi-part upload for key '{}', id '{}'", (Object)S3AFastOutputStream.this.key, (Object)this.uploadId);
                S3AFastOutputStream.this.client.completeMultipartUpload(new CompleteMultipartUploadRequest(S3AFastOutputStream.this.bucket, S3AFastOutputStream.this.key, this.uploadId, partETags));
            }
            catch (AmazonClientException e) {
                throw S3AUtils.translateException("Completing multi-part upload", S3AFastOutputStream.this.key, e);
            }
        }

        public void abort() {
            LOG.warn("Aborting multi-part upload with id '{}'", (Object)this.uploadId);
            try {
                S3AFastOutputStream.this.fs.incrementStatistic(Statistic.OBJECT_MULTIPART_UPLOAD_ABORTED);
                S3AFastOutputStream.this.client.abortMultipartUpload(new AbortMultipartUploadRequest(S3AFastOutputStream.this.bucket, S3AFastOutputStream.this.key, this.uploadId));
            }
            catch (Exception e2) {
                LOG.warn("Unable to abort multipart upload, you may need to purge  uploaded parts: {}", (Object)e2, (Object)e2);
            }
        }
    }
}

