/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.EnvironmentVariableCredentialsProvider;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.file.AccessDeniedException;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.s3a.AWSClientIOException;
import org.apache.hadoop.fs.s3a.AWSCredentialProviderList;
import org.apache.hadoop.fs.s3a.AWSS3IOException;
import org.apache.hadoop.fs.s3a.AWSServiceIOException;
import org.apache.hadoop.fs.s3a.BasicAWSCredentialsProvider;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3native.S3xLoginHelper;
import org.apache.hadoop.security.ProviderUtils;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public final class S3AUtils {
    private static final Logger LOG = S3AFileSystem.LOG;
    static final String CONSTRUCTOR_EXCEPTION = "constructor exception";
    static final String INSTANTIATION_EXCEPTION = "instantiation exception";
    static final String NOT_AWS_PROVIDER = "does not implement AWSCredentialsProvider";
    static final String ENDPOINT_KEY = "Endpoint";

    private S3AUtils() {
    }

    public static IOException translateException(String operation, Path path, AmazonClientException exception) {
        return S3AUtils.translateException(operation, path.toString(), exception);
    }

    public static IOException translateException(String operation, String path, AmazonClientException exception) {
        IOException ioe;
        String message = String.format("%s%s: %s", new Object[]{operation, path != null ? " on " + path : "", exception});
        if (!(exception instanceof AmazonServiceException)) {
            return new AWSClientIOException(message, exception);
        }
        AmazonServiceException ase = (AmazonServiceException)exception;
        AmazonS3Exception s3Exception = ase instanceof AmazonS3Exception ? (AmazonS3Exception)ase : null;
        int status = ase.getStatusCode();
        switch (status) {
            case 301: {
                if (s3Exception != null) {
                    if (s3Exception.getAdditionalDetails() != null && s3Exception.getAdditionalDetails().containsKey(ENDPOINT_KEY)) {
                        message = String.format("Received permanent redirect response to endpoint %s.  This likely indicates that the S3 endpoint configured in %s does not match the AWS region containing the bucket.", s3Exception.getAdditionalDetails().get(ENDPOINT_KEY), "fs.s3a.endpoint");
                    }
                    ioe = new AWSS3IOException(message, s3Exception);
                    break;
                }
                ioe = new AWSServiceIOException(message, ase);
                break;
            }
            case 401: 
            case 403: {
                ioe = new AccessDeniedException(path, null, message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 404: 
            case 410: {
                ioe = new FileNotFoundException(message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 416: {
                ioe = new EOFException(message);
                break;
            }
            default: {
                ioe = s3Exception != null ? new AWSS3IOException(message, s3Exception) : new AWSServiceIOException(message, ase);
            }
        }
        return ioe;
    }

    public static IOException extractException(String operation, String path, ExecutionException ee) {
        Throwable cause = ee.getCause();
        IOException ioe = cause instanceof AmazonClientException ? S3AUtils.translateException(operation, path, (AmazonClientException)cause) : (cause instanceof IOException ? (IOException)cause : new IOException(operation + " failed: " + cause, cause));
        return ioe;
    }

    public static String stringify(AmazonServiceException e) {
        StringBuilder builder = new StringBuilder(String.format("%s: %s error %d: %s; %s%s%n", e.getErrorType(), e.getServiceName(), e.getStatusCode(), e.getErrorCode(), e.getErrorMessage(), e.isRetryable() ? " (retryable)" : ""));
        String rawResponseContent = e.getRawResponseContent();
        if (rawResponseContent != null) {
            builder.append(rawResponseContent);
        }
        return builder.toString();
    }

    public static String stringify(AmazonS3Exception e) {
        StringBuilder builder = new StringBuilder(S3AUtils.stringify((AmazonServiceException)e));
        Map details = e.getAdditionalDetails();
        if (details != null) {
            builder.append('\n');
            for (Map.Entry d : details.entrySet()) {
                builder.append((String)d.getKey()).append('=').append((String)d.getValue()).append('\n');
            }
        }
        return builder.toString();
    }

    public static S3AFileStatus createFileStatus(Path keyPath, S3ObjectSummary summary, long blockSize) {
        if (S3AUtils.objectRepresentsDirectory(summary.getKey(), summary.getSize())) {
            return new S3AFileStatus(true, true, keyPath);
        }
        return new S3AFileStatus(summary.getSize(), S3AUtils.dateToLong(summary.getLastModified()), keyPath, blockSize);
    }

    public static boolean objectRepresentsDirectory(String name, long size) {
        return !name.isEmpty() && name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    public static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime();
    }

    public static AWSCredentialProviderList createAWSCredentialProviderSet(URI binding, Configuration conf, URI fsURI) throws IOException {
        Class[] awsClasses;
        AWSCredentialProviderList credentials = new AWSCredentialProviderList();
        try {
            awsClasses = conf.getClasses("fs.s3a.aws.credentials.provider", new Class[0]);
        }
        catch (RuntimeException e) {
            Throwable c = e.getCause() != null ? e.getCause() : e;
            throw new IOException("From option fs.s3a.aws.credentials.provider " + c, c);
        }
        if (awsClasses.length == 0) {
            S3xLoginHelper.Login creds = S3AUtils.getAWSAccessKeys(binding, conf);
            credentials.add(new BasicAWSCredentialsProvider(creds.getUser(), creds.getPassword()));
            credentials.add((AWSCredentialsProvider)new EnvironmentVariableCredentialsProvider());
            credentials.add((AWSCredentialsProvider)new InstanceProfileCredentialsProvider());
        } else {
            for (Class aClass : awsClasses) {
                credentials.add(S3AUtils.createAWSCredentialProvider(conf, aClass, fsURI));
            }
        }
        return credentials;
    }

    static AWSCredentialsProvider createAWSCredentialProvider(Configuration conf, Class<?> credClass, URI uri) throws IOException {
        AWSCredentialsProvider credentials;
        String className = credClass.getName();
        if (!AWSCredentialsProvider.class.isAssignableFrom(credClass)) {
            throw new IOException("Class " + credClass + " " + NOT_AWS_PROVIDER);
        }
        try {
            LOG.debug("Credential provider class is {}", (Object)className);
            try {
                credentials = (AWSCredentialsProvider)credClass.getDeclaredConstructor(URI.class, Configuration.class).newInstance(uri, conf);
            }
            catch (NoSuchMethodException | SecurityException e) {
                credentials = (AWSCredentialsProvider)credClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
        }
        catch (NoSuchMethodException | SecurityException e) {
            throw new IOException(String.format("%s constructor exception.  A class specified in %s must provide an accessible constructor accepting URI and Configuration, or an accessible default constructor.", className, "fs.s3a.aws.credentials.provider"), e);
        }
        catch (IllegalArgumentException | ReflectiveOperationException e) {
            throw new IOException(className + " " + INSTANTIATION_EXCEPTION + ".", e);
        }
        LOG.debug("Using {} for {}.", (Object)credentials, (Object)uri);
        return credentials;
    }

    public static S3xLoginHelper.Login getAWSAccessKeys(URI name, Configuration conf) throws IOException {
        S3xLoginHelper.Login login = S3xLoginHelper.extractLoginDetailsWithWarnings(name);
        Configuration c = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)conf, S3AFileSystem.class);
        String accessKey = S3AUtils.getPassword(c, "fs.s3a.access.key", login.getUser());
        String secretKey = S3AUtils.getPassword(c, "fs.s3a.secret.key", login.getPassword());
        return new S3xLoginHelper.Login(accessKey, secretKey);
    }

    static String getPassword(Configuration conf, String key, String val) throws IOException {
        return StringUtils.isEmpty((String)val) ? S3AUtils.lookupPassword(conf, key, "") : val;
    }

    static String lookupPassword(Configuration conf, String key, String defVal) throws IOException {
        try {
            char[] pass = conf.getPassword(key);
            return pass != null ? new String(pass).trim() : defVal;
        }
        catch (IOException ioe) {
            throw new IOException("Cannot find password option " + key, ioe);
        }
    }
}

