/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.AzureBlobFileSystemStore;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.FileSystemOperationUnhandledException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriAuthorityException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AzureServiceErrorCode;
import org.apache.hadoop.fs.azurebfs.extensions.AbfsAuthorizationException;
import org.apache.hadoop.fs.azurebfs.extensions.AbfsAuthorizer;
import org.apache.hadoop.fs.azurebfs.security.AbfsDelegationTokenManager;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsClientThrottlingIntercept;
import org.apache.hadoop.fs.azurebfs.services.AbfsInputStream;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceStability.Evolving
public class AzureBlobFileSystem
extends FileSystem {
    public static final Logger LOG = LoggerFactory.getLogger(AzureBlobFileSystem.class);
    private URI uri;
    private Path workingDir;
    private UserGroupInformation userGroupInformation;
    private String user;
    private String primaryUserGroup;
    private AzureBlobFileSystemStore abfsStore;
    private boolean isClosed;
    private boolean delegationTokenEnabled = false;
    private AbfsDelegationTokenManager delegationTokenManager;
    private AbfsAuthorizer authorizer;

    public void initialize(URI uri, Configuration configuration) throws IOException {
        uri = this.ensureAuthority(uri, configuration);
        super.initialize(uri, configuration);
        this.setConf(configuration);
        LOG.debug("Initializing AzureBlobFileSystem for {}", (Object)uri);
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.userGroupInformation = UserGroupInformation.getCurrentUser();
        this.user = this.userGroupInformation.getUserName();
        this.abfsStore = new AzureBlobFileSystemStore(uri, this.isSecureScheme(), configuration, this.userGroupInformation);
        AbfsConfiguration abfsConfiguration = this.abfsStore.getAbfsConfiguration();
        this.setWorkingDirectory(this.getHomeDirectory());
        if (abfsConfiguration.getCreateRemoteFileSystemDuringInitialization() && !this.fileSystemExists()) {
            try {
                this.createFileSystem();
            }
            catch (AzureBlobFileSystemException ex) {
                this.checkException(null, ex, AzureServiceErrorCode.FILE_SYSTEM_ALREADY_EXISTS);
            }
        }
        this.primaryUserGroup = !abfsConfiguration.getSkipUserGroupMetadataDuringInitialization() ? this.userGroupInformation.getPrimaryGroupName() : this.user;
        if (UserGroupInformation.isSecurityEnabled()) {
            this.delegationTokenEnabled = abfsConfiguration.isDelegationTokenManagerEnabled();
            if (this.delegationTokenEnabled) {
                LOG.debug("Initializing DelegationTokenManager for {}", (Object)uri);
                this.delegationTokenManager = abfsConfiguration.getDelegationTokenManager();
            }
        }
        AbfsClientThrottlingIntercept.initializeSingleton(abfsConfiguration.isAutoThrottlingEnabled());
        this.authorizer = abfsConfiguration.getAbfsAuthorizer();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("AzureBlobFileSystem{");
        sb.append("uri=").append(this.uri);
        sb.append(", user='").append(this.user).append('\'');
        sb.append(", primaryUserGroup='").append(this.primaryUserGroup).append('\'');
        sb.append('}');
        return sb.toString();
    }

    public boolean isSecureScheme() {
        return false;
    }

    public URI getUri() {
        return this.uri;
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        LOG.debug("AzureBlobFileSystem.open path: {} bufferSize: {}", (Object)path, (Object)bufferSize);
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.READ, qualifiedPath);
        try {
            AbfsInputStream inputStream = this.abfsStore.openFileForRead(qualifiedPath, this.statistics);
            return new FSDataInputStream((InputStream)((Object)inputStream));
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.debug("AzureBlobFileSystem.create path: {} permission: {} overwrite: {} bufferSize: {}", new Object[]{f, permission, overwrite, blockSize});
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            OutputStream outputStream = this.abfsStore.createFile(qualifiedPath, overwrite, permission == null ? FsPermission.getFileDefault() : permission, FsPermission.getUMask((Configuration)this.getConf()));
            return new FSDataOutputStream(outputStream, this.statistics);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path parent = f.getParent();
        FileStatus parentFileStatus = this.tryGetFileStatus(parent);
        if (parentFileStatus == null) {
            throw new FileNotFoundException("Cannot create file " + f.getName() + " because parent folder does not exist.");
        }
        return this.create(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        EnumSet<CreateFlag> createflags = EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE);
        boolean overwrite = flags.containsAll(createflags);
        return this.createNonRecursive(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream createNonRecursive(Path f, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.createNonRecursive(f, FsPermission.getFileDefault(), overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        LOG.debug("AzureBlobFileSystem.append path: {} bufferSize: {}", (Object)f.toString(), (Object)bufferSize);
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            OutputStream outputStream = this.abfsStore.openFileForWrite(qualifiedPath, false);
            return new FSDataOutputStream(outputStream, this.statistics);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public boolean rename(Path src, Path dst) throws IOException {
        LOG.debug("AzureBlobFileSystem.rename src: {} dst: {}", (Object)src.toString(), (Object)dst.toString());
        Path parentFolder = src.getParent();
        if (parentFolder == null) {
            return false;
        }
        FileStatus dstFileStatus = this.tryGetFileStatus(dst);
        try {
            String sourceFileName = src.getName();
            Path adjustedDst = dst;
            if (dstFileStatus != null) {
                if (!dstFileStatus.isDirectory()) {
                    return src.equals((Object)dst);
                }
                adjustedDst = new Path(dst, sourceFileName);
            }
            Path qualifiedSrcPath = this.makeQualified(src);
            Path qualifiedDstPath = this.makeQualified(adjustedDst);
            this.performAbfsAuthCheck(FsAction.READ_WRITE, qualifiedSrcPath, qualifiedDstPath);
            this.abfsStore.rename(qualifiedSrcPath, qualifiedDstPath);
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(src, ex, AzureServiceErrorCode.PATH_ALREADY_EXISTS, AzureServiceErrorCode.INVALID_RENAME_SOURCE_PATH, AzureServiceErrorCode.SOURCE_PATH_NOT_FOUND, AzureServiceErrorCode.INVALID_SOURCE_OR_DESTINATION_RESOURCE_TYPE, AzureServiceErrorCode.RENAME_DESTINATION_PARENT_PATH_NOT_FOUND, AzureServiceErrorCode.INTERNAL_OPERATION_ABORT);
            return false;
        }
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        LOG.debug("AzureBlobFileSystem.delete path: {} recursive: {}", (Object)f.toString(), (Object)recursive);
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        if (f.isRoot()) {
            if (!recursive) {
                return false;
            }
            return this.deleteRoot();
        }
        try {
            this.abfsStore.delete(qualifiedPath, recursive);
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, AzureServiceErrorCode.PATH_NOT_FOUND);
            return false;
        }
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        LOG.debug("AzureBlobFileSystem.listStatus path: {}", (Object)f.toString());
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.READ, qualifiedPath);
        try {
            FileStatus[] result = this.abfsStore.listStatus(qualifiedPath);
            return result;
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        LOG.debug("AzureBlobFileSystem.mkdirs path: {} permissions: {}", (Object)f, (Object)permission);
        Path parentFolder = f.getParent();
        if (parentFolder == null) {
            return true;
        }
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.createDirectory(qualifiedPath, permission == null ? FsPermission.getDirDefault() : permission, FsPermission.getUMask((Configuration)this.getConf()));
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, AzureServiceErrorCode.PATH_ALREADY_EXISTS);
            return true;
        }
    }

    public synchronized void close() throws IOException {
        if (this.isClosed) {
            return;
        }
        super.close();
        LOG.debug("AzureBlobFileSystem.close");
        this.isClosed = true;
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        LOG.debug("AzureBlobFileSystem.getFileStatus path: {}", (Object)f);
        Path qualifiedPath = this.makeQualified(f);
        this.performAbfsAuthCheck(FsAction.READ, qualifiedPath);
        try {
            return this.abfsStore.getFileStatus(qualifiedPath);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public Path makeQualified(Path path) {
        if (path != null) {
            String uriPath = path.toUri().getPath();
            path = uriPath.isEmpty() ? path : new Path(uriPath);
        }
        return super.makeQualified(path);
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir.isAbsolute() ? newDir : new Path(this.workingDir, newDir);
    }

    public String getScheme() {
        return "abfs";
    }

    public Path getHomeDirectory() {
        return this.makeQualified(new Path("/user/" + this.userGroupInformation.getShortUserName()));
    }

    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) {
        if (file == null) {
            return null;
        }
        if (start < 0L || len < 0L) {
            throw new IllegalArgumentException("Invalid start or len parameter");
        }
        if (file.getLen() < start) {
            return new BlockLocation[0];
        }
        String blobLocationHost = this.abfsStore.getAbfsConfiguration().getAzureBlockLocationHost();
        String[] name = new String[]{blobLocationHost};
        String[] host = new String[]{blobLocationHost};
        long blockSize = file.getBlockSize();
        if (blockSize <= 0L) {
            throw new IllegalArgumentException("The block size for the given file is not a positive number: " + blockSize);
        }
        int numberOfLocations = (int)(len / blockSize) + (len % blockSize == 0L ? 0 : 1);
        BlockLocation[] locations = new BlockLocation[numberOfLocations];
        for (int i = 0; i < locations.length; ++i) {
            long currentOffset = start + (long)i * blockSize;
            long currentLength = Math.min(blockSize, start + len - currentOffset);
            locations[i] = new BlockLocation(name, host, currentOffset, currentLength);
        }
        return locations;
    }

    protected void finalize() throws Throwable {
        LOG.debug("finalize() called.");
        this.close();
        super.finalize();
    }

    public String getOwnerUser() {
        return this.user;
    }

    public String getOwnerUserPrimaryGroup() {
        return this.primaryUserGroup;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean deleteRoot() throws IOException {
        LOG.debug("Deleting root content");
        ExecutorService executorService = Executors.newFixedThreadPool(10);
        try {
            FileStatus[] ls = this.listStatus(this.makeQualified(new Path(File.separator)));
            ArrayList<Future<Void>> deleteTasks = new ArrayList<Future<Void>>();
            for (final FileStatus fs : ls) {
                Future<Void> deleteTask = executorService.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        AzureBlobFileSystem.this.delete(fs.getPath(), fs.isDirectory());
                        return null;
                    }
                });
                deleteTasks.add(deleteTask);
            }
            for (final Future future : deleteTasks) {
                this.execute("deleteRoot", new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        future.get();
                        return null;
                    }
                });
            }
        }
        finally {
            executorService.shutdownNow();
        }
        return true;
    }

    public void setOwner(Path path, String owner, String group) throws IOException {
        LOG.debug("AzureBlobFileSystem.setOwner path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            super.setOwner(path, owner, group);
            return;
        }
        if ((owner == null || owner.isEmpty()) && (group == null || group.isEmpty())) {
            throw new IllegalArgumentException("A valid owner or group must be specified.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.setOwner(qualifiedPath, owner, group);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void setPermission(Path path, FsPermission permission) throws IOException {
        LOG.debug("AzureBlobFileSystem.setPermission path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            super.setPermission(path, permission);
            return;
        }
        if (permission == null) {
            throw new IllegalArgumentException("The permission can't be null");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.setPermission(qualifiedPath, permission);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void modifyAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.modifyAclEntries path: {}", (Object)path.toString());
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("modifyAclEntries is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.isEmpty()) {
            throw new IllegalArgumentException("The value of the aclSpec parameter is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.modifyAclEntries(qualifiedPath, aclSpec);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeAclEntries path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("removeAclEntries is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.isEmpty()) {
            throw new IllegalArgumentException("The aclSpec argument is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.removeAclEntries(qualifiedPath, aclSpec);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeDefaultAcl(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeDefaultAcl path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("removeDefaultAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.removeDefaultAcl(qualifiedPath);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeAcl(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeAcl path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("removeAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.removeAcl(qualifiedPath);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void setAcl(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.setAcl path: {}", (Object)path);
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("setAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.size() == 0) {
            throw new IllegalArgumentException("The aclSpec argument is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.WRITE, qualifiedPath);
        try {
            this.abfsStore.setAcl(qualifiedPath, aclSpec);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public AclStatus getAclStatus(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.getAclStatus path: {}", (Object)path.toString());
        if (!this.getIsNamespaceEnabled()) {
            throw new UnsupportedOperationException("getAclStatus is only supported by storage account with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        this.performAbfsAuthCheck(FsAction.READ, qualifiedPath);
        try {
            return this.abfsStore.getAclStatus(qualifiedPath);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(path, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    private FileStatus tryGetFileStatus(Path f) {
        try {
            return this.getFileStatus(f);
        }
        catch (IOException ex) {
            LOG.debug("File not found {}", (Object)f);
            return null;
        }
    }

    private boolean fileSystemExists() throws IOException {
        LOG.debug("AzureBlobFileSystem.fileSystemExists uri: {}", (Object)this.uri);
        try {
            this.abfsStore.getFilesystemProperties();
        }
        catch (AzureBlobFileSystemException ex) {
            try {
                this.checkException(null, ex, new AzureServiceErrorCode[0]);
            }
            catch (FileNotFoundException e) {
                return false;
            }
        }
        return true;
    }

    private void createFileSystem() throws IOException {
        LOG.debug("AzureBlobFileSystem.createFileSystem uri: {}", (Object)this.uri);
        try {
            this.abfsStore.createFilesystem();
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkException(null, ex, new AzureServiceErrorCode[0]);
        }
    }

    private URI ensureAuthority(URI uri, Configuration conf) {
        URI defaultUri;
        Preconditions.checkNotNull((Object)uri, (Object)"uri");
        if (uri.getAuthority() == null && (defaultUri = FileSystem.getDefaultUri((Configuration)conf)) != null && this.isAbfsScheme(defaultUri.getScheme())) {
            try {
                uri = new URI(uri.getScheme(), defaultUri.getAuthority(), uri.getPath(), uri.getQuery(), uri.getFragment());
            }
            catch (URISyntaxException e) {
                throw new IllegalArgumentException(new InvalidUriException(uri.toString()));
            }
        }
        if (uri.getAuthority() == null) {
            throw new IllegalArgumentException(new InvalidUriAuthorityException(uri.toString()));
        }
        return uri;
    }

    private boolean isAbfsScheme(String scheme) {
        if (scheme == null) {
            return false;
        }
        return scheme.equals("abfs") || scheme.equals("abfss");
    }

    @VisibleForTesting
    <T> FileSystemOperation<T> execute(String scopeDescription, Callable<T> callableFileOperation) throws IOException {
        return this.execute(scopeDescription, callableFileOperation, null);
    }

    @VisibleForTesting
    <T> FileSystemOperation<T> execute(String scopeDescription, Callable<T> callableFileOperation, T defaultResultValue) throws IOException {
        try {
            T executionResult = callableFileOperation.call();
            return new FileSystemOperation<T>(executionResult, null);
        }
        catch (AbfsRestOperationException abfsRestOperationException) {
            return new FileSystemOperation<T>(defaultResultValue, abfsRestOperationException);
        }
        catch (AzureBlobFileSystemException azureBlobFileSystemException) {
            throw new IOException(azureBlobFileSystemException);
        }
        catch (Exception exception) {
            if (exception instanceof ExecutionException) {
                exception = (Exception)this.getRootCause(exception);
            }
            FileSystemOperationUnhandledException fileSystemOperationUnhandledException = new FileSystemOperationUnhandledException(exception);
            throw new IOException(fileSystemOperationUnhandledException);
        }
    }

    private void checkException(Path path, AzureBlobFileSystemException exception, AzureServiceErrorCode ... allowedErrorCodesList) throws IOException {
        if (exception instanceof AbfsRestOperationException) {
            AbfsRestOperationException ere = (AbfsRestOperationException)exception;
            if (ArrayUtils.contains((Object[])allowedErrorCodesList, (Object)((Object)ere.getErrorCode()))) {
                return;
            }
            int statusCode = ere.getStatusCode();
            if (statusCode == 404) {
                throw (IOException)new FileNotFoundException(ere.getMessage()).initCause(exception);
            }
            if (statusCode == 409) {
                throw (IOException)new FileAlreadyExistsException(ere.getMessage()).initCause((Throwable)exception);
            }
            throw ere;
        }
        if (path == null) {
            throw exception;
        }
        throw new PathIOException(path.toString(), (Throwable)exception);
    }

    private Throwable getRootCause(Throwable throwable) {
        if (throwable == null) {
            throw new IllegalArgumentException("throwable can not be null");
        }
        Throwable result = throwable;
        while (result.getCause() != null) {
            result = result.getCause();
        }
        return result;
    }

    public synchronized Token<?> getDelegationToken(String renewer) throws IOException {
        return this.delegationTokenEnabled ? this.delegationTokenManager.getDelegationToken(renewer) : super.getDelegationToken(renewer);
    }

    @VisibleForTesting
    FileSystem.Statistics getFsStatistics() {
        return this.statistics;
    }

    @VisibleForTesting
    AzureBlobFileSystemStore getAbfsStore() {
        return this.abfsStore;
    }

    @VisibleForTesting
    AbfsClient getAbfsClient() {
        return this.abfsStore.getClient();
    }

    @VisibleForTesting
    boolean getIsNamespaceEnabled() throws AzureBlobFileSystemException {
        return this.abfsStore.getIsNamespaceEnabled();
    }

    private void performAbfsAuthCheck(FsAction action, Path ... paths) throws AbfsAuthorizationException, IOException {
        if (this.authorizer == null) {
            LOG.debug("ABFS authorizer is not initialized. No authorization check will be performed.");
        } else {
            Preconditions.checkArgument((paths.length > 0 ? 1 : 0) != 0, (Object)"no paths supplied for authorization check");
            LOG.debug("Auth check for action: {} on paths: {}", (Object)action.toString(), (Object)Arrays.toString(paths));
            if (!this.authorizer.isAuthorized(action, paths)) {
                throw new AbfsAuthorizationException("User is not authorized for action " + action.toString() + " on paths: " + Arrays.toString(paths));
            }
        }
    }

    @VisibleForTesting
    static class FileSystemOperation<T> {
        private final T result;
        private final AbfsRestOperationException exception;

        FileSystemOperation(T result, AbfsRestOperationException exception) {
            this.result = result;
            this.exception = exception;
        }

        public boolean failed() {
            return this.exception != null;
        }
    }
}

