/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.common.CounterType;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.LeaseDataAccess;
import io.hops.metadata.hdfs.dal.LeasePathDataAccess;
import io.hops.metadata.hdfs.entity.LeasePath;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.INodeLock;
import io.hops.transaction.lock.Lock;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.Lease;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
    }

    Lease getLease(String holder) throws StorageException, TransactionContextException {
        return (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolder, (Object[])new Object[]{holder, Lease.getHolderId(holder)});
    }

    @VisibleForTesting
    int getNumSortedLeases() throws IOException {
        HopsTransactionalRequestHandler getNumSortedLeasesHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_SORTED_LEASES){

            public void acquireLock(TransactionLocks locks) throws IOException {
            }

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return da.countAll();
            }
        };
        return (Integer)getNumSortedLeasesHandler.handle(this.fsnamesystem);
    }

    private SortedSet<Lease> getSortedLeases() throws IOException {
        HopsTransactionalRequestHandler getSortedLeasesHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_SORTED_LEASES){

            public void acquireLock(TransactionLocks locks) throws IOException {
            }

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return da.findAll();
            }
        };
        return new TreeSet<Lease>((Collection)getSortedLeasesHandler.handle(this.fsnamesystem));
    }

    synchronized long getNumUnderConstructionBlocks() throws IOException {
        final AtomicLong numUCBlocks = new AtomicLong(0L);
        for (final Lease lease : this.getSortedLeases()) {
            new HopsTransactionalRequestHandler(HDFSOperationType.GET_LISTING){
                private Set<String> leasePaths;
                {
                    super(opType);
                    this.leasePaths = null;
                }

                @Override
                public void setUp() throws StorageException {
                    String holder = lease.getHolder();
                    this.leasePaths = INodeUtil.findPathsByLeaseHolder(holder);
                    if (this.leasePaths != null) {
                        LOG.debug((Object)("Total Paths " + this.leasePaths.size() + " Paths: " + Arrays.toString(this.leasePaths.toArray())));
                    }
                }

                public void acquireLock(TransactionLocks locks) throws IOException {
                    String holder = lease.getHolder();
                    LockFactory lf = LockFactory.getInstance();
                    INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.READ, TransactionLockTypes.INodeResolveType.PATH_AND_IMMEDIATE_CHILDREN, this.leasePaths.toArray(new String[this.leasePaths.size()])).setNameNodeID(LeaseManager.this.fsnamesystem.getNameNode().getId()).setActiveNameNodes(LeaseManager.this.fsnamesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                    locks.add((Lock)il).add(lf.getLeaseLockAllPaths(TransactionLockTypes.LockType.READ, holder, LeaseManager.this.fsnamesystem.getLeaseCreationLockRows())).add(lf.getLeasePathLock(TransactionLockTypes.LockType.READ)).add(lf.getBlockLock()).add(lf.getBlockRelated(LockFactory.BLK.RE, LockFactory.BLK.CR, LockFactory.BLK.ER, LockFactory.BLK.UC, LockFactory.BLK.UR));
                }

                public Object performTask() throws IOException {
                    for (LeasePath leasePath : lease.getPaths()) {
                        INodeFile cons;
                        String path = leasePath.getPath();
                        try {
                            INode inode = LeaseManager.this.fsnamesystem.getFSDirectory().getINode(path);
                            if (inode == null) {
                                LOG.warn((Object)("Unable to find inode for the lease " + path));
                                continue;
                            }
                            cons = inode.asFile();
                            Preconditions.checkState((boolean)cons.isUnderConstruction());
                        }
                        catch (UnresolvedLinkException e) {
                            throw new AssertionError((Object)"Lease files should reside on this FS");
                        }
                        BlockInfoContiguous[] blocks = cons.getBlocks();
                        if (blocks == null) continue;
                        for (BlockInfoContiguous b : blocks) {
                            if (b.isComplete()) continue;
                            numUCBlocks.incrementAndGet();
                        }
                    }
                    return null;
                }
            }.handle();
        }
        LOG.info((Object)("Number of blocks under construction: " + numUCBlocks.get()));
        return numUCBlocks.get();
    }

    public Lease getLeaseByPath(String src) throws StorageException, TransactionContextException {
        LeasePath leasePath = (LeasePath)EntityManager.find((FinderType)LeasePath.Finder.ByPath, (Object[])new Object[]{src});
        if (leasePath != null) {
            int holderID = leasePath.getHolderId();
            Lease lease = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolderId, (Object[])new Object[]{holderID});
            return lease;
        }
        return null;
    }

    public int countLease() throws IOException {
        return (Integer)new LightWeightRequestHandler(HDFSOperationType.COUNT_LEASE){

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return da.countAll();
            }
        }.handle((Object)this.fsnamesystem);
    }

    int countPath() throws StorageException, TransactionContextException {
        return EntityManager.count((CounterType)Lease.Counter.All, (Object[])new Object[0]);
    }

    Lease addLease(String holder, String src) throws StorageException, TransactionContextException {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder, Lease.getHolderId(holder), Time.now(), 0);
            lease.savePersistent();
        } else {
            this.renewLease(lease);
        }
        if (src != null) {
            LeasePath lPath = new LeasePath(src, lease.getHolderID());
            lease.addPath(lPath);
            lPath.savePersistent();
        }
        return lease;
    }

    void removeLease(Lease lease, LeasePath src) throws StorageException, TransactionContextException {
        if (lease == null) {
            LOG.warn((Object)"Lease not found. Removing lease path");
            src.deletePersistent();
            return;
        }
        if (!lease.removePath(src)) {
            LOG.error((Object)(src + " not found in lease.paths (=" + lease.getPaths() + ")"));
        }
        src.deletePersistent();
        if (!lease.hasPath()) {
            lease.deletePersistent();
        }
    }

    void removeLease(String holder, String src) throws StorageException, TransactionContextException {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, new LeasePath(src, lease.getHolderID()));
        } else {
            LOG.warn((Object)("Removing non-existent lease! holder=" + holder + " src=" + src));
        }
    }

    void removeAllLeases() throws IOException {
        new LightWeightRequestHandler(HDFSOperationType.REMOVE_ALL_LEASES){

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess lda = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                LeasePathDataAccess lpda = (LeasePathDataAccess)HdfsStorageFactory.getDataAccess(LeasePathDataAccess.class);
                lda.removeAll();
                lpda.removeAll();
                return null;
            }
        }.handle((Object)this.fsnamesystem);
    }

    Lease reassignLease(Lease lease, String src, String newHolder) throws StorageException, TransactionContextException {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            LeasePath lp = new LeasePath(src, lease.getHolderID());
            if (!lease.removePath(lp)) {
                LOG.error((Object)(src + " not found in lease.paths (=" + lease.getPaths() + ")"));
            }
            lp.deletePersistent();
            if (!lease.hasPath() && !lease.getHolder().equals(newHolder)) {
                lease.deletePersistent();
            }
        }
        Lease newLease = this.getLease(newHolder);
        LeasePath lPath = null;
        if (newLease == null) {
            newLease = new Lease(newHolder, Lease.getHolderId(newHolder), Time.now(), 0);
            newLease.savePersistent();
            lPath = new LeasePath(src, newLease.getHolderID());
            newLease.addPath(lPath);
        } else {
            this.renewLease(newLease);
            lPath = new LeasePath(src, newLease.getHolderID());
            newLease.addPath(lPath);
        }
        lPath.savePersistent();
        return newLease;
    }

    void renewLease(String holder) throws StorageException, TransactionContextException {
        this.renewLease(this.getLease(holder));
    }

    void renewLease(Lease lease) throws StorageException, TransactionContextException {
        if (lease != null) {
            lease.setLastUpdate(Time.now());
            lease.savePersistent();
        }
    }

    void changeLease(String src, String dst) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".changelease:  src=" + src + ", dest=" + dst));
        }
        int len = src.length();
        Collection<LeasePath> paths = this.findLeasePathsWithPrefix(src);
        ArrayList<LeasePath> newLPs = new ArrayList<LeasePath>(paths.size());
        ArrayList<LeasePath> deletedLPs = new ArrayList<LeasePath>(paths.size());
        for (LeasePath oldPath : paths) {
            int holderId = oldPath.getHolderId();
            LeasePath newpath = new LeasePath(dst + oldPath.getPath().substring(len), holderId, oldPath.getLastBlockId(), oldPath.getPenultimateBlockId());
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("changeLease: replacing " + oldPath + " with " + newpath));
            }
            newLPs.add(newpath);
            deletedLPs.add(oldPath);
        }
        for (LeasePath newPath : newLPs) {
            newPath.savePersistent();
        }
        for (LeasePath deletedLP : deletedLPs) {
            deletedLP.deletePersistent();
        }
    }

    void removeLeaseWithPrefixPath(String prefix) throws StorageException, TransactionContextException {
        for (Map.Entry<LeasePath, Lease> entry : this.findLeaseWithPrefixPath(prefix).entrySet()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".removeLeaseWithPrefixPath: entry=" + entry));
            }
            this.removeLease(entry.getValue(), entry.getKey());
        }
    }

    private Map<LeasePath, Lease> findLeaseWithPrefixPath(String prefix) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        Collection leasePathSet = EntityManager.findList((FinderType)LeasePath.Finder.ByPrefix, (Object[])new Object[]{prefix});
        HashMap<LeasePath, Lease> entries = new HashMap<LeasePath, Lease>();
        int srclen = prefix.length();
        if (prefix.charAt(srclen - 1) == '/') {
            --srclen;
        }
        for (LeasePath lPath : leasePathSet) {
            if (!lPath.getPath().startsWith(prefix)) {
                LOG.warn((Object)("LeasePath fetched by prefix does not start with the prefix: \nLeasePath: " + lPath + "\t Prefix: " + prefix));
                return entries;
            }
            if (lPath.getPath().length() != srclen && lPath.getPath().charAt(srclen) != '/') continue;
            Lease lease = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolderId, (Object[])new Object[]{lPath.getHolderId()});
            entries.put(lPath, lease);
        }
        return entries;
    }

    private Collection<LeasePath> findLeasePathsWithPrefix(String prefix) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        Collection leasePathSet = EntityManager.findList((FinderType)LeasePath.Finder.ByPrefix, (Object[])new Object[]{prefix});
        return leasePathSet;
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    @VisibleForTesting
    boolean checkLeases() throws StorageException, TransactionContextException, IOException {
        boolean needSync = false;
        SortedSet sortedLeases = (SortedSet)new LightWeightRequestHandler(HDFSOperationType.PREPARE_LEASE_MANAGER_MONITOR){

            public Object performTask() throws StorageException, IOException {
                long expiredTime = Time.now() - LeaseManager.this.hardLimit;
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return new TreeSet(da.findByTimeLimit(expiredTime));
            }
        }.handle((Object)this.fsnamesystem);
        if (sortedLeases != null) {
            for (Lease expiredLease : sortedLeases) {
                HopsTransactionalRequestHandler expiredLeaseHandler = new HopsTransactionalRequestHandler(HDFSOperationType.LEASE_MANAGER_MONITOR){
                    private Set<String> leasePaths;
                    {
                        this.leasePaths = null;
                    }

                    @Override
                    public void setUp() throws StorageException {
                        String holder = (String)this.getParams()[0];
                        this.leasePaths = INodeUtil.findPathsByLeaseHolder(holder);
                        if (this.leasePaths != null) {
                            LOG.debug((Object)("Total Paths " + this.leasePaths.size() + " Paths: " + Arrays.toString(this.leasePaths.toArray())));
                        }
                    }

                    public void acquireLock(TransactionLocks locks) throws IOException {
                        String holder = (String)this.getParams()[0];
                        LockFactory lf = LockFactory.getInstance();
                        INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, this.leasePaths.toArray(new String[this.leasePaths.size()])).setNameNodeID(LeaseManager.this.fsnamesystem.getNameNode().getId()).setActiveNameNodes(LeaseManager.this.fsnamesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                        locks.add((Lock)il).add(lf.getNameNodeLeaseLock(TransactionLockTypes.LockType.WRITE)).add(lf.getLeaseLockAllPaths(TransactionLockTypes.LockType.WRITE, holder, LeaseManager.this.fsnamesystem.getLeaseCreationLockRows())).add(lf.getLeasePathLock(TransactionLockTypes.LockType.WRITE, this.leasePaths.size())).add(lf.getBlockLock()).add(lf.getBlockRelated(LockFactory.BLK.RE, LockFactory.BLK.CR, LockFactory.BLK.ER, LockFactory.BLK.UC, LockFactory.BLK.UR));
                    }

                    public Object performTask() throws StorageException, IOException {
                        String holder = (String)this.getParams()[0];
                        boolean needSync = false;
                        if (holder != null) {
                            Lease leaseToCheck = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolder, (Object[])new Object[]{holder, Lease.getHolderId(holder)});
                            if (!LeaseManager.this.expiredHardLimit(leaseToCheck)) {
                                LOG.warn((Object)("Unable to release hard-limit expired lease: " + leaseToCheck));
                                return needSync;
                            }
                            LOG.info((Object)("Lease " + leaseToCheck + " has expired hard limit"));
                            if (!leaseToCheck.hasPath()) {
                                leaseToCheck.deletePersistent();
                                return true;
                            }
                            ArrayList<LeasePath> removing = new ArrayList<LeasePath>();
                            LeasePath[] leasePaths = new LeasePath[leaseToCheck.getPaths().size()];
                            leaseToCheck.getPaths().toArray(leasePaths);
                            for (LeasePath p : leasePaths) {
                                try {
                                    INodesInPath iip = LeaseManager.this.fsnamesystem.getFSDirectory().getINodesInPath(p.getPath(), true);
                                    if (iip == null || iip.getLastINode() == null) {
                                        removing.add(p);
                                        continue;
                                    }
                                    boolean completed = LeaseManager.this.fsnamesystem.internalReleaseLease(leaseToCheck, p.getPath(), iip, "HDFS_NameNode");
                                    if (LOG.isDebugEnabled()) {
                                        if (completed) {
                                            LOG.debug((Object)("Lease recovery for " + p + " is complete. File closed."));
                                        } else {
                                            LOG.debug((Object)("Started block recovery " + p + " lease " + leaseToCheck));
                                        }
                                    }
                                    if (needSync || completed) continue;
                                    needSync = true;
                                }
                                catch (IOException e) {
                                    LOG.error((Object)("Cannot release the path " + p + " in the lease " + leaseToCheck), (Throwable)e);
                                    removing.add(p);
                                }
                            }
                            for (LeasePath p : removing) {
                                if (!leaseToCheck.getPaths().contains(p)) continue;
                                LeaseManager.this.removeLease(leaseToCheck, p);
                            }
                        }
                        return needSync;
                    }
                };
                needSync = needSync || (Boolean)expiredLeaseHandler.setParams(new Object[]{expiredLease.getHolder()}).handle((Object)this.fsnamesystem) != false;
            }
        }
        return needSync;
    }

    void startMonitor() {
        Preconditions.checkState((this.lmthread == null ? 1 : 0) != 0, (Object)"Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon(new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn((Object)"Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    void triggerMonitorCheckNow() {
        Preconditions.checkState((this.lmthread != null ? 1 : 0) != 0, (Object)"Lease monitor is not running");
        this.lmthread.interrupt();
    }

    private boolean expiredHardLimit(Lease lease) {
        return Time.now() - lease.getLastUpdate() > this.hardLimit;
    }

    public boolean expiredSoftLimit(Lease lease) {
        return Time.now() - lease.getLastUpdate() > this.softLimit;
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();

        Monitor() {
        }

        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                try {
                    if (LeaseManager.this.fsnamesystem.isLeader()) {
                        try {
                            LeaseManager.this.checkLeases();
                        }
                        catch (IOException ex) {
                            LOG.error((Object)ex);
                        }
                    }
                    Thread.sleep(2000L);
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(this.name + " is interrupted"), (Throwable)ie);
                }
            }
        }
    }
}

