/*
 * Decompiled with CFR 0.152.
 */
package io.hops.leader_election.watchdog;

import com.google.common.annotations.VisibleForTesting;
import io.hops.leader_election.watchdog.AliveWatchdogPoller;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.util.BackOff;
import org.apache.hadoop.util.ExponentialBackOff;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.concurrent.HadoopExecutors;

public class AliveWatchdogService
extends AbstractService {
    private static final Log LOG = LogFactory.getLog(AliveWatchdogService.class);
    private final TimeUnit intervalTimeunit = TimeUnit.MILLISECONDS;
    private long interval;
    private AliveWatchdogPoller poller;
    private ScheduledExecutorService executorService;

    public AliveWatchdogService() {
        super("Active watchdog service");
    }

    @Override
    public void serviceInit(Configuration conf) throws Exception {
        this.interval = conf.getTimeDuration("hops.alive-watchdog.interval", "5s", this.intervalTimeunit);
        if (this.interval < 20L) {
            LOG.warn((Object)"Interval is set too low. Setting it to 20ms");
            this.interval = 20L;
        }
        super.serviceInit(conf);
    }

    @Override
    public void serviceStart() throws Exception {
        LOG.info((Object)"Starting Alive Watchdog service");
        LOG.info((Object)("Polling interval is " + this.getConfig().get("hops.alive-watchdog.interval")));
        if (this.poller == null) {
            this.poller = this.loadPoller();
        }
        this.poller.init();
        LOG.debug((Object)"Loaded poller class");
        ExponentialBackOff backOff = new ExponentialBackOff.Builder().setInitialIntervalMillis(5L).setMaximumIntervalMillis(this.interval).build();
        if (!this.shouldIBeAliveWithRetry(backOff)) {
            LOG.info((Object)"We should not be alive. Bye");
            this.exit();
        } else {
            this.intervalTimeunit.sleep(this.interval + 1000L);
            PollerDaemon watchdog = new PollerDaemon(backOff);
            LOG.debug((Object)"Starting watchdog");
            this.executorService = HadoopExecutors.newScheduledThreadPool(1);
            this.executorService.scheduleAtFixedRate(watchdog, 0L, this.interval, this.intervalTimeunit);
        }
        super.serviceStart();
    }

    @VisibleForTesting
    ExecutorService getExecutorService() {
        return this.executorService;
    }

    @Override
    public void serviceStop() throws Exception {
        LOG.info((Object)"Stopping Alive Watchdog service");
        this.stopExecutorService();
        if (this.poller != null) {
            this.poller.destroy();
        }
    }

    private void stopExecutorService() {
        if (this.executorService != null) {
            try {
                this.executorService.shutdown();
                if (!this.executorService.awaitTermination(100L, TimeUnit.MILLISECONDS)) {
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException ex) {
                this.executorService.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    private AliveWatchdogPoller loadPoller() throws ClassNotFoundException {
        String pollerClass = this.getConfig().get("hops.alive-watchdog.poller-class");
        if (pollerClass == null) {
            throw new IllegalArgumentException("Alive watchdog service is enabled but poller class is not set");
        }
        LOG.info((Object)("Loading poller class " + pollerClass));
        Class<?> clazz = this.getConfig().getClassByName(pollerClass);
        return (AliveWatchdogPoller)ReflectionUtils.newInstance(clazz, this.getConfig());
    }

    @VisibleForTesting
    void setPoller(AliveWatchdogPoller poller) {
        this.poller = poller;
    }

    void exit() {
        this.stopExecutorService();
        System.exit(1);
    }

    private boolean shouldIBeAliveWithRetry(BackOff backOff) throws Exception {
        backOff.reset();
        while (true) {
            try {
                return this.poller.shouldIBeAlive();
            }
            catch (Exception ex) {
                long backOffTimeout = backOff.getBackOffInMillis();
                if (backOffTimeout != -1L) {
                    LOG.debug((Object)("Error while polling to check whether we should be alive or not. Retrying in " + backOffTimeout + "ms"));
                    try {
                        TimeUnit.MILLISECONDS.sleep(backOffTimeout);
                    }
                    catch (InterruptedException iex) {
                        Thread.currentThread().interrupt();
                    }
                    continue;
                }
                throw ex;
            }
            break;
        }
    }

    private class PollerDaemon
    implements Runnable {
        private final BackOff backOff;

        private PollerDaemon(BackOff backOff) {
            this.backOff = backOff;
        }

        @Override
        public void run() {
            LOG.debug((Object)"Checking whether we should be alive or not");
            try {
                if (!AliveWatchdogService.this.shouldIBeAliveWithRetry(this.backOff)) {
                    LOG.info((Object)"Watchdog decided we should not be alive. Bye...");
                    AliveWatchdogService.this.exit();
                }
                LOG.debug((Object)"We should be alive");
            }
            catch (Exception ex) {
                LOG.fatal((Object)"Watchdog could not contact oracle service. Shutting down for safety");
                AliveWatchdogService.this.exit();
            }
        }
    }
}

