/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security.ssl;

import com.google.common.annotations.VisibleForTesting;
import io.hops.hadoop.shaded.org.apache.log4j.LogManager;
import io.hops.hadoop.shaded.org.apache.log4j.Logger;
import io.hops.security.HopsFileBasedKeyStoresFactory;
import io.hops.security.HopsUtil;
import io.hops.security.SuperuserKeystoresLoader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.Security;
import java.security.cert.CRLException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CommonConfigurationKeys;
import org.apache.hadoop.net.hopssslchecks.HopsSSLCryptoMaterial;
import org.apache.hadoop.security.ssl.FileBasedKeyStoresFactory;
import org.apache.hadoop.security.ssl.SSLFactory;
import org.apache.hadoop.security.ssl.X509SecurityMaterial;

public class CRLValidator {
    private static final Logger LOG = LogManager.getLogger(CRLValidator.class);
    private final Configuration conf;
    private final Configuration sslConf;
    private final Path crl;
    private final File trustStoreLocation;
    private final File truststorePasswordLocation;
    private final AtomicReference<X509CRL> crlReference;
    private final AtomicReference<KeyStore> trustStoreReference;
    private CertificateFactory certificateFactory;
    private TimeUnit reloadTimeunit;
    private long reloadInterval = -1L;
    private long crlLastLoadedTimestamp = 0L;
    private long trustStoreLastLoadedTimestamp = 0L;
    private Thread reloaderThread;
    private final RetryAction<X509CRL> loadCRLWithRetry;
    private final RetryAction<KeyStore> loadTruststoreWithRetry;

    CRLValidator(Configuration conf) throws IOException, GeneralSecurityException {
        this(conf, null);
    }

    CRLValidator(Configuration conf, Configuration sslConf) throws IOException, GeneralSecurityException {
        this.conf = conf;
        this.sslConf = sslConf != null ? sslConf : this.readSSLConfiguration();
        Security.setProperty("ocsp.enable", "false");
        System.setProperty("com.sun.security.enableCRLDP", "false");
        this.certificateFactory = CertificateFactory.getInstance("X.509");
        this.loadCRLWithRetry = new RetryAction<X509CRL>(){

            @Override
            X509CRL operationToPerform() throws GeneralSecurityException, IOException {
                return CRLValidator.this.loadCRL();
            }
        };
        this.crl = Paths.get(conf.get("hops.crl.output.file", CommonConfigurationKeys.HOPS_CRL_OUTPUT_FILE_DEFAULT), new String[0]);
        this.crlReference = new AtomicReference<Object>(((RetryAction)this.loadCRLWithRetry).retry());
        this.loadTruststoreWithRetry = new RetryAction<KeyStore>(){

            @Override
            KeyStore operationToPerform() throws GeneralSecurityException, IOException {
                return CRLValidator.this.loadTruststore();
            }
        };
        SuperuserKeystoresLoader loader = new SuperuserKeystoresLoader(conf);
        X509SecurityMaterial x509Material = loader.loadSuperUserMaterial();
        if (x509Material.getTrustStoreLocation().toFile().exists() && x509Material.getPasswdLocation().toFile().exists()) {
            this.trustStoreLocation = x509Material.getTrustStoreLocation().toFile();
            this.truststorePasswordLocation = x509Material.getPasswdLocation().toFile();
        } else {
            HopsFileBasedKeyStoresFactory factory = new HopsFileBasedKeyStoresFactory();
            factory.setConf(conf);
            factory.setSystemConf(conf);
            HopsSSLCryptoMaterial material = factory.loadCryptoMaterial(SSLFactory.Mode.SERVER);
            this.trustStoreLocation = new File(material.getTrustStoreLocation());
            this.truststorePasswordLocation = new File(material.getPasswordFileLocation());
        }
        this.trustStoreReference = new AtomicReference<Object>(((RetryAction)this.loadTruststoreWithRetry).retry());
    }

    public void startReloadingThread() {
        if (this.reloadTimeunit == null) {
            this.reloadTimeunit = TimeUnit.MINUTES;
        }
        if (this.reloadInterval == -1L) {
            this.reloadInterval = 60L;
        }
        this.reloaderThread = new ReloaderThread();
        this.reloaderThread.setName("CRL Validator reloader thread");
        this.reloaderThread.setDaemon(true);
        this.reloaderThread.start();
    }

    public void stopReloaderThread() {
        if (this.reloaderThread != null) {
            this.reloaderThread.interrupt();
        }
    }

    @VisibleForTesting
    public void setReloadTimeunit(TimeUnit reloadTimeunit) {
        this.reloadTimeunit = reloadTimeunit;
    }

    public void setReloadInterval(long reloadInterval) {
        this.reloadInterval = reloadInterval;
    }

    @VisibleForTesting
    public void setCertificateFactory(CertificateFactory certificateFactory) {
        this.certificateFactory = certificateFactory;
    }

    @VisibleForTesting
    public TimeUnit getReloadTimeunit() {
        return this.reloadTimeunit;
    }

    @VisibleForTesting
    public long getReloadInterval() {
        return this.reloadInterval;
    }

    public void validate(Certificate[] certificateChain) throws CertificateException {
        X509CRL crl = this.crlReference.get();
        for (Certificate certificate : certificateChain) {
            if (!(certificate instanceof X509Certificate)) {
                throw new CertificateException("Certificate is not X.509");
            }
            X509Certificate x509Certificate = (X509Certificate)certificate;
            X509CRLEntry crlEntry = crl.getRevokedCertificate(x509Certificate);
            if (crlEntry == null) continue;
            String revocationReason = crlEntry.getRevocationReason() != null ? " REASON: " + crlEntry.getRevocationReason().toString() : "";
            throw new CertificateException("HopsCRLValidator: Certificate " + x509Certificate.getSubjectDN().toString() + " has been revoked by " + crl.getIssuerX500Principal().getName() + revocationReason);
        }
        LOG.debug((Object)("Certificate " + certificateChain[0] + " is valid"));
    }

    private Configuration readSSLConfiguration() {
        Configuration sslConf = new Configuration(false);
        String sslConfResource = this.conf.get("hadoop.ssl.server.conf", "ssl-server.xml");
        sslConf.addResource(sslConfResource);
        return sslConf;
    }

    private KeyStore loadTruststore() throws GeneralSecurityException, IOException {
        String type = this.sslConf.get(FileBasedKeyStoresFactory.resolvePropertyName(SSLFactory.Mode.SERVER, "ssl.{0}.truststore.type"), "jks");
        String trustStorePassword = HopsUtil.readCryptoMaterialPassword(this.truststorePasswordLocation);
        KeyStore trustStore = KeyStore.getInstance(type);
        try (FileInputStream in = new FileInputStream(this.trustStoreLocation);){
            trustStore.load(in, trustStorePassword.toCharArray());
        }
        this.trustStoreLastLoadedTimestamp = this.trustStoreLocation.lastModified();
        return trustStore;
    }

    private X509CRL loadCRL() throws IOException, CertificateException, CRLException {
        try (InputStream in = Files.newInputStream(this.crl, StandardOpenOption.READ);){
            this.crlLastLoadedTimestamp = this.crl.toFile().lastModified();
            X509CRL x509CRL = (X509CRL)this.certificateFactory.generateCRL(in);
            return x509CRL;
        }
    }

    private class ReloaderThread
    extends Thread {
        private ReloaderThread() {
        }

        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    CRLValidator.this.reloadTimeunit.sleep(CRLValidator.this.reloadInterval);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                if (this.crlNeedsReload()) {
                    try {
                        CRLValidator.this.crlReference.set(((RetryAction)CRLValidator.this.loadCRLWithRetry).retry());
                    }
                    catch (IOException | GeneralSecurityException ex) {
                        LOG.error((Object)"Could not reload CRL", (Throwable)ex);
                    }
                }
                if (!this.trustStoreNeedsReload()) continue;
                try {
                    CRLValidator.this.trustStoreReference.set(((RetryAction)CRLValidator.this.loadTruststoreWithRetry).retry());
                }
                catch (IOException | GeneralSecurityException ex) {
                    LOG.error((Object)"Could not reload TrustStore", (Throwable)ex);
                }
            }
        }

        private boolean crlNeedsReload() {
            return CRLValidator.this.crl.toFile().exists() && CRLValidator.this.crl.toFile().lastModified() > CRLValidator.this.crlLastLoadedTimestamp;
        }

        private boolean trustStoreNeedsReload() {
            return CRLValidator.this.trustStoreLocation.exists() && CRLValidator.this.trustStoreLocation.lastModified() > CRLValidator.this.trustStoreLastLoadedTimestamp;
        }
    }

    private abstract class RetryAction<T> {
        private int numberOfFailures = 0;

        private RetryAction() {
        }

        abstract T operationToPerform() throws GeneralSecurityException, IOException;

        private T retry() throws GeneralSecurityException, IOException {
            while (true) {
                try {
                    return this.operationToPerform();
                }
                catch (IOException | GeneralSecurityException ex) {
                    if (this.numberOfFailures > 5) {
                        throw ex;
                    }
                    ++this.numberOfFailures;
                    try {
                        TimeUnit.MILLISECONDS.sleep(100L);
                    }
                    catch (InterruptedException iex) {
                        throw new IOException(iex);
                    }
                }
            }
        }
    }
}

