/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.yarn.api.records.ContainerId;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.api.records.ResourceUtilization;
import org.apache.hadoop.yarn.event.AsyncDispatcher;
import org.apache.hadoop.yarn.event.Dispatcher;
import org.apache.hadoop.yarn.event.Event;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.server.nodemanager.ContainerExecutor;
import org.apache.hadoop.yarn.server.nodemanager.Context;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.Container;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.ContainerImpl;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.ContainerKillEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.CGroupElasticMemoryController;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.MemoryResourceHandler;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.ResourceHandlerModule;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ChangeMonitoringContainerResourceEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainerMetrics;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainerStartMonitoringEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainersMonitor;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainersMonitorEvent;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.monitor.ContainersMonitorEventType;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.runtime.ContainerExecutionException;
import org.apache.hadoop.yarn.server.nodemanager.metrics.NodeManagerMetrics;
import org.apache.hadoop.yarn.server.nodemanager.timelineservice.NMTimelinePublisher;
import org.apache.hadoop.yarn.server.nodemanager.util.NodeManagerHardwareUtils;
import org.apache.hadoop.yarn.util.ResourceCalculatorPlugin;
import org.apache.hadoop.yarn.util.ResourceCalculatorProcessTree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainersMonitorImpl
extends AbstractService
implements ContainersMonitor {
    private static final Logger LOG = LoggerFactory.getLogger(ContainersMonitorImpl.class);
    private static final Logger AUDITLOG = LoggerFactory.getLogger((String)(ContainersMonitorImpl.class.getName() + ".audit"));
    private long monitoringInterval;
    private MonitoringThread monitoringThread;
    private CGroupElasticMemoryController oomListenerThread;
    private boolean containerMetricsEnabled;
    private long containerMetricsPeriodMs;
    private long containerMetricsUnregisterDelayMs;
    @VisibleForTesting
    final Map<ContainerId, ProcessTreeInfo> trackingContainers = new ConcurrentHashMap<ContainerId, ProcessTreeInfo>();
    private final ContainerExecutor containerExecutor;
    private final Dispatcher eventDispatcher;
    private final Context context;
    private ResourceCalculatorPlugin resourceCalculatorPlugin;
    private Configuration conf;
    private static float vmemRatio;
    private Class<? extends ResourceCalculatorProcessTree> processTreeClass;
    private long maxVmemAllottedForContainers = -1L;
    private long maxPmemAllottedForContainers = -1L;
    private boolean pmemCheckEnabled;
    private boolean vmemCheckEnabled;
    private boolean elasticMemoryEnforcement;
    private boolean strictMemoryEnforcement;
    private boolean containersMonitorEnabled;
    private long maxVCoresAllottedForContainers;
    private static final long UNKNOWN_MEMORY_LIMIT = -1L;
    private int nodeCpuPercentageForYARN;
    private ResourceUtilization containersUtilization;
    private volatile boolean stopped = false;

    public ContainersMonitorImpl(ContainerExecutor exec, AsyncDispatcher dispatcher, Context context) {
        super("containers-monitor");
        this.containerExecutor = exec;
        this.eventDispatcher = dispatcher;
        this.context = context;
        this.monitoringThread = new MonitoringThread();
        this.containersUtilization = ResourceUtilization.newInstance((int)0, (int)0, (float)0.0f);
    }

    protected void serviceInit(Configuration myConf) throws Exception {
        this.conf = myConf;
        this.monitoringInterval = this.conf.getLong("yarn.nodemanager.container-monitor.interval-ms", this.conf.getLong("yarn.nodemanager.resource-monitor.interval-ms", 3000L));
        this.resourceCalculatorPlugin = ResourceCalculatorPlugin.getContainersMonitorPlugin((Configuration)this.conf);
        LOG.info(" Using ResourceCalculatorPlugin : " + this.resourceCalculatorPlugin);
        this.processTreeClass = this.conf.getClass("yarn.nodemanager.container-monitor.process-tree.class", null, ResourceCalculatorProcessTree.class);
        LOG.info(" Using ResourceCalculatorProcessTree : " + this.processTreeClass);
        this.containerMetricsEnabled = this.conf.getBoolean("yarn.nodemanager.container-metrics.enable", true);
        this.containerMetricsPeriodMs = this.conf.getLong("yarn.nodemanager.container-metrics.period-ms", -1L);
        this.containerMetricsUnregisterDelayMs = this.conf.getLong("yarn.nodemanager.container-metrics.unregister-delay-ms", 10000L);
        long configuredPMemForContainers = NodeManagerHardwareUtils.getContainerMemoryMB(this.resourceCalculatorPlugin, this.conf) * 1024L * 1024L;
        long configuredVCoresForContainers = NodeManagerHardwareUtils.getVCores(this.resourceCalculatorPlugin, this.conf);
        this.maxPmemAllottedForContainers = configuredPMemForContainers;
        this.maxVCoresAllottedForContainers = configuredVCoresForContainers;
        vmemRatio = this.conf.getFloat("yarn.nodemanager.vmem-pmem-ratio", 2.1f);
        Preconditions.checkArgument((vmemRatio > 0.99f ? 1 : 0) != 0, (Object)"yarn.nodemanager.vmem-pmem-ratio should be at least 1.0");
        this.maxVmemAllottedForContainers = (long)(vmemRatio * (float)configuredPMemForContainers);
        this.pmemCheckEnabled = this.conf.getBoolean("yarn.nodemanager.pmem-check-enabled", true);
        this.vmemCheckEnabled = this.conf.getBoolean("yarn.nodemanager.vmem-check-enabled", true);
        this.elasticMemoryEnforcement = this.conf.getBoolean("yarn.nodemanager.elastic-memory-control.enabled", false);
        this.strictMemoryEnforcement = this.conf.getBoolean("yarn.nodemanager.resource.memory.enforced", true);
        LOG.info("Physical memory check enabled: " + this.pmemCheckEnabled);
        LOG.info("Virtual memory check enabled: " + this.vmemCheckEnabled);
        LOG.info("Elastic memory control enabled: " + this.elasticMemoryEnforcement);
        LOG.info("Strict memory control enabled: " + this.strictMemoryEnforcement);
        if (this.elasticMemoryEnforcement) {
            if (!CGroupElasticMemoryController.isAvailable()) {
                throw new YarnException("CGroup Elastic Memory controller enabled but it is not available. Exiting.");
            }
            this.oomListenerThread = new CGroupElasticMemoryController(this.conf, this.context, ResourceHandlerModule.getCGroupsHandler(), this.pmemCheckEnabled, this.vmemCheckEnabled, this.pmemCheckEnabled ? this.maxPmemAllottedForContainers : this.maxVmemAllottedForContainers);
        }
        this.containersMonitorEnabled = this.isContainerMonitorEnabled() && this.monitoringInterval > 0L;
        LOG.info("ContainersMonitor enabled: " + this.containersMonitorEnabled);
        this.nodeCpuPercentageForYARN = NodeManagerHardwareUtils.getNodeCpuPercentage(this.conf);
        if (this.pmemCheckEnabled) {
            long totalPhysicalMemoryOnNM = -1L;
            if (this.resourceCalculatorPlugin != null && (totalPhysicalMemoryOnNM = this.resourceCalculatorPlugin.getPhysicalMemorySize()) <= 0L) {
                LOG.warn("NodeManager's totalPmem could not be calculated. Setting it to -1");
                totalPhysicalMemoryOnNM = -1L;
            }
            if (totalPhysicalMemoryOnNM != -1L && (float)this.maxPmemAllottedForContainers > (float)totalPhysicalMemoryOnNM * 0.8f) {
                LOG.warn("NodeManager configured with " + StringUtils.TraditionalBinaryPrefix.long2String((long)this.maxPmemAllottedForContainers, (String)"", (int)1) + " physical memory allocated to containers, which is more than 80% of the total physical memory available (" + StringUtils.TraditionalBinaryPrefix.long2String((long)totalPhysicalMemoryOnNM, (String)"", (int)1) + "). Thrashing might happen.");
            }
        }
        super.serviceInit(this.conf);
    }

    private boolean isContainerMonitorEnabled() {
        return this.conf.getBoolean("yarn.nodemanager.container-monitor.enabled", true);
    }

    private ResourceCalculatorProcessTree getResourceCalculatorProcessTree(String pId) {
        return ResourceCalculatorProcessTree.getResourceCalculatorProcessTree((String)pId, this.processTreeClass, (Configuration)this.conf);
    }

    private boolean isResourceCalculatorAvailable() {
        if (this.resourceCalculatorPlugin == null) {
            LOG.info("ResourceCalculatorPlugin is unavailable on this system. " + this.getClass().getName() + " is disabled.");
            return false;
        }
        if (this.getResourceCalculatorProcessTree("0") == null) {
            LOG.info("ResourceCalculatorProcessTree is unavailable on this system. " + this.getClass().getName() + " is disabled.");
            return false;
        }
        return true;
    }

    protected void serviceStart() throws Exception {
        if (this.containersMonitorEnabled) {
            this.monitoringThread.start();
        }
        if (this.oomListenerThread != null) {
            this.oomListenerThread.start();
        }
        super.serviceStart();
    }

    protected void serviceStop() throws Exception {
        if (this.containersMonitorEnabled) {
            this.stopped = true;
            this.monitoringThread.interrupt();
            try {
                this.monitoringThread.join();
            }
            catch (InterruptedException e) {
                LOG.info("ContainersMonitorImpl monitoring thread interrupted");
            }
            if (this.oomListenerThread != null) {
                this.oomListenerThread.stopListening();
                try {
                    this.oomListenerThread.join();
                }
                finally {
                    this.oomListenerThread = null;
                }
            }
        }
        super.serviceStop();
    }

    private boolean isProcessTreeOverLimit(String containerId, long currentMemUsage, long curMemUsageOfAgedProcesses, long memLimit) {
        boolean isOverLimit = false;
        if (currentMemUsage > 2L * memLimit) {
            LOG.warn("Process tree for container: " + containerId + " running over twice the configured limit. Limit=" + memLimit + ", current usage = " + currentMemUsage);
            isOverLimit = true;
        } else if (curMemUsageOfAgedProcesses > memLimit) {
            LOG.warn("Process tree for container: " + containerId + " has processes older than 1 iteration running over the configured limit. Limit=" + memLimit + ", current usage = " + curMemUsageOfAgedProcesses);
            isOverLimit = true;
        }
        return isOverLimit;
    }

    boolean isProcessTreeOverLimit(ResourceCalculatorProcessTree pTree, String containerId, long limit) {
        long currentMemUsage = pTree.getVirtualMemorySize();
        long curMemUsageOfAgedProcesses = pTree.getVirtualMemorySize(1);
        return this.isProcessTreeOverLimit(containerId, currentMemUsage, curMemUsageOfAgedProcesses, limit);
    }

    private void updateContainerMetrics(ContainersMonitorEvent monitoringEvent) {
        if (!this.containerMetricsEnabled || monitoringEvent == null) {
            return;
        }
        ContainerId containerId = monitoringEvent.getContainerId();
        switch ((ContainersMonitorEventType)monitoringEvent.getType()) {
            case START_MONITORING_CONTAINER: {
                ContainerMetrics usageMetrics = ContainerMetrics.forContainer(containerId, this.containerMetricsPeriodMs, this.containerMetricsUnregisterDelayMs);
                ContainerStartMonitoringEvent startEvent = (ContainerStartMonitoringEvent)monitoringEvent;
                usageMetrics.recordStateChangeDurations(startEvent.getLaunchDuration(), startEvent.getLocalizationDuration());
                int cpuVcores = startEvent.getCpuVcores();
                int vmemLimitMBs = (int)(startEvent.getVmemLimit() >> 20);
                int pmemLimitMBs = (int)(startEvent.getPmemLimit() >> 20);
                usageMetrics.recordResourceLimit(vmemLimitMBs, pmemLimitMBs, cpuVcores);
                break;
            }
            case STOP_MONITORING_CONTAINER: {
                ContainerMetrics usageMetrics = ContainerMetrics.getContainerMetrics(containerId);
                if (usageMetrics == null) break;
                usageMetrics.finished();
                break;
            }
            case CHANGE_MONITORING_CONTAINER_RESOURCE: {
                ContainerMetrics usageMetrics = ContainerMetrics.forContainer(containerId, this.containerMetricsPeriodMs, this.containerMetricsUnregisterDelayMs);
                ChangeMonitoringContainerResourceEvent changeEvent = (ChangeMonitoringContainerResourceEvent)monitoringEvent;
                Resource resource = changeEvent.getResource();
                int pmemLimitMBs = (int)resource.getMemorySize();
                int vmemLimitMBs = (int)((float)pmemLimitMBs * vmemRatio);
                int cpuVcores = resource.getVirtualCores();
                usageMetrics.recordResourceLimit(vmemLimitMBs, pmemLimitMBs, cpuVcores);
                break;
            }
        }
    }

    @Override
    public long getVmemAllocatedForContainers() {
        return this.maxVmemAllottedForContainers;
    }

    @Override
    public boolean isPmemCheckEnabled() {
        return this.pmemCheckEnabled;
    }

    @Override
    public long getPmemAllocatedForContainers() {
        return this.maxPmemAllottedForContainers;
    }

    @Override
    public long getVCoresAllocatedForContainers() {
        return this.maxVCoresAllottedForContainers;
    }

    @Override
    public boolean isVmemCheckEnabled() {
        return this.vmemCheckEnabled;
    }

    @Override
    public ResourceUtilization getContainersUtilization() {
        return this.containersUtilization;
    }

    private void setContainersUtilization(ResourceUtilization utilization) {
        this.containersUtilization = utilization;
    }

    @Override
    public void subtractNodeResourcesFromResourceUtilization(ResourceUtilization resourceUtil) {
        resourceUtil.subtractFrom((int)(this.getPmemAllocatedForContainers() >> 20), (int)(this.getVmemAllocatedForContainers() >> 20), 1.0f);
    }

    @Override
    public float getVmemRatio() {
        return vmemRatio;
    }

    public void handle(ContainersMonitorEvent monitoringEvent) {
        ContainerId containerId = monitoringEvent.getContainerId();
        switch ((ContainersMonitorEventType)monitoringEvent.getType()) {
            case START_MONITORING_CONTAINER: {
                this.onStartMonitoringContainer(monitoringEvent, containerId);
                break;
            }
            case STOP_MONITORING_CONTAINER: {
                this.onStopMonitoringContainer(monitoringEvent, containerId);
                break;
            }
            case CHANGE_MONITORING_CONTAINER_RESOURCE: {
                this.onChangeMonitoringContainerResource(monitoringEvent, containerId);
                break;
            }
        }
    }

    private void onChangeMonitoringContainerResource(ContainersMonitorEvent monitoringEvent, ContainerId containerId) {
        ChangeMonitoringContainerResourceEvent changeEvent = (ChangeMonitoringContainerResourceEvent)monitoringEvent;
        if (this.containersMonitorEnabled) {
            ProcessTreeInfo processTreeInfo = this.trackingContainers.get(containerId);
            if (processTreeInfo == null) {
                LOG.warn("Failed to track container " + containerId.toString() + ". It may have already completed.");
                return;
            }
            LOG.info("Changing resource-monitoring for " + containerId);
            this.updateContainerMetrics(monitoringEvent);
            long pmemLimit = changeEvent.getResource().getMemorySize() * 1024L * 1024L;
            long vmemLimit = (long)((float)pmemLimit * vmemRatio);
            int cpuVcores = changeEvent.getResource().getVirtualCores();
            processTreeInfo.setResourceLimit(pmemLimit, vmemLimit, cpuVcores);
        }
    }

    private void onStopMonitoringContainer(ContainersMonitorEvent monitoringEvent, ContainerId containerId) {
        LOG.info("Stopping resource-monitoring for " + containerId);
        this.updateContainerMetrics(monitoringEvent);
        this.trackingContainers.remove(containerId);
    }

    private void onStartMonitoringContainer(ContainersMonitorEvent monitoringEvent, ContainerId containerId) {
        ContainerStartMonitoringEvent startEvent = (ContainerStartMonitoringEvent)monitoringEvent;
        LOG.info("Starting resource-monitoring for " + containerId);
        this.updateContainerMetrics(monitoringEvent);
        this.trackingContainers.put(containerId, new ProcessTreeInfo(containerId, null, null, startEvent.getVmemLimit(), startEvent.getPmemLimit(), startEvent.getCpuVcores()));
    }

    private class MonitoringThread
    extends Thread {
        MonitoringThread() {
            super("Container Monitor");
        }

        @Override
        public void run() {
            while (!ContainersMonitorImpl.this.stopped && !Thread.currentThread().isInterrupted()) {
                if (LOG.isDebugEnabled()) {
                    StringBuilder tmp = new StringBuilder("[ ");
                    for (ProcessTreeInfo p : ContainersMonitorImpl.this.trackingContainers.values()) {
                        tmp.append(p.getPID());
                        tmp.append(" ");
                    }
                    LOG.debug("Current ProcessTree list : " + tmp.substring(0, tmp.length()) + "]");
                }
                ResourceUtilization trackedContainersUtilization = ResourceUtilization.newInstance((int)0, (int)0, (float)0.0f);
                long vmemUsageByAllContainers = 0L;
                long pmemByAllContainers = 0L;
                long cpuUsagePercentPerCoreByAllContainers = 0L;
                for (Map.Entry<ContainerId, ProcessTreeInfo> entry : ContainersMonitorImpl.this.trackingContainers.entrySet()) {
                    ContainerId containerId = entry.getKey();
                    ProcessTreeInfo ptInfo = entry.getValue();
                    try {
                        String pId = ptInfo.getPID();
                        this.initializeProcessTrees(entry);
                        if (pId == null || !ContainersMonitorImpl.this.isResourceCalculatorAvailable()) continue;
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Constructing ProcessTree for : PID = " + pId + " ContainerId = " + containerId);
                        }
                        ResourceCalculatorProcessTree pTree = ptInfo.getProcessTree();
                        pTree.updateProcessTree();
                        long currentVmemUsage = pTree.getVirtualMemorySize();
                        long currentPmemUsage = pTree.getRssMemorySize();
                        float cpuUsagePercentPerCore = pTree.getCpuUsagePercent();
                        if (cpuUsagePercentPerCore < 0.0f) {
                            LOG.info("Skipping monitoring container " + containerId + " since CPU usage is not yet available.");
                            continue;
                        }
                        this.recordUsage(containerId, pId, pTree, ptInfo, currentVmemUsage, currentPmemUsage, trackedContainersUtilization);
                        this.checkLimit(containerId, pId, pTree, ptInfo, currentVmemUsage, currentPmemUsage);
                        vmemUsageByAllContainers += currentVmemUsage;
                        pmemByAllContainers += currentPmemUsage;
                        cpuUsagePercentPerCoreByAllContainers = (long)((float)cpuUsagePercentPerCoreByAllContainers + cpuUsagePercentPerCore);
                        this.reportResourceUsage(containerId, currentPmemUsage, cpuUsagePercentPerCore);
                    }
                    catch (Exception e) {
                        LOG.warn("Uncaught exception in ContainersMonitorImpl while monitoring resource of " + containerId, (Throwable)e);
                    }
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Total Resource Usage stats in NM by all containers : Virtual Memory= " + vmemUsageByAllContainers + ", Physical Memory= " + pmemByAllContainers + ", Total CPU usage(% per core)= " + cpuUsagePercentPerCoreByAllContainers);
                }
                ContainersMonitorImpl.this.setContainersUtilization(trackedContainersUtilization);
                NodeManagerMetrics nmMetrics = ContainersMonitorImpl.this.context.getNodeManagerMetrics();
                if (nmMetrics != null) {
                    nmMetrics.setContainerUsedMemGB(trackedContainersUtilization.getPhysicalMemory());
                    nmMetrics.setContainerUsedVMemGB(trackedContainersUtilization.getVirtualMemory());
                    nmMetrics.setContainerCpuUtilization(trackedContainersUtilization.getCPU());
                }
                try {
                    Thread.sleep(ContainersMonitorImpl.this.monitoringInterval);
                }
                catch (InterruptedException e) {
                    LOG.warn(ContainersMonitorImpl.class.getName() + " is interrupted. Exiting.");
                    break;
                }
            }
        }

        private void initializeProcessTrees(Map.Entry<ContainerId, ProcessTreeInfo> entry) throws ContainerExecutionException {
            ContainerId containerId = entry.getKey();
            ProcessTreeInfo ptInfo = entry.getValue();
            String pId = ptInfo.getPID();
            if (pId == null && (pId = ContainersMonitorImpl.this.containerExecutor.getProcessId(ptInfo.getContainerId())) != null) {
                Container container;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Tracking ProcessTree " + pId + " for the first time");
                }
                ResourceCalculatorProcessTree pt = ContainersMonitorImpl.this.getResourceCalculatorProcessTree(pId);
                ptInfo.setPid(pId);
                ptInfo.setProcessTree(pt);
                if (ContainersMonitorImpl.this.containerMetricsEnabled) {
                    ContainerMetrics usageMetrics = ContainerMetrics.forContainer(containerId, ContainersMonitorImpl.this.containerMetricsPeriodMs, ContainersMonitorImpl.this.containerMetricsUnregisterDelayMs);
                    usageMetrics.recordProcessId(pId);
                }
                if ((container = (Container)ContainersMonitorImpl.this.context.getContainers().get(containerId)) != null) {
                    Object[] ipAndHost = ContainersMonitorImpl.this.containerExecutor.getIpAndHost(container);
                    if (ipAndHost != null && ipAndHost[0] != null && ipAndHost[1] != null) {
                        container.setIpAndHost((String[])ipAndHost);
                        LOG.info(containerId + "'s ip = " + (String)ipAndHost[0] + ", and hostname = " + (String)ipAndHost[1]);
                    } else {
                        LOG.info("Can not get both ip and hostname: " + Arrays.toString(ipAndHost));
                    }
                } else {
                    LOG.info(containerId + " is missing. Not setting ip and hostname");
                }
            }
        }

        private void recordUsage(ContainerId containerId, String pId, ResourceCalculatorProcessTree pTree, ProcessTreeInfo ptInfo, long currentVmemUsage, long currentPmemUsage, ResourceUtilization trackedContainersUtilization) {
            float cpuUsagePercentPerCore = pTree.getCpuUsagePercent();
            float cpuUsageTotalCoresPercentage = cpuUsagePercentPerCore / (float)ContainersMonitorImpl.this.resourceCalculatorPlugin.getNumProcessors();
            int milliVcoresUsed = (int)(cpuUsageTotalCoresPercentage * 1000.0f * (float)ContainersMonitorImpl.this.maxVCoresAllottedForContainers / (float)ContainersMonitorImpl.this.nodeCpuPercentageForYARN);
            long vmemLimit = ptInfo.getVmemLimit();
            long pmemLimit = ptInfo.getPmemLimit();
            if (AUDITLOG.isDebugEnabled()) {
                AUDITLOG.debug(String.format("Resource usage of ProcessTree %s for container-id %s: %s CPU:%f CPU/core:%f", pId, containerId.toString(), this.formatUsageString(currentVmemUsage, vmemLimit, currentPmemUsage, pmemLimit), Float.valueOf(cpuUsagePercentPerCore), Float.valueOf(cpuUsageTotalCoresPercentage)));
            }
            trackedContainersUtilization.addTo((int)(currentPmemUsage >> 20), (int)(currentVmemUsage >> 20), (float)milliVcoresUsed / 1000.0f);
            if (ContainersMonitorImpl.this.containerMetricsEnabled) {
                ContainerMetrics.forContainer(containerId, ContainersMonitorImpl.this.containerMetricsPeriodMs, ContainersMonitorImpl.this.containerMetricsUnregisterDelayMs).recordMemoryUsage((int)(currentPmemUsage >> 20));
                ContainerMetrics.forContainer(containerId, ContainersMonitorImpl.this.containerMetricsPeriodMs, ContainersMonitorImpl.this.containerMetricsUnregisterDelayMs).recordCpuUsage((int)cpuUsagePercentPerCore, milliVcoresUsed);
            }
        }

        private void checkLimit(ContainerId containerId, String pId, ResourceCalculatorProcessTree pTree, ProcessTreeInfo ptInfo, long currentVmemUsage, long currentPmemUsage) {
            Optional<Object> isMemoryOverLimit = Optional.empty();
            String msg = "";
            int containerExitStatus = -1000;
            if (ContainersMonitorImpl.this.strictMemoryEnforcement && ContainersMonitorImpl.this.elasticMemoryEnforcement) {
                MemoryResourceHandler handler = ResourceHandlerModule.getMemoryResourceHandler();
                if (handler != null) {
                    isMemoryOverLimit = handler.isUnderOOM(containerId);
                    containerExitStatus = -104;
                    msg = containerId + " is under oom because it exceeded its physical memory limit";
                }
            } else if (ContainersMonitorImpl.this.strictMemoryEnforcement || ContainersMonitorImpl.this.elasticMemoryEnforcement) {
                isMemoryOverLimit = Optional.of(false);
            }
            if (!isMemoryOverLimit.isPresent()) {
                long vmemLimit = ptInfo.getVmemLimit();
                long pmemLimit = ptInfo.getPmemLimit();
                long curMemUsageOfAgedProcesses = pTree.getVirtualMemorySize(1);
                long curRssMemUsageOfAgedProcesses = pTree.getRssMemorySize(1);
                if (ContainersMonitorImpl.this.isVmemCheckEnabled() && ContainersMonitorImpl.this.isProcessTreeOverLimit(containerId.toString(), currentVmemUsage, curMemUsageOfAgedProcesses, vmemLimit)) {
                    long delta = currentVmemUsage - vmemLimit;
                    msg = this.formatErrorMessage("virtual", this.formatUsageString(currentVmemUsage, vmemLimit, currentPmemUsage, pmemLimit), pId, containerId, pTree, delta);
                    isMemoryOverLimit = Optional.of(true);
                    containerExitStatus = -103;
                } else if (ContainersMonitorImpl.this.isPmemCheckEnabled() && ContainersMonitorImpl.this.isProcessTreeOverLimit(containerId.toString(), currentPmemUsage, curRssMemUsageOfAgedProcesses, pmemLimit)) {
                    long delta = currentPmemUsage - pmemLimit;
                    msg = this.formatErrorMessage("physical", this.formatUsageString(currentVmemUsage, vmemLimit, currentPmemUsage, pmemLimit), pId, containerId, pTree, delta);
                    isMemoryOverLimit = Optional.of(true);
                    containerExitStatus = -104;
                }
            }
            if (isMemoryOverLimit.isPresent() && ((Boolean)isMemoryOverLimit.get()).booleanValue()) {
                LOG.warn(msg);
                if (!pTree.checkPidPgrpidForMatch()) {
                    LOG.error("Killed container process with PID " + pId + " but it is not a process group leader.");
                }
                ContainersMonitorImpl.this.eventDispatcher.getEventHandler().handle((Event)new ContainerKillEvent(containerId, containerExitStatus, msg));
                ContainersMonitorImpl.this.trackingContainers.remove(containerId);
                LOG.info("Removed ProcessTree with root " + pId);
            }
        }

        private void reportResourceUsage(ContainerId containerId, long currentPmemUsage, float cpuUsagePercentPerCore) {
            ContainerImpl container = (ContainerImpl)ContainersMonitorImpl.this.context.getContainers().get(containerId);
            if (container != null) {
                NMTimelinePublisher nmMetricsPublisher = container.getNMTimelinePublisher();
                if (nmMetricsPublisher != null) {
                    nmMetricsPublisher.reportContainerResourceUsage(container, currentPmemUsage, Float.valueOf(cpuUsagePercentPerCore));
                }
            } else {
                LOG.info(containerId + " does not exist to report");
            }
        }

        private String formatErrorMessage(String memTypeExceeded, String usageString, String pId, ContainerId containerId, ResourceCalculatorProcessTree pTree, long delta) {
            return String.format("Container [pid=%s,containerID=%s] is running %dB beyond the '%S' memory limit. ", pId, containerId, delta, memTypeExceeded) + "Current usage: " + usageString + ". Killing container.\nDump of the process-tree for " + containerId + " :\n" + pTree.getProcessTreeDump();
        }

        private String formatUsageString(long currentVmemUsage, long vmemLimit, long currentPmemUsage, long pmemLimit) {
            return String.format("%sB of %sB physical memory used; %sB of %sB virtual memory used", StringUtils.TraditionalBinaryPrefix.long2String((long)currentPmemUsage, (String)"", (int)1), StringUtils.TraditionalBinaryPrefix.long2String((long)pmemLimit, (String)"", (int)1), StringUtils.TraditionalBinaryPrefix.long2String((long)currentVmemUsage, (String)"", (int)1), StringUtils.TraditionalBinaryPrefix.long2String((long)vmemLimit, (String)"", (int)1));
        }
    }

    public static class ProcessTreeInfo {
        private ContainerId containerId;
        private String pid;
        private ResourceCalculatorProcessTree pTree;
        private long vmemLimit;
        private long pmemLimit;
        private int cpuVcores;

        public ProcessTreeInfo(ContainerId containerId, String pid, ResourceCalculatorProcessTree pTree, long vmemLimit, long pmemLimit, int cpuVcores) {
            this.containerId = containerId;
            this.pid = pid;
            this.pTree = pTree;
            this.vmemLimit = vmemLimit;
            this.pmemLimit = pmemLimit;
            this.cpuVcores = cpuVcores;
        }

        public ContainerId getContainerId() {
            return this.containerId;
        }

        public String getPID() {
            return this.pid;
        }

        public void setPid(String pid) {
            this.pid = pid;
        }

        ResourceCalculatorProcessTree getProcessTree() {
            return this.pTree;
        }

        void setProcessTree(ResourceCalculatorProcessTree mypTree) {
            this.pTree = mypTree;
        }

        public synchronized long getVmemLimit() {
            return this.vmemLimit;
        }

        public synchronized long getPmemLimit() {
            return this.pmemLimit;
        }

        public synchronized int getCpuVcores() {
            return this.cpuVcores;
        }

        synchronized void setResourceLimit(long myPmemLimit, long myVmemLimit, int myCpuVcores) {
            this.pmemLimit = myPmemLimit;
            this.vmemLimit = myVmemLimit;
            this.cpuVcores = myCpuVcores;
        }
    }

    @InterfaceAudience.Private
    public static enum ContainerMetric {
        CPU,
        MEMORY;

    }
}

