/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.entity.INodeIdentifier;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import io.hops.util.Slicer;
import java.io.IOException;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockCollection;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.NumberReplicas;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.util.CyclicIteration;
import org.apache.hadoop.util.ChunkedArrayList;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DecommissionManager {
    private static final Logger LOG = LoggerFactory.getLogger(DecommissionManager.class);
    private final Namesystem namesystem;
    private final BlockManager blockManager;
    private final HeartbeatManager hbManager;
    private final ScheduledExecutorService executor;
    private final TreeMap<DatanodeDescriptor, AbstractList<BlockInfoContiguous>> decomNodeBlocks;
    private final Queue<DatanodeDescriptor> pendingNodes;
    private Monitor monitor = null;

    DecommissionManager(Namesystem namesystem, BlockManager blockManager, HeartbeatManager hbManager) {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.hbManager = hbManager;
        this.executor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("DecommissionMonitor-%d").setDaemon(true).build());
        this.decomNodeBlocks = new TreeMap();
        this.pendingNodes = new LinkedList<DatanodeDescriptor>();
    }

    void activate(Configuration conf) {
        int intervalSecs = conf.getInt("dfs.namenode.decommission.interval", 30);
        Preconditions.checkArgument((intervalSecs >= 0 ? 1 : 0) != 0, (Object)"Cannot set a negative value for dfs.namenode.decommission.interval");
        int blocksPerInterval = conf.getInt("dfs.namenode.decommission.blocks.per.interval", 500000);
        int nodesPerInterval = Integer.MAX_VALUE;
        String deprecatedKey = "dfs.namenode.decommission.nodes.per.interval";
        String strNodes = conf.get("dfs.namenode.decommission.nodes.per.interval");
        if (strNodes != null) {
            nodesPerInterval = Integer.parseInt(strNodes);
            blocksPerInterval = Integer.MAX_VALUE;
            LOG.warn("Using deprecated configuration key {} value of {}.", (Object)"dfs.namenode.decommission.nodes.per.interval", (Object)nodesPerInterval);
            LOG.warn("Please update your configuration to use {} instead.", (Object)"dfs.namenode.decommission.blocks.per.interval");
        }
        Preconditions.checkArgument((blocksPerInterval > 0 ? 1 : 0) != 0, (Object)"Must set a positive value for dfs.namenode.decommission.blocks.per.interval");
        int maxConcurrentTrackedNodes = conf.getInt("dfs.namenode.decommission.max.concurrent.tracked.nodes", 100);
        Preconditions.checkArgument((maxConcurrentTrackedNodes >= 0 ? 1 : 0) != 0, (Object)"Cannot set a negative value for dfs.namenode.decommission.max.concurrent.tracked.nodes");
        this.monitor = new Monitor(blocksPerInterval, nodesPerInterval, maxConcurrentTrackedNodes);
        this.executor.scheduleAtFixedRate(this.monitor, intervalSecs, intervalSecs, TimeUnit.SECONDS);
        LOG.debug("Activating DecommissionManager with interval {} seconds, {} max blocks per interval, {} max nodes per interval, {} max concurrently tracked nodes.", new Object[]{intervalSecs, blocksPerInterval, nodesPerInterval, maxConcurrentTrackedNodes});
    }

    void close() {
        this.executor.shutdownNow();
        try {
            this.executor.awaitTermination(3000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @VisibleForTesting
    public void startDecommission(DatanodeDescriptor node) throws IOException {
        if (!node.isDecommissionInProgress() && !node.isDecommissioned()) {
            this.hbManager.startDecommission(node);
            if (node.isDecommissionInProgress()) {
                for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                    LOG.info("Starting decommission of {} {} with {} blocks", new Object[]{node, storage, storage.numBlocks()});
                }
                node.decommissioningStatus.setStartTime(Time.monotonicNow());
                this.pendingNodes.add(node);
            }
        } else {
            LOG.trace("startDecommission: Node {} in {}, nothing to do." + (Object)((Object)node), (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void stopDecommission(DatanodeDescriptor node) throws IOException {
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            this.hbManager.stopDecommission(node);
            if (node.isAlive) {
                this.blockManager.processOverReplicatedBlocksOnReCommission(node);
            }
            this.pendingNodes.remove((Object)node);
            this.decomNodeBlocks.remove((Object)node);
        } else {
            LOG.trace("stopDecommission: Node {} in {}, nothing to do." + (Object)((Object)node), (Object)node.getAdminState());
        }
    }

    private void setDecommissioned(DatanodeDescriptor dn) {
        dn.setDecommissioned();
        LOG.info("Decommissioning complete for node {}", (Object)dn);
    }

    private boolean isSufficientlyReplicated(BlockInfoContiguous block, BlockCollection bc, NumberReplicas numberReplicas) throws StorageException, TransactionContextException, IOException {
        int numLive;
        short numExpected = bc.getBlockReplication();
        if (!this.blockManager.isNeededReplication(block, numExpected, numLive = numberReplicas.liveReplicas())) {
            LOG.trace("Block {} does not need replication.", (Object)block);
            return true;
        }
        LOG.trace("Block {} numExpected={}, numLive={}", new Object[]{block, (int)numExpected, numLive});
        if (numExpected > numLive) {
            if (bc.isUnderConstruction() && block.equals((Object)bc.getLastBlock())) {
                if (numLive >= this.blockManager.minReplication) {
                    LOG.trace("UC block {} sufficiently-replicated since numLive ({}) >= minR ({})", new Object[]{block, numLive, this.blockManager.minReplication});
                    return true;
                }
                LOG.trace("UC block {} insufficiently-replicated since numLive ({}) < minR ({})", new Object[]{block, numLive, this.blockManager.minReplication});
            } else if (numLive >= this.blockManager.defaultReplication) {
                return true;
            }
        }
        return false;
    }

    private static void logBlockReplicationInfo(Block block, BlockCollection bc, DatanodeDescriptor srcNode, NumberReplicas num, Iterable<DatanodeStorageInfo> storages) {
        int curReplicas = num.liveReplicas();
        short curExpectedReplicas = bc.getBlockReplication();
        StringBuilder nodeList = new StringBuilder();
        for (DatanodeStorageInfo storage : storages) {
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            nodeList.append((Object)node);
            nodeList.append(" ");
        }
        LOG.info("Block: " + block + ", Expected Replicas: " + curExpectedReplicas + ", live replicas: " + curReplicas + ", corrupt replicas: " + num.corruptReplicas() + ", decommissioned replicas: " + num.decommissioned() + ", decommissioning replicas: " + num.decommissioning() + ", excess replicas: " + num.excessReplicas() + ", Is Open File: " + bc.isUnderConstruction() + ", Datanodes having this block: " + nodeList + ", Current Datanode: " + (Object)((Object)srcNode) + ", Is current datanode decommissioning: " + srcNode.isDecommissionInProgress());
    }

    @VisibleForTesting
    public int getNumPendingNodes() {
        return this.pendingNodes.size();
    }

    @VisibleForTesting
    public int getNumTrackedNodes() {
        return this.decomNodeBlocks.size();
    }

    @VisibleForTesting
    public int getNumNodesChecked() {
        return this.monitor.numNodesChecked;
    }

    @VisibleForTesting
    void runMonitor() throws ExecutionException, InterruptedException {
        Future<?> f = this.executor.submit(this.monitor);
        f.get();
    }

    private class Monitor
    implements Runnable {
        private final int numBlocksPerCheck;
        private final int numNodesPerCheck;
        private final int maxConcurrentTrackedNodes;
        private int numBlocksChecked = 0;
        private int numNodesChecked = 0;
        private DatanodeDescriptor iterkey = new DatanodeDescriptor(null, new DatanodeID("", "", "", 0, 0, 0, 0));

        Monitor(int numBlocksPerCheck, int numNodesPerCheck, int maxConcurrentTrackedNodes) {
            this.numBlocksPerCheck = numBlocksPerCheck;
            this.numNodesPerCheck = numNodesPerCheck;
            this.maxConcurrentTrackedNodes = maxConcurrentTrackedNodes;
        }

        private boolean exceededNumBlocksPerCheck() {
            LOG.trace("Processed {} blocks so far this tick", (Object)this.numBlocksChecked);
            return this.numBlocksChecked >= this.numBlocksPerCheck;
        }

        @Deprecated
        private boolean exceededNumNodesPerCheck() {
            LOG.trace("Processed {} nodes so far this tick", (Object)this.numNodesChecked);
            return this.numNodesChecked >= this.numNodesPerCheck;
        }

        @Override
        public void run() {
            if (!DecommissionManager.this.namesystem.isRunning()) {
                LOG.info("Namesystem is not running, skipping decommissioning checks.");
                return;
            }
            this.numBlocksChecked = 0;
            this.numNodesChecked = 0;
            this.processPendingNodes();
            try {
                this.check();
            }
            catch (IOException ex) {
                LOG.warn("Failled to check decommission blocks", (Throwable)ex);
            }
            if (this.numBlocksChecked + this.numNodesChecked > 0) {
                LOG.info("Checked {} blocks and {} nodes this tick", (Object)this.numBlocksChecked, (Object)this.numNodesChecked);
            }
        }

        private void processPendingNodes() {
            while (!(DecommissionManager.this.pendingNodes.isEmpty() || this.maxConcurrentTrackedNodes != 0 && DecommissionManager.this.decomNodeBlocks.size() >= this.maxConcurrentTrackedNodes)) {
                DecommissionManager.this.decomNodeBlocks.put(DecommissionManager.this.pendingNodes.poll(), null);
            }
        }

        private void check() throws IOException {
            Iterator it = new CyclicIteration(DecommissionManager.this.decomNodeBlocks, this.iterkey).iterator();
            LinkedList<DatanodeDescriptor> toRemove = new LinkedList<DatanodeDescriptor>();
            while (it.hasNext() && !this.exceededNumBlocksPerCheck() && !this.exceededNumNodesPerCheck()) {
                ++this.numNodesChecked;
                Map.Entry entry = it.next();
                DatanodeDescriptor dn = entry.getKey();
                AbstractList<BlockInfoContiguous> blocks = (AbstractList<BlockInfoContiguous>)entry.getValue();
                boolean fullScan = false;
                if (blocks == null) {
                    LOG.debug("Newly-added node {}, doing full scan to find insufficiently-replicated blocks.", (Object)dn);
                    blocks = this.handleInsufficientlyReplicated(dn);
                    DecommissionManager.this.decomNodeBlocks.put(dn, blocks);
                    fullScan = true;
                } else {
                    LOG.debug("Processing decommission-in-progress node {}", (Object)dn);
                    this.pruneSufficientlyReplicated(dn, blocks);
                }
                if (blocks.size() == 0) {
                    if (!fullScan) {
                        LOG.debug("Node {} has finished replicating current set of blocks, checking with the full block map.", (Object)dn);
                        blocks = this.handleInsufficientlyReplicated(dn);
                        DecommissionManager.this.decomNodeBlocks.put(dn, blocks);
                    }
                    boolean isHealthy = DecommissionManager.this.blockManager.isNodeHealthyForDecommission(dn);
                    if (blocks.size() == 0 && isHealthy) {
                        DecommissionManager.this.setDecommissioned(dn);
                        toRemove.add(dn);
                        LOG.debug("Node {} is sufficiently replicated and healthy, marked as decommissioned.", (Object)dn);
                    } else if (LOG.isDebugEnabled()) {
                        StringBuilder b = new StringBuilder("Node {} ");
                        if (isHealthy) {
                            b.append("is ");
                        } else {
                            b.append("isn't ");
                        }
                        b.append("healthy and still needs to replicate {} more blocks, decommissioning is still in progress.");
                        LOG.debug(b.toString(), (Object)dn, (Object)blocks.size());
                    }
                } else {
                    LOG.debug("Node {} still has {} blocks to replicate before it is a candidate to finish decommissioning.", (Object)dn, (Object)blocks.size());
                }
                this.iterkey = dn;
            }
            for (DatanodeDescriptor dn : toRemove) {
                Preconditions.checkState((boolean)dn.isDecommissioned(), (Object)"Removing a node that is not yet decommissioned!");
                DecommissionManager.this.decomNodeBlocks.remove((Object)dn);
            }
        }

        private void pruneSufficientlyReplicated(final DatanodeDescriptor datanode, AbstractList<BlockInfoContiguous> blocks) throws IOException {
            HashSet<Long> inodeIdsSet = new HashSet<Long>();
            final HashMap<Long, ArrayList<BlockInfoContiguous>> blocksPerInodes = new HashMap<Long, ArrayList<BlockInfoContiguous>>();
            for (BlockInfoContiguous block : blocks) {
                inodeIdsSet.add(block.getInodeId());
                ArrayList<BlockInfoContiguous> blocksForInode = (ArrayList<BlockInfoContiguous>)blocksPerInodes.get(block.getInodeId());
                if (blocksForInode == null) {
                    blocksForInode = new ArrayList<BlockInfoContiguous>();
                    blocksPerInodes.put(block.getInodeId(), blocksForInode);
                }
                blocksForInode.add(block);
            }
            final ArrayList inodeIds = new ArrayList(inodeIdsSet);
            final ConcurrentLinkedQueue toRemove = new ConcurrentLinkedQueue();
            final AtomicInteger underReplicatedBlocks = new AtomicInteger(0);
            final AtomicInteger decommissionOnlyReplicas = new AtomicInteger(0);
            final AtomicInteger underReplicatedInOpenFiles = new AtomicInteger(0);
            try {
                Slicer.slice((int)inodeIds.size(), (int)((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getRemovalBatchSize(), (int)((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getRemovalNoThreads(), (ExecutorService)((FSNamesystem)DecommissionManager.this.namesystem).getFSOperationsExecutor(), (Slicer.OperationHandler)new Slicer.OperationHandler(){

                    public void handle(int startIndex, int endIndex) throws Exception {
                        final List ids = inodeIds.subList(startIndex, endIndex);
                        HopsTransactionalRequestHandler checkReplicationHandler = new HopsTransactionalRequestHandler(HDFSOperationType.CHECK_REPLICATION_IN_PROGRESS){
                            List<INodeIdentifier> inodeIdentifiers;

                            @Override
                            public void setUp() throws StorageException {
                                this.inodeIdentifiers = INodeUtil.resolveINodesFromIds(ids);
                            }

                            public void acquireLock(TransactionLocks locks) throws IOException {
                                LockFactory lf = LockFactory.getInstance();
                                if (!this.inodeIdentifiers.isEmpty()) {
                                    locks.add(lf.getMultipleINodesLock(this.inodeIdentifiers, TransactionLockTypes.INodeLockType.WRITE)).add(lf.getSqlBatchedBlocksLock()).add(lf.getSqlBatchedBlocksRelated(LockFactory.BLK.RE, LockFactory.BLK.ER, LockFactory.BLK.CR, LockFactory.BLK.UR, LockFactory.BLK.PE));
                                }
                            }

                            public Object performTask() throws IOException {
                                ArrayList blocksToProcess = new ArrayList();
                                for (Long inodeIds : ids) {
                                    blocksToProcess.addAll((Collection)blocksPerInodes.get(inodeIds));
                                }
                                HashSet<Long> existingInodes = new HashSet<Long>();
                                for (INodeIdentifier inode : this.inodeIdentifiers) {
                                    existingInodes.add(inode.getInodeId());
                                }
                                Monitor.this.processBlocksForDecomInternal(datanode, blocksToProcess.iterator(), null, true, existingInodes, toRemove, underReplicatedBlocks, underReplicatedInOpenFiles, decommissionOnlyReplicas);
                                return null;
                            }
                        };
                        checkReplicationHandler.handle();
                    }
                });
                blocks.removeAll(toRemove);
            }
            catch (Exception ex) {
                throw new IOException(ex);
            }
            datanode.decommissioningStatus.set(underReplicatedBlocks.get(), decommissionOnlyReplicas.get(), underReplicatedInOpenFiles.get());
        }

        private AbstractList<BlockInfoContiguous> handleInsufficientlyReplicated(final DatanodeDescriptor datanode) throws IOException {
            final ConcurrentLinkedQueue insuf = new ConcurrentLinkedQueue();
            Map<Long, Long> blocksOnNode = datanode.getAllStorageReplicas(((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getNumBuckets(), ((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getBlockFetcherNBThreads(), ((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getBlockFetcherBucketsPerThread(), ((FSNamesystem)DecommissionManager.this.namesystem).getFSOperationsExecutor());
            final HashMap<Long, ArrayList<Long>> inodeIdsToBlockMap = new HashMap<Long, ArrayList<Long>>();
            for (Map.Entry<Long, Long> entry : blocksOnNode.entrySet()) {
                ArrayList<Long> list = (ArrayList<Long>)inodeIdsToBlockMap.get(entry.getValue());
                if (list == null) {
                    list = new ArrayList<Long>();
                    inodeIdsToBlockMap.put(entry.getValue(), list);
                }
                list.add(entry.getKey());
            }
            final ArrayList inodeIds = new ArrayList(inodeIdsToBlockMap.keySet());
            final ConcurrentLinkedQueue toRemove = new ConcurrentLinkedQueue();
            final AtomicInteger underReplicatedBlocks = new AtomicInteger(0);
            final AtomicInteger decommissionOnlyReplicas = new AtomicInteger(0);
            final AtomicInteger underReplicatedInOpenFiles = new AtomicInteger(0);
            try {
                Slicer.slice((int)inodeIds.size(), (int)((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getRemovalBatchSize(), (int)((FSNamesystem)DecommissionManager.this.namesystem).getBlockManager().getRemovalNoThreads(), (ExecutorService)((FSNamesystem)DecommissionManager.this.namesystem).getFSOperationsExecutor(), (Slicer.OperationHandler)new Slicer.OperationHandler(){

                    public void handle(int startIndex, int endIndex) throws Exception {
                        final List ids = inodeIds.subList(startIndex, endIndex);
                        HopsTransactionalRequestHandler checkReplicationHandler = new HopsTransactionalRequestHandler(HDFSOperationType.CHECK_REPLICATION_IN_PROGRESS){
                            List<INodeIdentifier> inodeIdentifiers;

                            @Override
                            public void setUp() throws StorageException {
                                this.inodeIdentifiers = INodeUtil.resolveINodesFromIds(ids);
                            }

                            public void acquireLock(TransactionLocks locks) throws IOException {
                                LockFactory lf = LockFactory.getInstance();
                                locks.add(lf.getMultipleINodesLock(this.inodeIdentifiers, TransactionLockTypes.INodeLockType.WRITE)).add(lf.getSqlBatchedBlocksLock()).add(lf.getSqlBatchedBlocksRelated(LockFactory.BLK.RE, LockFactory.BLK.ER, LockFactory.BLK.CR, LockFactory.BLK.UR, LockFactory.BLK.PE));
                            }

                            public Object performTask() throws IOException {
                                ArrayList<BlockInfoContiguous> toCheck = new ArrayList<BlockInfoContiguous>();
                                HashSet<Long> existingInodes = new HashSet<Long>();
                                for (INodeIdentifier identifier : this.inodeIdentifiers) {
                                    existingInodes.add(identifier.getInodeId());
                                    Iterator iterator = ((List)inodeIdsToBlockMap.get(identifier.getInodeId())).iterator();
                                    while (iterator.hasNext()) {
                                        long blockId = (Long)iterator.next();
                                        BlockInfoContiguous block = (BlockInfoContiguous)((Object)EntityManager.find((FinderType)BlockInfoContiguous.Finder.ByBlockIdAndINodeId, (Object[])new Object[]{blockId}));
                                        toCheck.add(block);
                                    }
                                }
                                Monitor.this.processBlocksForDecomInternal(datanode, toCheck.iterator(), insuf, false, existingInodes, toRemove, underReplicatedBlocks, underReplicatedInOpenFiles, decommissionOnlyReplicas);
                                return null;
                            }
                        };
                        checkReplicationHandler.handle();
                    }
                });
            }
            catch (Exception ex) {
                throw new IOException(ex);
            }
            datanode.decommissioningStatus.set(underReplicatedBlocks.get(), decommissionOnlyReplicas.get(), underReplicatedInOpenFiles.get());
            ChunkedArrayList insufficient = new ChunkedArrayList();
            insufficient.addAll(insuf);
            return insufficient;
        }

        private void processBlocksForDecomInternal(DatanodeDescriptor datanode, Iterator<BlockInfoContiguous> it, Queue<BlockInfoContiguous> insufficientlyReplicated, boolean pruneSufficientlyReplicated, Set<Long> existingInodes, Queue<BlockInfoContiguous> toRemove, AtomicInteger underReplicatedBlocks, AtomicInteger underReplicatedInOpenFiles, AtomicInteger decommissionOnlyReplicas) throws IOException {
            boolean firstReplicationLog = true;
            while (it.hasNext()) {
                int liveReplicas;
                ++this.numBlocksChecked;
                BlockInfoContiguous block = it.next();
                if (((DecommissionManager)DecommissionManager.this).blockManager.blocksMap.getStoredBlock(block) == null || !existingInodes.contains(block.getInodeId())) {
                    LOG.trace("Removing unknown block {}", (Object)block);
                    toRemove.add(block);
                    continue;
                }
                BlockCollection bc = ((DecommissionManager)DecommissionManager.this).blockManager.blocksMap.getBlockCollection(block);
                if (bc == null) continue;
                NumberReplicas num = DecommissionManager.this.blockManager.countNodes(block);
                int curReplicas = liveReplicas = num.liveReplicas();
                if (DecommissionManager.this.blockManager.isNeededReplication(block, bc.getBlockReplication(), liveReplicas) && !((DecommissionManager)DecommissionManager.this).blockManager.neededReplications.contains(block) && ((DecommissionManager)DecommissionManager.this).blockManager.pendingReplications.getNumReplicas(block) == 0 && DecommissionManager.this.namesystem.isPopulatingReplQueues()) {
                    ((DecommissionManager)DecommissionManager.this).blockManager.neededReplications.add(block, curReplicas, num.decommissionedAndDecommissioning(), bc.getBlockReplication());
                }
                if (DecommissionManager.this.isSufficientlyReplicated(block, bc, num)) {
                    if (!pruneSufficientlyReplicated) continue;
                    toRemove.add(block);
                    continue;
                }
                if (insufficientlyReplicated != null) {
                    insufficientlyReplicated.add(block);
                }
                if (firstReplicationLog) {
                    DecommissionManager.logBlockReplicationInfo(block, bc, datanode, num, ((DecommissionManager)DecommissionManager.this).blockManager.blocksMap.getStorages(block));
                    firstReplicationLog = false;
                }
                underReplicatedBlocks.incrementAndGet();
                if (bc.isUnderConstruction()) {
                    underReplicatedInOpenFiles.incrementAndGet();
                }
                if (curReplicas != 0 || num.decommissionedAndDecommissioning() <= 0) continue;
                decommissionOnlyReplicas.incrementAndGet();
            }
        }
    }
}

