/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.resourcemanager.security;

import com.google.common.annotations.VisibleForTesting;
import io.hops.hadoop.shaded.org.apache.commons.math3.random.RandomDataGenerator;
import io.hops.hadoop.shaded.org.apache.commons.math3.util.Pair;
import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalUnit;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.util.BackOff;
import org.apache.hadoop.util.DateUtils;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.event.EventHandler;
import org.apache.hadoop.yarn.server.resourcemanager.RMContext;
import org.apache.hadoop.yarn.server.resourcemanager.rmapp.RMAppSecurityMaterialRenewedEvent;
import org.apache.hadoop.yarn.server.resourcemanager.security.RMAppSecurityActions;
import org.apache.hadoop.yarn.server.resourcemanager.security.RMAppSecurityHandler;
import org.apache.hadoop.yarn.server.resourcemanager.security.RMAppSecurityManager;

public class JWTSecurityHandler
implements RMAppSecurityHandler<JWTSecurityManagerMaterial, JWTMaterialParameter> {
    private static final Log LOG = LogFactory.getLog(JWTSecurityHandler.class);
    private final RMContext rmContext;
    private final RMAppSecurityManager rmAppSecurityManager;
    private final EventHandler eventHandler;
    private String[] jwtAudience;
    private Configuration config;
    private boolean jwtEnabled;
    private RMAppSecurityActions rmAppSecurityActions;
    private Pair<Long, TemporalUnit> validityPeriod;
    private final Map<ApplicationId, ScheduledFuture> renewalTasks;
    private ScheduledExecutorService renewalExecutorService;
    private Long leeway;
    private Thread invalidationEventsHandler;
    private static final int INVALIDATION_EVENTS_QUEUE_SIZE = 100;
    private final BlockingQueue<JWTInvalidationEvent> invalidationEvents;
    private final RandomDataGenerator random;

    public JWTSecurityHandler(RMContext rmContext, RMAppSecurityManager rmAppSecurityManager) {
        this.rmContext = rmContext;
        this.rmAppSecurityManager = rmAppSecurityManager;
        this.renewalTasks = new ConcurrentHashMap<ApplicationId, ScheduledFuture>();
        this.invalidationEvents = new ArrayBlockingQueue<JWTInvalidationEvent>(100);
        this.eventHandler = rmContext.getDispatcher().getEventHandler();
        this.random = new RandomDataGenerator();
    }

    @Override
    public void init(Configuration config) throws Exception {
        LOG.info((Object)"Initializing JWT Security Handler");
        this.config = config;
        this.jwtEnabled = config.getBoolean(YarnConfiguration.RM_JWT_ENABLED, false);
        this.jwtAudience = config.getTrimmedStrings(YarnConfiguration.RM_JWT_AUDIENCE, new String[]{"job"});
        this.renewalExecutorService = this.rmAppSecurityManager.getRenewalExecutorService();
        String validity = config.get(YarnConfiguration.RM_JWT_VALIDITY_PERIOD, "30m");
        this.validityPeriod = this.rmAppSecurityManager.parseInterval(validity, YarnConfiguration.RM_JWT_VALIDITY_PERIOD);
        String expirationLeewayConf = config.get(YarnConfiguration.RM_JWT_EXPIRATION_LEEWAY, "30m");
        Pair<Long, TemporalUnit> expirationLeeway = this.rmAppSecurityManager.parseInterval(expirationLeewayConf, YarnConfiguration.RM_JWT_EXPIRATION_LEEWAY);
        if (((ChronoUnit)expirationLeeway.getSecond()).compareTo(ChronoUnit.SECONDS) < 0) {
            throw new IllegalArgumentException("Value of " + YarnConfiguration.RM_JWT_EXPIRATION_LEEWAY + " should be at least seconds");
        }
        this.leeway = Duration.of((Long)expirationLeeway.getFirst(), (TemporalUnit)expirationLeeway.getSecond()).getSeconds();
        if (this.jwtEnabled) {
            this.rmAppSecurityActions = this.rmAppSecurityManager.getRmAppCertificateActions();
        }
    }

    @Override
    public void start() throws Exception {
        LOG.info((Object)"Starting JWT Security Handler");
        if (this.isJWTEnabled()) {
            this.invalidationEventsHandler = this.createInvalidationEventsHandler();
            this.invalidationEventsHandler.setDaemon(false);
            this.invalidationEventsHandler.setName("JWT-InvalidationEventsHandler");
            this.invalidationEventsHandler.start();
        }
    }

    @Override
    public void stop() throws Exception {
        LOG.info((Object)"Stopping JWT Security Handler");
        if (this.invalidationEventsHandler != null) {
            this.invalidationEventsHandler.interrupt();
        }
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected Thread createInvalidationEventsHandler() {
        return new InvalidationEventsHandler();
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    public BlockingQueue<JWTInvalidationEvent> getInvalidationEvents() {
        return this.invalidationEvents;
    }

    @Override
    public JWTSecurityManagerMaterial generateMaterial(JWTMaterialParameter parameter) throws Exception {
        if (!this.isJWTEnabled()) {
            return null;
        }
        ApplicationId appId = parameter.getApplicationId();
        LOG.debug((Object)("Generating JWT for application " + appId));
        this.prepareJWTGenerationParameters(parameter);
        String jwt = this.generateInternal(parameter);
        return new JWTSecurityManagerMaterial(appId, jwt, parameter.getExpirationDate());
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected void prepareJWTGenerationParameters(JWTMaterialParameter parameter) {
        parameter.setAudiences(this.jwtAudience);
        LocalDateTime now = this.getNow();
        LocalDateTime expirationTime = now.plus((Long)this.validityPeriod.getFirst(), (TemporalUnit)this.validityPeriod.getSecond());
        parameter.setExpirationDate(expirationTime);
        parameter.setValidNotBefore(now);
        parameter.setRenewable(false);
        parameter.setExpLeeway(this.leeway.intValue());
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected String generateInternal(JWTMaterialParameter parameter) throws URISyntaxException, IOException, GeneralSecurityException {
        return this.rmAppSecurityActions.generateJWT(parameter);
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected LocalDateTime getNow() {
        return DateUtils.getNow();
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected Pair<Long, TemporalUnit> getValidityPeriod() {
        return this.validityPeriod;
    }

    @VisibleForTesting
    protected Map<ApplicationId, ScheduledFuture> getRenewalTasks() {
        return this.renewalTasks;
    }

    @VisibleForTesting
    protected Configuration getConfig() {
        return this.config;
    }

    @VisibleForTesting
    protected RMAppSecurityManager getRmAppSecurityManager() {
        return this.rmAppSecurityManager;
    }

    @Override
    public void registerRenewer(JWTMaterialParameter parameter) {
        if (!this.isJWTEnabled()) {
            return;
        }
        if (!this.renewalTasks.containsKey(parameter.getApplicationId())) {
            ScheduledFuture<?> task = this.renewalExecutorService.schedule(this.createJWTRenewalTask(parameter.getApplicationId(), parameter.appUser, parameter.token), this.computeScheduledDelay(parameter.getExpirationDate()), TimeUnit.SECONDS);
            this.renewalTasks.put(parameter.getApplicationId(), task);
        }
    }

    private long computeScheduledDelay(LocalDateTime expiration) {
        long upperLimit = Math.max(this.leeway - 5L, 5L);
        long delayFromExpiration = this.random.nextLong(3L, upperLimit);
        Duration duration = Duration.between(this.getNow(), expiration);
        return duration.getSeconds() + delayFromExpiration;
    }

    public void deregisterFromRenewer(ApplicationId appId) {
        if (!this.isJWTEnabled()) {
            return;
        }
        ScheduledFuture task = this.renewalTasks.get(appId);
        if (task != null) {
            task.cancel(true);
        }
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected Runnable createJWTRenewalTask(ApplicationId appId, String appUser, String token) {
        return new JWTRenewer(appId, appUser, token);
    }

    @Override
    public boolean revokeMaterial(JWTMaterialParameter parameter, Boolean blocking) {
        if (!this.isJWTEnabled()) {
            return true;
        }
        ApplicationId appId = parameter.getApplicationId();
        try {
            LOG.info((Object)("Invalidating JWT for application: " + appId));
            this.deregisterFromRenewer(appId);
            this.putToInvalidationQueue(appId);
            return true;
        }
        catch (InterruptedException ex) {
            LOG.warn((Object)("Shutting down while putting invalidation event to queue for application " + appId));
            return false;
        }
    }

    private void putToInvalidationQueue(ApplicationId appId) throws InterruptedException {
        this.invalidationEvents.put(new JWTInvalidationEvent(appId.toString()));
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected void revokeInternal(String signingKeyName) {
        if (!this.isJWTEnabled()) {
            return;
        }
        try {
            this.rmAppSecurityActions.invalidateJWT(signingKeyName);
        }
        catch (IOException | URISyntaxException | GeneralSecurityException ex) {
            LOG.error((Object)("Could not invalidate JWT with signing key " + signingKeyName), (Throwable)ex);
        }
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected String renewInternal(JWTMaterialParameter param) throws URISyntaxException, IOException, GeneralSecurityException {
        if (!this.isJWTEnabled()) {
            return null;
        }
        return this.rmAppSecurityActions.renewJWT(param);
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected boolean isJWTEnabled() {
        return this.jwtEnabled;
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    protected class InvalidationEventsHandler
    extends Thread {
        protected InvalidationEventsHandler() {
        }

        private void drain() {
            ArrayList events = new ArrayList(JWTSecurityHandler.this.invalidationEvents.size());
            JWTSecurityHandler.this.invalidationEvents.drainTo(events);
            for (JWTInvalidationEvent event : events) {
                JWTSecurityHandler.this.revokeInternal(event.signingKeyName);
            }
        }

        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    JWTInvalidationEvent event = (JWTInvalidationEvent)JWTSecurityHandler.this.invalidationEvents.take();
                    JWTSecurityHandler.this.revokeInternal(event.signingKeyName);
                }
                catch (InterruptedException ex) {
                    LOG.info((Object)"JWT InvalidationEventHandler interrupted. Draining queue...");
                    this.drain();
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    protected static class JWTInvalidationEvent {
        private final String signingKeyName;

        protected JWTInvalidationEvent(String signingKeyName) {
            this.signingKeyName = signingKeyName;
        }

        protected String getSigningKeyName() {
            return this.signingKeyName;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o instanceof JWTInvalidationEvent) {
                return this.signingKeyName.equals(((JWTInvalidationEvent)o).signingKeyName);
            }
            return false;
        }

        public int hashCode() {
            return this.signingKeyName.hashCode();
        }
    }

    private class JWTRenewer
    implements Runnable {
        private final ApplicationId appId;
        private final String appUser;
        private final String token;
        private final BackOff backOff;
        private long backOffTime = 0L;

        public JWTRenewer(ApplicationId appId, String appUser, String token) {
            this.appId = appId;
            this.appUser = appUser;
            this.token = token;
            this.backOff = JWTSecurityHandler.this.rmAppSecurityManager.createBackOffPolicy();
        }

        @Override
        public void run() {
            try {
                LOG.debug((Object)("Renewing JWT for application " + this.appId));
                JWTMaterialParameter jwtParam = new JWTMaterialParameter(this.appId, this.appUser);
                jwtParam.setToken(this.token);
                JWTSecurityHandler.this.prepareJWTGenerationParameters(jwtParam);
                String jwt = JWTSecurityHandler.this.renewInternal(jwtParam);
                JWTSecurityHandler.this.renewalTasks.remove(this.appId);
                JWTSecurityManagerMaterial jwtMaterial = new JWTSecurityManagerMaterial(this.appId, jwt, jwtParam.getExpirationDate());
                JWTSecurityHandler.this.eventHandler.handle(new RMAppSecurityMaterialRenewedEvent<JWTSecurityManagerMaterial>(this.appId, jwtMaterial));
                LOG.debug((Object)("Renewed JWT for application " + this.appId));
            }
            catch (Exception ex) {
                JWTSecurityHandler.this.renewalTasks.remove(this.appId);
                this.backOffTime = this.backOff.getBackOffInMillis();
                if (this.backOffTime != -1L) {
                    LOG.warn((Object)("Failed to renew JWT for application " + this.appId + ". Retrying in " + this.backOffTime + " ms"));
                    ScheduledFuture<?> task = JWTSecurityHandler.this.renewalExecutorService.schedule(this, this.backOffTime, TimeUnit.MILLISECONDS);
                    JWTSecurityHandler.this.renewalTasks.put(this.appId, task);
                }
                LOG.error((Object)("Failed to renew JWT for application " + this.appId + ". Failed more than 4 times, giving up"), (Throwable)ex);
            }
        }
    }

    public static class JWTMaterialParameter
    extends RMAppSecurityManager.SecurityManagerMaterial {
        private final String appUser;
        private String token;
        private String[] audiences;
        private LocalDateTime expirationDate;
        private LocalDateTime validNotBefore;
        private boolean renewable;
        private int expLeeway;

        public JWTMaterialParameter(ApplicationId applicationId, String appUser) {
            super(applicationId);
            this.appUser = appUser;
        }

        public String getAppUser() {
            return this.appUser;
        }

        public String[] getAudiences() {
            return this.audiences;
        }

        public void setAudiences(String[] audiences) {
            this.audiences = audiences;
        }

        public LocalDateTime getExpirationDate() {
            return this.expirationDate;
        }

        public void setExpirationDate(LocalDateTime expirationDate) {
            this.expirationDate = expirationDate;
        }

        public LocalDateTime getValidNotBefore() {
            return this.validNotBefore;
        }

        public void setValidNotBefore(LocalDateTime validNotBefore) {
            this.validNotBefore = validNotBefore;
        }

        public boolean isRenewable() {
            return this.renewable;
        }

        public void setRenewable(boolean renewable) {
            this.renewable = renewable;
        }

        public int getExpLeeway() {
            return this.expLeeway;
        }

        public void setExpLeeway(int expLeeway) {
            this.expLeeway = expLeeway;
        }

        public String getToken() {
            return this.token;
        }

        public void setToken(String token) {
            this.token = token;
        }

        public int hashCode() {
            int result = 17;
            result = 31 * result + this.appUser.hashCode();
            result = 31 * result + this.getApplicationId().hashCode();
            if (this.expirationDate != null) {
                result = 31 * result + this.expirationDate.hashCode();
            }
            return result;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o instanceof JWTMaterialParameter) {
                JWTMaterialParameter otherMaterial = (JWTMaterialParameter)o;
                if (this.expirationDate != null) {
                    return this.appUser.equals(otherMaterial.appUser) && this.getApplicationId().equals((Object)otherMaterial.getApplicationId()) && this.expirationDate.equals(otherMaterial.getExpirationDate());
                }
                return this.appUser.equals(otherMaterial.appUser) && this.getApplicationId().equals((Object)otherMaterial.getApplicationId());
            }
            return false;
        }
    }

    public class JWTSecurityManagerMaterial
    extends RMAppSecurityManager.SecurityManagerMaterial {
        private final String token;
        private final LocalDateTime expirationDate;

        public JWTSecurityManagerMaterial(ApplicationId applicationId, String token, LocalDateTime expirationDate) {
            super(applicationId);
            this.token = token;
            this.expirationDate = expirationDate;
        }

        public String getToken() {
            return this.token;
        }

        public LocalDateTime getExpirationDate() {
            return this.expirationDate;
        }
    }
}

