/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ipc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.ipc.ClientId;
import org.apache.hadoop.ipc.RpcConstants;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.ipc.metrics.RetryCacheMetrics;
import org.apache.hadoop.util.LightWeightCache;
import org.apache.hadoop.util.LightWeightGSet;

@InterfaceAudience.Private
public class RetryCache {
    public static final Log LOG = LogFactory.getLog(RetryCache.class);
    protected final RetryCacheMetrics retryCacheMetrics;
    protected static final int MAX_CAPACITY = 16;
    protected LightWeightGSet<CacheEntry, CacheEntry> set;
    protected final long expirationTime;
    private String cacheName;
    protected final ReentrantLock lock = new ReentrantLock();

    public RetryCache(String cacheName, double percentage, long expirationTime) {
        int capacity = LightWeightGSet.computeCapacity(percentage, cacheName);
        capacity = capacity > 16 ? capacity : 16;
        this.set = new LightWeightCache<CacheEntry, CacheEntry>(capacity, capacity, expirationTime, 0L);
        this.expirationTime = expirationTime;
        this.cacheName = cacheName;
        this.retryCacheMetrics = RetryCacheMetrics.create(this);
    }

    protected static boolean skipRetryCache() {
        return !Server.isRpcInvocation() || Server.getCallId() < 0 || Arrays.equals(Server.getClientId(), RpcConstants.DUMMY_CLIENT_ID);
    }

    public void lock() {
        this.lock.lock();
    }

    public void unlock() {
        this.lock.unlock();
    }

    protected void incrCacheClearedCounter() {
        this.retryCacheMetrics.incrCacheCleared();
    }

    @VisibleForTesting
    public LightWeightGSet<CacheEntry, CacheEntry> getCacheSet() {
        return this.set;
    }

    @VisibleForTesting
    public RetryCacheMetrics getMetricsForTests() {
        return this.retryCacheMetrics;
    }

    public String getCacheName() {
        return this.cacheName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheEntry waitForCompletion(CacheEntry newEntry) {
        CacheEntry mapEntry = null;
        this.lock.lock();
        try {
            mapEntry = this.set.get(newEntry);
            if (mapEntry == null) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Adding Rpc request clientId " + newEntry.clientIdMsb + newEntry.clientIdLsb + " callId " + newEntry.callId + " to retryCache"));
                }
                this.set.put(newEntry);
                this.retryCacheMetrics.incrCacheUpdated();
                CacheEntry cacheEntry = newEntry;
                return cacheEntry;
            }
            this.retryCacheMetrics.incrCacheHit();
        }
        finally {
            this.lock.unlock();
        }
        Preconditions.checkNotNull((Object)mapEntry, (Object)"Entry from the cache should not be null");
        CacheEntry cacheEntry = mapEntry;
        synchronized (cacheEntry) {
            while (mapEntry.state == 0) {
                try {
                    mapEntry.wait();
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
            if (mapEntry.state != 1) {
                mapEntry.state = 0;
            }
        }
        return mapEntry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addCacheEntry(byte[] clientId, int callId) {
        CacheEntry newEntry = new CacheEntry(clientId, callId, System.currentTimeMillis() + this.expirationTime, true);
        this.lock.lock();
        try {
            this.set.put(newEntry);
        }
        finally {
            this.lock.unlock();
        }
        this.retryCacheMetrics.incrCacheUpdated();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addCacheEntryWithPayload(byte[] clientId, int callId, Object payload) {
        CacheEntryWithPayload newEntry = new CacheEntryWithPayload(clientId, callId, payload, System.currentTimeMillis() + this.expirationTime, true);
        this.lock.lock();
        try {
            this.set.put(newEntry);
        }
        finally {
            this.lock.unlock();
        }
        this.retryCacheMetrics.incrCacheUpdated();
    }

    private static CacheEntry newEntry(long expirationTime) {
        return new CacheEntry(Server.getClientId(), Server.getCallId(), System.currentTimeMillis() + expirationTime);
    }

    private static CacheEntryWithPayload newEntry(Object payload, long expirationTime) {
        return new CacheEntryWithPayload(Server.getClientId(), Server.getCallId(), payload, System.currentTimeMillis() + expirationTime);
    }

    public static CacheEntry waitForCompletion(RetryCache cache) {
        if (RetryCache.skipRetryCache()) {
            return null;
        }
        return cache != null ? cache.waitForCompletion(RetryCache.newEntry(cache.expirationTime)) : null;
    }

    public static CacheEntryWithPayload waitForCompletion(RetryCache cache, Object payload) {
        if (RetryCache.skipRetryCache()) {
            return null;
        }
        return (CacheEntryWithPayload)(cache != null ? cache.waitForCompletion(RetryCache.newEntry(payload, cache.expirationTime)) : null);
    }

    public static void setState(CacheEntry e, boolean success) {
        if (e == null) {
            return;
        }
        e.completed(success);
    }

    public static void setState(CacheEntryWithPayload e, boolean success, Object payload) {
        if (e == null) {
            return;
        }
        e.payload = payload;
        e.completed(success);
    }

    public static void clear(RetryCache cache) {
        if (cache != null) {
            cache.set.clear();
            cache.incrCacheClearedCounter();
        }
    }

    public static class CacheEntryWithPayload
    extends CacheEntry {
        private Object payload;

        CacheEntryWithPayload(byte[] clientId, int callId, Object payload, long expirationTime) {
            super(clientId, callId, expirationTime);
            this.payload = payload;
        }

        CacheEntryWithPayload(byte[] clientId, int callId, Object payload, long expirationTime, boolean success) {
            super(clientId, callId, expirationTime, success);
            this.payload = payload;
        }

        @Override
        public boolean equals(Object obj) {
            return super.equals(obj);
        }

        @Override
        public int hashCode() {
            return super.hashCode();
        }

        public Object getPayload() {
            return this.payload;
        }
    }

    public static class CacheEntry
    implements LightWeightCache.Entry {
        public static final byte INPROGRESS = 0;
        public static final byte SUCCESS = 1;
        public static final byte FAILED = 2;
        protected byte state = 0;
        protected final long clientIdMsb;
        protected final long clientIdLsb;
        protected final byte[] clientIdByte;
        protected final int callId;
        private final long expirationTime;
        private LightWeightGSet.LinkedElement next;

        CacheEntry(byte[] clientId, int callId, long expirationTime) {
            Preconditions.checkArgument((clientId.length == 16 ? 1 : 0) != 0, (Object)("Invalid clientId - length is " + clientId.length + " expected length " + 16));
            this.clientIdMsb = ClientId.getMsb(clientId);
            this.clientIdLsb = ClientId.getLsb(clientId);
            this.callId = callId;
            this.expirationTime = expirationTime;
            this.clientIdByte = clientId;
        }

        CacheEntry(byte[] clientId, int callId, long expirationTime, boolean success) {
            this(clientId, callId, expirationTime);
            this.state = (byte)(success ? 1 : 2);
        }

        public byte getState() {
            return this.state;
        }

        public int getCallId() {
            return this.callId;
        }

        public byte[] getClientId() {
            return this.clientIdByte;
        }

        private static int hashCode(long value) {
            return (int)(value ^ value >>> 32);
        }

        public int hashCode() {
            return (CacheEntry.hashCode(this.clientIdMsb) * 31 + CacheEntry.hashCode(this.clientIdLsb)) * 31 + this.callId;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof CacheEntry)) {
                return false;
            }
            CacheEntry other = (CacheEntry)obj;
            return this.callId == other.callId && this.clientIdMsb == other.clientIdMsb && this.clientIdLsb == other.clientIdLsb;
        }

        @Override
        public void setNext(LightWeightGSet.LinkedElement next) {
            this.next = next;
        }

        @Override
        public LightWeightGSet.LinkedElement getNext() {
            return this.next;
        }

        synchronized void completed(boolean success) {
            this.state = (byte)(success ? 1 : 2);
            this.notifyAll();
        }

        public synchronized boolean isSuccess() {
            return this.state == 1;
        }

        @Override
        public void setExpirationTime(long timeNano) {
        }

        @Override
        public long getExpirationTime() {
            return this.expirationTime;
        }

        public String toString() {
            return new UUID(this.clientIdMsb, this.clientIdLsb).toString() + ":" + this.callId + ":" + this.state;
        }
    }
}

