/*
 * Decompiled with CFR 0.152.
 */
package io.hops.security;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.hops.security.HopsUtil;
import io.hops.security.HopsX509AuthenticationException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.cert.X509Certificate;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;

public class HopsX509Authenticator {
    private static final Log LOG = LogFactory.getLog(HopsX509Authenticator.class);
    private final Configuration conf;
    private final Cache<String, InetAddress> trustedHostnames;

    public HopsX509Authenticator(Configuration conf) {
        this.conf = conf;
        this.trustedHostnames = CacheBuilder.newBuilder().maximumSize(500L).expireAfterWrite(30L, TimeUnit.MINUTES).build();
    }

    public void authenticateConnection(UserGroupInformation user, X509Certificate clientCertificate, InetAddress remoteAddress) throws HopsX509AuthenticationException {
        this.authenticateConnection(user, clientCertificate, remoteAddress, null);
    }

    public void authenticateConnection(UserGroupInformation user, X509Certificate clientCertificate, InetAddress remoteAddress, String protocolName) throws HopsX509AuthenticationException {
        if (!this.isHopsTLS()) {
            return;
        }
        Preconditions.checkNotNull((Object)user, (Object)"UserGroupInformation should not be null");
        Preconditions.checkNotNull((Object)clientCertificate, (Object)"Client X.509 certificate should not be null");
        LOG.debug((Object)("Authenticating user: " + user.getUserName()));
        String username = user.getUserName();
        if (username == null) {
            throw new HopsX509AuthenticationException("Could not extract username from UGI");
        }
        String subjectDN = clientCertificate.getSubjectX500Principal().getName("RFC2253");
        String cn = HopsUtil.extractCNFromSubject(subjectDN);
        if (cn == null) {
            throw new HopsX509AuthenticationException("Problematic CN in client certificate: " + subjectDN);
        }
        String org = HopsUtil.extractOFromSubject(subjectDN);
        if (org != null && (Strings.isNullOrEmpty((String)protocolName) || !protocolName.equalsIgnoreCase("WebHDFS"))) {
            user.addApplicationId(org);
        }
        if (username.equals(cn)) {
            LOG.debug((Object)("Authenticated user " + username + " - Username matches CN"));
            return;
        }
        Preconditions.checkNotNull((Object)remoteAddress, (Object)"Remote address should not be null");
        String locality = HopsUtil.extractLFromSubject(subjectDN);
        if (locality == null) {
            throw new HopsX509AuthenticationException("Incoming RPC claims to be a from a system user but the Locality (L) field of its X.509 is null or cannot be parsed");
        }
        if (this.isTrustedConnection(remoteAddress, cn, username, locality)) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Could not authenticate client with CN ").append(cn).append(" remote IP ").append(remoteAddress).append(" and username ").append(username);
        if (protocolName != null) {
            sb.append(" for protocol ").append(protocolName);
        }
        throw new HopsX509AuthenticationException(sb.toString());
    }

    public boolean isTrustedConnection(InetAddress remoteAddress, String cnFQDN) throws HopsX509AuthenticationException {
        return this.isTrustedConnection(remoteAddress, cnFQDN, null, null);
    }

    public boolean isTrustedConnection(InetAddress remoteAddress, String cnFQDN, String username, String locality) throws HopsX509AuthenticationException {
        InetAddress address = this.isTrustedFQDN(cnFQDN);
        if (address != null && this.isTrustedConnectionInternal(remoteAddress, address, username, locality)) {
            LOG.debug((Object)("CN " + cnFQDN + " is an FQDN and it has already been authenticated"));
            return true;
        }
        try {
            address = InetAddress.getByName(cnFQDN);
            if (this.isTrustedConnectionInternal(remoteAddress, address, username, locality)) {
                this.trustedHostnames.put((Object)cnFQDN, (Object)address);
                LOG.debug((Object)("CN " + cnFQDN + " is an FQDN and we managed to resolve it and it matches the remote address"));
                return true;
            }
        }
        catch (UnknownHostException ex) {
            LOG.error((Object)("Could not resolve host " + cnFQDN), (Throwable)ex);
            throw new HopsX509AuthenticationException("Hostname " + cnFQDN + " is not resolvable and could not authenticate user " + username);
        }
        return false;
    }

    private boolean isHopsTLS() {
        return this.conf.getBoolean("ipc.server.ssl.enabled", false);
    }

    private boolean isTrustedConnectionInternal(InetAddress expectedAddress, InetAddress actualAddress, String expectedUsername, String actualUsername) {
        if (expectedUsername == null && actualUsername == null) {
            return this.doesAddressMatch(expectedAddress, actualAddress);
        }
        return this.doesAddressMatch(expectedAddress, actualAddress) && this.doesUsernameMatch(expectedUsername, actualUsername);
    }

    private boolean doesAddressMatch(InetAddress expected, InetAddress actual) {
        if (expected.isLoopbackAddress() && actual.isLoopbackAddress()) {
            return true;
        }
        return expected.equals(actual);
    }

    private boolean doesUsernameMatch(String expected, String actual) {
        return expected.equals(actual);
    }

    @VisibleForTesting
    protected InetAddress isTrustedFQDN(String fqdn) {
        return (InetAddress)this.trustedHostnames.getIfPresent((Object)fqdn);
    }
}

