/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.PrivilegedExceptionAction;
import java.util.concurrent.Semaphore;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.test.LambdaTestUtils;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestFileSystemCaching {
    @Test
    public void testCacheEnabled() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fs1 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        FileSystem fs2 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        Assert.assertSame((Object)fs1, (Object)fs2);
    }

    @Test
    public void testDefaultFsUris() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.defaultfs.impl", DefaultFs.class.getName());
        URI defaultUri = URI.create("defaultfs://host");
        FileSystem.setDefaultUri((Configuration)conf, (URI)defaultUri);
        FileSystem defaultFs = FileSystem.get((Configuration)conf);
        Assert.assertEquals((Object)defaultUri, (Object)defaultFs.getUri());
        Assert.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs:/"), (Configuration)conf));
        Assert.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs:///"), (Configuration)conf));
        Assert.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs://host"), (Configuration)conf));
        Assert.assertNotSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("defaultfs://host2"), (Configuration)conf));
        Assert.assertSame((Object)defaultFs, (Object)FileSystem.get((URI)URI.create("/"), (Configuration)conf));
        LambdaTestUtils.intercept(UnsupportedFileSystemException.class, () -> FileSystem.get((URI)URI.create("//host"), (Configuration)conf));
        LambdaTestUtils.intercept(UnsupportedFileSystemException.class, () -> FileSystem.get((URI)URI.create("//host2"), (Configuration)conf));
    }

    @Test
    public void testCacheEnabledWithInitializeForeverFS() throws Exception {
        final Configuration conf = new Configuration();
        Thread t = new Thread(){

            @Override
            public void run() {
                conf.set("fs.localfs1.impl", "org.apache.hadoop.fs.TestFileSystemCaching$InitializeForeverFileSystem");
                try {
                    FileSystem.get((URI)new URI("localfs1://a"), (Configuration)conf);
                }
                catch (IOException | URISyntaxException e) {
                    e.printStackTrace();
                }
            }
        };
        t.start();
        InitializeForeverFileSystem.sem.acquire();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
        t.interrupt();
        t.join();
    }

    @Test
    public void testCacheDisabled() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.uncachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        conf.setBoolean("fs.uncachedfile.impl.disable.cache", true);
        FileSystem fs1 = FileSystem.get((URI)new URI("uncachedfile://a"), (Configuration)conf);
        FileSystem fs2 = FileSystem.get((URI)new URI("uncachedfile://a"), (Configuration)conf);
        Assert.assertNotSame((Object)fs1, (Object)fs2);
    }

    @Test
    public <T extends TokenIdentifier> void testCacheForUgi() throws Exception {
        Configuration conf = this.newConf();
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser((String)"foo");
        UserGroupInformation ugiB = UserGroupInformation.createRemoteUser((String)"bar");
        FileSystem fsA = this.getCachedFS(ugiA, conf);
        FileSystem fsA1 = this.getCachedFS(ugiA, conf);
        Assert.assertSame((Object)fsA, (Object)fsA1);
        FileSystem fsB = this.getCachedFS(ugiB, conf);
        Assert.assertNotSame((Object)fsA, (Object)fsB);
        Token t1 = (Token)Mockito.mock(Token.class);
        UserGroupInformation ugiA2 = UserGroupInformation.createRemoteUser((String)"foo");
        fsA = this.getCachedFS(ugiA2, conf);
        Assert.assertNotSame((Object)fsA, (Object)fsA1);
        ugiA.addToken(t1);
        fsA = this.getCachedFS(ugiA, conf);
        Assert.assertSame((Object)fsA, (Object)fsA1);
    }

    @Test
    public void testDifferentFSForDifferentAppID() throws Exception {
        final Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser((String)"foo");
        FileSystem fsA = (FileSystem)ugiA.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
            }
        });
        UserGroupInformation ugiAppA = UserGroupInformation.createRemoteUser((String)"foo");
        ugiAppA.addApplicationId("appA");
        FileSystem fsAppA = (FileSystem)ugiAppA.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
            }
        });
        UserGroupInformation ugiAppB = UserGroupInformation.createRemoteUser((String)"foo");
        ugiAppA.addApplicationId("appB");
        FileSystem fsAppB = (FileSystem)ugiAppB.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf);
            }
        });
        Assert.assertNotSame((Object)fsA, (Object)fsAppA);
        Assert.assertNotSame((Object)fsA, (Object)fsAppB);
        Assert.assertNotSame((Object)fsAppA, (Object)fsAppB);
    }

    private FileSystem getCachedFS(UserGroupInformation ugi, Configuration conf) throws IOException, InterruptedException {
        return (FileSystem)ugi.doAs(() -> FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf));
    }

    @Test
    public void testUserFS() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fsU1 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        FileSystem fsU2 = FileSystem.get((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"foo");
        Assert.assertNotSame((Object)fsU1, (Object)fsU2);
    }

    private Configuration newConf() throws IOException {
        Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass((String)"file", null).getName());
        return conf;
    }

    @Test
    public void testFsUniqueness() throws Exception {
        Configuration conf = this.newConf();
        FileSystem fs1 = FileSystem.get((Configuration)conf);
        FileSystem fs2 = FileSystem.get((Configuration)conf);
        Assert.assertSame((Object)fs1, (Object)fs2);
        fs1 = FileSystem.newInstance((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        fs2 = FileSystem.newInstance((URI)new URI("cachedfile://a"), (Configuration)conf, (String)"bar");
        Assert.assertTrue((fs1 != fs2 && !fs1.equals(fs2) ? 1 : 0) != 0);
        fs1.close();
        fs2.close();
    }

    @Test
    public void testCloseAllForUGI() throws Exception {
        Configuration conf = this.newConf();
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser((String)"foo");
        FileSystem fsA = this.getCachedFS(ugiA, conf);
        FileSystem fsA1 = this.getCachedFS(ugiA, conf);
        Assert.assertSame((Object)fsA, (Object)fsA1);
        FileSystem.closeAllForUGI((UserGroupInformation)ugiA);
        fsA1 = this.getCachedFS(ugiA, conf);
        Assert.assertNotSame((Object)fsA, (Object)fsA1);
    }

    @Test
    public void testDelete() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        fs.delete(path, false);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)false));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        fs.delete(path, true);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        Path path = new Path("/a");
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            Assert.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExitFNF() throws IOException {
        Path path;
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            path = new Path("/a");
            Assert.assertFalse((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testDeleteOnExitRemoved() throws IOException {
        Path path;
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            path = new Path("/a");
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            Assert.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testCancelDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        try (FilterFileSystem fs = new FilterFileSystem(mockFs);){
            Path path = new Path("/a");
            Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
            Assert.assertTrue((boolean)fs.deleteOnExit(path));
            ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
            Assert.assertTrue((boolean)fs.cancelDeleteOnExit(path));
            Assert.assertFalse((boolean)fs.cancelDeleteOnExit(path));
            Mockito.reset((Object[])new FileSystem[]{mockFs});
            fs.close();
        }
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)Mockito.any(Path.class));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testCacheIncludesURIUserInfo() throws Throwable {
        URI containerA = new URI("wasb://a@account.blob.core.windows.net");
        URI containerB = new URI("wasb://b@account.blob.core.windows.net");
        Configuration conf = new Configuration(false);
        FileSystem.Cache.Key keyA = new FileSystem.Cache.Key(containerA, conf);
        FileSystem.Cache.Key keyB = new FileSystem.Cache.Key(containerB, conf);
        Assert.assertNotEquals((Object)keyA, (Object)keyB);
        Assert.assertNotEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://account.blob.core.windows.net"), conf));
        Assert.assertEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://A@account.blob.core.windows.net"), conf));
        Assert.assertNotEquals((Object)keyA, (Object)new FileSystem.Cache.Key(new URI("wasb://a:password@account.blob.core.windows.net"), conf));
    }

    public static class InitializeForeverFileSystem
    extends LocalFileSystem {
        static final Semaphore sem = new Semaphore(0);

        public void initialize(URI uri, Configuration conf) throws IOException {
            sem.release();
            try {
                while (true) {
                    Thread.sleep(1000L);
                }
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }

    private static class DefaultFs
    extends LocalFileSystem {
        URI uri;

        private DefaultFs() {
        }

        public void initialize(URI uri, Configuration conf) {
            this.uri = uri;
        }

        public URI getUri() {
            return this.uri;
        }
    }
}

