/*
 * Decompiled with CFR 0.152.
 */
package io.hops.security;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jwt.JWTClaimsSet;
import io.hops.security.MockJWTIssuer;
import io.hops.security.ServiceJWTManager;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.channels.FileLock;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.math3.util.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.ssl.KeyStoreTestUtil;
import org.apache.hadoop.util.DateUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class TestServiceJWTManager {
    private static final Log LOG = LogFactory.getLog(TestServiceJWTManager.class);
    private static MockJWTIssuer jwtIssuer;
    private static String sslConfFilename;
    private static Path sslServerPath;
    private static String classpath;
    private Configuration conf;
    private ServiceJWTManager jwtManager;

    @BeforeClass
    public static void beforeClass() throws Exception {
        byte[] jwtIssuerSecret = new byte[32];
        Random rand = new Random();
        rand.nextBytes(jwtIssuerSecret);
        jwtIssuer = new MockJWTIssuer(jwtIssuerSecret);
        classpath = KeyStoreTestUtil.getClasspathDir(TestServiceJWTManager.class);
        sslConfFilename = TestServiceJWTManager.class.getSimpleName() + ".ssl-server.xml";
        sslServerPath = Paths.get(classpath, sslConfFilename);
    }

    @Before
    public void beforeTest() throws Exception {
        this.conf = new Configuration();
        Configuration sslConf = new Configuration(false);
        Pair<String, String[]> tokens = this.generateMasterAndRenewTokens(5);
        sslConf.set("hops.jwt-manager.master-token", (String)tokens.getFirst());
        for (int i = 0; i < ((String[])tokens.getSecond()).length; ++i) {
            sslConf.set(String.format("hops.jwt-manager.renew-token-%d", i), ((String[])tokens.getSecond())[i]);
        }
        KeyStoreTestUtil.saveConfig(sslServerPath.toFile(), sslConf);
        this.conf.set("hadoop.ssl.server.conf", sslConfFilename);
        this.conf.setBoolean("ipc.server.ssl.enabled", true);
    }

    @After
    public void after() throws Exception {
        if (this.jwtManager != null) {
            this.jwtManager.stop();
        }
        if (sslServerPath != null) {
            sslServerPath.toFile().delete();
        }
    }

    @Test
    public void testUpdatingJWTConf() throws Exception {
        LocalDateTime now = DateUtils.getNow();
        LocalDateTime expiresAt = now.plus(10L, ChronoUnit.MINUTES);
        Pair<String, String[]> tokens = this.generateMasterAndRenewTokens(5, now, expiresAt);
        this.jwtManager = new TestingServiceJWTManager("TestingJWTManager", (String)tokens.getFirst(), (String[])tokens.getSecond(), expiresAt);
        this.jwtManager.init(this.conf);
        this.jwtManager.start();
        TimeUnit.MILLISECONDS.sleep(500L);
        Assert.assertTrue((boolean)((TestingServiceJWTManager)this.jwtManager).renewed);
        Configuration sslConf = new Configuration(false);
        sslConf.addResource(this.conf.get("hadoop.ssl.server.conf"));
        String newMasterTokenConf = sslConf.get("hops.jwt-manager.master-token", "");
        Assert.assertEquals((Object)tokens.getFirst(), (Object)newMasterTokenConf);
        for (int i = 0; i < ((String[])tokens.getSecond()).length; ++i) {
            String newRenewalTokenConf = sslConf.get(String.format("hops.jwt-manager.renew-token-%d", i), "");
            Assert.assertEquals((Object)((String[])tokens.getSecond())[i], (Object)newRenewalTokenConf);
        }
        Assert.assertTrue((boolean)this.isUpdaterThreadStillRunning(this.jwtManager.getExecutorService()));
    }

    @Test
    public void testUpdateRetryOnFailure() throws Exception {
        LocalDateTime now = DateUtils.getNow();
        LocalDateTime expiresAt = now.plus(10L, ChronoUnit.MINUTES);
        Pair<String, String[]> tokens = this.generateMasterAndRenewTokens(5, now, expiresAt);
        this.jwtManager = new FailingTestingServiceJWTManager("FailingTestingJWTManager", (String)tokens.getFirst(), (String[])tokens.getSecond(), expiresAt, 3);
        this.jwtManager.init(this.conf);
        this.jwtManager.start();
        int secondsWaited = 0;
        while (!((TestingServiceJWTManager)this.jwtManager).renewed && secondsWaited++ < 10) {
            TimeUnit.SECONDS.sleep(1L);
        }
        Assert.assertTrue((boolean)((TestingServiceJWTManager)this.jwtManager).renewed);
        Assert.assertTrue((((FailingTestingServiceJWTManager)this.jwtManager).usedOneTimeTokens.size() > 1 ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.isUpdaterThreadStillRunning(this.jwtManager.getExecutorService()));
    }

    private boolean isUpdaterThreadStillRunning(ExecutorService executor) {
        Future<?> future = executor.submit(new Runnable(){

            @Override
            public void run() {
            }
        });
        try {
            future.get(500L, TimeUnit.MILLISECONDS);
        }
        catch (Exception ex) {
            return true;
        }
        return false;
    }

    @Test
    public void testGiveUpAfterRetries() throws Exception {
        LocalDateTime now = DateUtils.getNow();
        LocalDateTime expiresAt = now.plus(10L, ChronoUnit.MINUTES);
        int numOfRenewTokens = 5;
        Pair<String, String[]> tokens = this.generateMasterAndRenewTokens(numOfRenewTokens, now, expiresAt);
        this.jwtManager = new FailingTestingServiceJWTManager("FailingTestingJWTManager", (String)tokens.getFirst(), (String[])tokens.getSecond(), expiresAt, Integer.MAX_VALUE);
        this.jwtManager.init(this.conf);
        this.jwtManager.start();
        int secondsWaited = 0;
        while (((FailingTestingServiceJWTManager)this.jwtManager).usedOneTimeTokens.size() < numOfRenewTokens && secondsWaited++ < 30) {
            TimeUnit.SECONDS.sleep(1L);
        }
        Assert.assertFalse((boolean)((TestingServiceJWTManager)this.jwtManager).renewed);
        Assert.assertEquals((long)numOfRenewTokens, (long)((FailingTestingServiceJWTManager)this.jwtManager).usedOneTimeTokens.size());
        Assert.assertFalse((boolean)this.isUpdaterThreadStillRunning(this.jwtManager.getExecutorService()));
    }

    @Test
    public void onlyOneRenewerShouldRun() throws Exception {
        int numOfRenewTokens = 5;
        LocalDateTime now = DateUtils.getNow();
        LocalDateTime expiresAt = now.plus(10L, ChronoUnit.MINUTES);
        Pair<String, String[]> tokens = this.generateMasterAndRenewTokens(numOfRenewTokens, now, expiresAt);
        this.conf.set("hops.jwt-manager.master-token-validity", "2s");
        TestingServiceJWTManager jwtManager0 = new TestingServiceJWTManager("TestingServiceJWTMgm0", (String)tokens.getFirst(), (String[])tokens.getSecond(), expiresAt);
        jwtManager0.init(this.conf);
        this.jwtManager = new TestingServiceJWTManager("TestingServiceJWTMgm1", (String)tokens.getFirst(), (String[])tokens.getSecond(), expiresAt);
        this.jwtManager.init(this.conf);
        String initialJWTManagerMasterToken = this.jwtManager.getMasterToken();
        Assert.assertNotEquals((Object)"", (Object)initialJWTManagerMasterToken);
        jwtManager0.start();
        int secondsWaited = 0;
        while (!jwtManager0.renewed && secondsWaited++ < 5) {
            TimeUnit.SECONDS.sleep(1L);
        }
        Assert.assertTrue((boolean)jwtManager0.renewed);
        this.jwtManager.start();
        secondsWaited = 0;
        while (!((TestingServiceJWTManager)this.jwtManager).tried2lock && secondsWaited++ < 5) {
            TimeUnit.SECONDS.sleep(1L);
        }
        Assert.assertTrue((boolean)((TestingServiceJWTManager)this.jwtManager).tried2lock);
        Assert.assertFalse((boolean)((TestingServiceJWTManager)this.jwtManager).renewed);
        TimeUnit.SECONDS.sleep(this.conf.getTimeDuration("hops.jwt-manager.master-token-validity", 2L, TimeUnit.SECONDS));
        Assert.assertNotEquals((Object)initialJWTManagerMasterToken, (Object)this.jwtManager.getMasterToken());
        Assert.assertEquals((Object)tokens.getFirst(), (Object)this.jwtManager.getMasterToken());
        jwtManager0.stop();
        secondsWaited = 0;
        while (!((TestingServiceJWTManager)this.jwtManager).renewed && secondsWaited++ < 5) {
            TimeUnit.SECONDS.sleep(1L);
        }
        Assert.assertTrue((boolean)((TestingServiceJWTManager)this.jwtManager).renewed);
    }

    private JWTClaimsSet.Builder generateJWTClaimsBuilder(LocalDateTime nbf, LocalDateTime expiration, String subject) {
        JWTClaimsSet.Builder claimsSetBuilder = new JWTClaimsSet.Builder();
        claimsSetBuilder.subject(subject);
        claimsSetBuilder.notBeforeTime(DateUtils.localDateTime2Date((LocalDateTime)nbf));
        claimsSetBuilder.expirationTime(DateUtils.localDateTime2Date((LocalDateTime)expiration));
        return claimsSetBuilder;
    }

    private Pair<String, String[]> generateMasterAndRenewTokens(int numOfRenewTokens) throws JOSEException {
        LocalDateTime now = DateUtils.getNow();
        return this.generateMasterAndRenewTokens(numOfRenewTokens, now, now.plus(10L, ChronoUnit.MINUTES));
    }

    private Pair<String, String[]> generateMasterAndRenewTokens(int numOfRenewTokens, LocalDateTime nbf, LocalDateTime expiresAt) throws JOSEException {
        String masterToken = jwtIssuer.generate(this.generateJWTClaimsBuilder(nbf, expiresAt, "master_token"));
        Assert.assertNotNull((Object)masterToken);
        String[] renewTokens = new String[numOfRenewTokens];
        for (int i = 0; i < renewTokens.length; ++i) {
            String renewToken = jwtIssuer.generate(this.generateJWTClaimsBuilder(nbf, expiresAt, "renew_token_" + i));
            Assert.assertNotNull((Object)renewToken);
            renewTokens[i] = renewToken;
        }
        return new Pair((Object)masterToken, (Object)renewTokens);
    }

    private class FailingTestingServiceJWTManager
    extends TestingServiceJWTManager {
        private final int succeedAfterRetries;
        private int failures;
        private final Set<String> usedOneTimeTokens;

        public FailingTestingServiceJWTManager(String name, String newMasterToken, String[] newRenewalTokens, LocalDateTime expiresAt, int succeedAfterRetries) {
            super(name, newMasterToken, newRenewalTokens, expiresAt);
            this.failures = 0;
            this.usedOneTimeTokens = new HashSet<String>(newRenewalTokens.length);
            this.succeedAfterRetries = succeedAfterRetries;
        }

        @Override
        protected ServiceJWTManager.ServiceTokenDTO renewServiceJWT(String token, String oneTimeToken, LocalDateTime expiresAt, LocalDateTime notBefore) throws URISyntaxException, IOException {
            this.usedOneTimeTokens.add(oneTimeToken);
            if (this.failures++ < this.succeedAfterRetries) {
                throw new IOException("oops");
            }
            return super.renewServiceJWT(token, oneTimeToken, expiresAt, notBefore);
        }
    }

    private class TestingServiceJWTManager
    extends ServiceJWTManager {
        private final String newMasterToken;
        private final String[] newRenewalTokens;
        private final LocalDateTime expiresAt;
        private boolean renewed;
        private boolean tried2lock;

        public TestingServiceJWTManager(String name, String newMasterToken, String[] newRenewalTokens, LocalDateTime expiresAt) {
            super(name);
            this.renewed = false;
            this.tried2lock = false;
            this.newMasterToken = newMasterToken;
            this.newRenewalTokens = newRenewalTokens;
            this.expiresAt = expiresAt;
        }

        protected FileLock tryAndGetLock() {
            this.tried2lock = true;
            return super.tryAndGetLock();
        }

        protected ServiceJWTManager.ServiceTokenDTO renewServiceJWT(String token, String oneTimeToken, LocalDateTime expiresAt, LocalDateTime notBefore) throws URISyntaxException, IOException {
            ServiceJWTManager.JWTDTO jwt = new ServiceJWTManager.JWTDTO();
            jwt.setToken(this.newMasterToken);
            jwt.setNbf(DateUtils.localDateTime2Date((LocalDateTime)notBefore));
            jwt.setExpiresAt(DateUtils.localDateTime2Date((LocalDateTime)this.expiresAt));
            ServiceJWTManager.ServiceTokenDTO serviceTokenResponse = new ServiceJWTManager.ServiceTokenDTO((ServiceJWTManager)this);
            serviceTokenResponse.setJwt(jwt);
            serviceTokenResponse.setRenewTokens(this.newRenewalTokens);
            this.renewed = true;
            return serviceTokenResponse;
        }

        protected void invalidateServiceJWT(String token2invalidate) throws URISyntaxException, IOException {
        }

        protected boolean isTime2Renew(LocalDateTime now, LocalDateTime tokenExpiration) {
            return !this.renewed;
        }
    }
}

