/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.lock.INodeLock;
import io.hops.transaction.lock.Lock;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.IOException;
import java.util.Collection;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguousUnderConstruction;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.log4j.Level;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.core.Is;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;

public class TestFileTruncate {
    static final Log LOG;
    static final int BLOCK_SIZE = 4;
    static final short REPLICATION = 3;
    static final int DATANODE_NUM = 3;
    static final int SUCCESS_ATTEMPTS = 300;
    static final int RECOVERY_ATTEMPTS = 600;
    static final long SLEEP = 100L;
    static final long LOW_SOFTLIMIT = 100L;
    static final long LOW_HARDLIMIT = 200L;
    static final int SHORT_HEARTBEAT = 1;
    static Configuration conf;
    static MiniDFSCluster cluster;
    static DistributedFileSystem fs;

    @BeforeClass
    public static void startUp() throws IOException {
        conf = new HdfsConfiguration();
        conf.setLong("dfs.blockreport.numbuckets", 1L);
        conf.setLong("dfs.namenode.fs-limits.min-block-size", 4L);
        conf.setInt("dfs.bytes-per-checksum", 4);
        conf.setInt("dfs.heartbeat.interval", 1);
        conf.setLong("dfs.namenode.replication.pending.timeout-sec", 1L);
        cluster = new MiniDFSCluster.Builder(conf).format(true).numDataNodes(3).nameNodePort(8020).waitSafeMode(true).build();
        fs = cluster.getFileSystem();
    }

    @AfterClass
    public static void tearDown() throws IOException {
        if (fs != null) {
            fs.close();
        }
        if (cluster != null) {
            cluster.shutdown();
        }
    }

    @Test
    public void testBasicTruncate() throws IOException {
        int startingFileSize = 12;
        Path parent = new Path("/test");
        fs.mkdirs(parent);
        fs.setQuota(parent, 100L, 1000L);
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        for (int fileLength = startingFileSize; fileLength > 0; fileLength -= 3) {
            for (int toTruncate = 0; toTruncate <= fileLength; ++toTruncate) {
                Path p = new Path(parent, "testBasicTruncate" + fileLength);
                TestFileTruncate.writeContents(contents, fileLength, p);
                int newLength = fileLength - toTruncate;
                boolean isReady = fs.truncate(p, (long)newLength);
                LOG.info((Object)("fileLength=" + fileLength + ", newLength=" + newLength + ", toTruncate=" + toTruncate + ", isReady=" + isReady));
                Assert.assertEquals((String)"File must be closed for zero truncate or truncating at the block boundary", (Object)isReady, (Object)(toTruncate == 0 || newLength % 4 == 0 ? 1 : 0));
                if (!isReady) {
                    TestFileTruncate.checkBlockRecovery(p);
                }
                ContentSummary cs = fs.getContentSummary(parent);
                Assert.assertEquals((String)"Bad disk space usage", (long)cs.getSpaceConsumed(), (long)(newLength * 3));
                TestFileTruncate.checkFullFile(p, newLength, contents);
            }
        }
        fs.delete(parent, true);
    }

    @Test
    public void testMultipleTruncate() throws IOException {
        Path dir = new Path("/testMultipleTruncate");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[400];
        DFSUtil.getRandom().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int n = data.length;
        while (n > 0) {
            int newLength = DFSUtil.getRandom().nextInt(n);
            boolean isReady = fs.truncate(p, (long)newLength);
            LOG.info((Object)("newLength=" + newLength + ", isReady=" + isReady));
            Assert.assertEquals((String)"File must be closed for truncating at the block boundary", (Object)isReady, (Object)(newLength % 4 == 0 ? 1 : 0));
            Assert.assertEquals((String)"Truncate is not idempotent", (Object)isReady, (Object)fs.truncate(p, (long)newLength));
            if (!isReady) {
                TestFileTruncate.checkBlockRecovery(p);
            }
            TestFileTruncate.checkFullFile(p, newLength, data);
            n = newLength;
        }
        fs.delete(dir, true);
    }

    @Test
    public void testTruncateWithOtherOperations() throws IOException {
        Path dir = new Path("/testTruncateOtherOperations");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[8];
        DFSUtil.getRandom().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int newLength = data.length - 1;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        fs.setReplication(p, (short)2);
        fs.setPermission(p, FsPermission.createImmutable((short)292));
        Path q = new Path(dir, "newFile");
        fs.rename(p, q);
        TestFileTruncate.checkBlockRecovery(q);
        TestFileTruncate.checkFullFile(q, newLength, data);
        cluster.restartNameNode(new String[0]);
        TestFileTruncate.checkFullFile(q, newLength, data);
        fs.delete(dir, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testTruncateFailure() throws IOException {
        int startingFileSize = 10;
        int toTruncate = 1;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path dir = new Path("/dir");
        Path p = new Path(dir, "testTruncateFailure");
        out.write(contents, 0, startingFileSize);
        try (FSDataOutputStream out = fs.create(p, false, 4, (short)3, 4L);){
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail on open file.");
        }
        out = fs.append(p);
        try {
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail for append.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        finally {
            out.close();
        }
        try {
            fs.truncate(p, -1L);
            Assert.fail((String)"Truncate must fail for a negative new length.");
        }
        catch (HadoopIllegalArgumentException expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a negative file size", (Throwable)expected);
        }
        try {
            fs.truncate(p, (long)(startingFileSize + 1));
            Assert.fail((String)"Truncate must fail for a larger new length.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a larger file size", (Throwable)expected);
        }
        try {
            fs.truncate(dir, 0L);
            Assert.fail((String)"Truncate must fail for a directory.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Path is not a file", (Throwable)expected);
        }
        try {
            fs.truncate(new Path(dir, "non-existing"), 0L);
            Assert.fail((String)"Truncate must fail for a non-existing file.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"File does not exist", (Throwable)expected);
        }
        fs.setPermission(p, FsPermission.createImmutable((short)436));
        UserGroupInformation fooUgi = UserGroupInformation.createUserForTesting((String)"foo", (String[])new String[]{"foo"});
        try {
            FileSystem foofs = DFSTestUtil.getFileSystemAs(fooUgi, conf);
            foofs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail for no WRITE permission.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Permission denied", (Throwable)expected);
        }
        cluster.shutdownDataNodes();
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(100L, 200L);
        int newLength = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertThat((String)"truncate should have triggered block recovery.", (Object)isReady, (Matcher)Is.is((Object)false));
        try {
            fs.truncate(p, 0L);
            Assert.fail((String)"Truncate must fail since a trancate is already in pregress.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        boolean recoveryTriggered = false;
        for (int i = 0; i < 600; ++i) {
            String leaseHolder = NameNodeAdapter.getLeaseHolderForPath(cluster.getNameNode(), p.toUri().getPath());
            if (leaseHolder.equals("HDFS_NameNode")) {
                recoveryTriggered = true;
                break;
            }
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        Assert.assertThat((String)"lease recovery should have occurred in ~60000 ms.", (Object)recoveryTriggered, (Matcher)Is.is((Object)true));
        cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(60000L, 3600000L);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(p, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=60000L)
    public void testTruncateWithDataNodesRestart() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path parent = new Path("/test");
        Path p = new Path(parent, "testTruncateWithDataNodesRestart");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn = 0;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        cluster.getDataNodes().get(dn).shutdown();
        try {
            boolean isReady = fs.truncate(p, (long)newLength);
            Assert.assertFalse((boolean)isReady);
        }
        finally {
            cluster.restartDataNode(dn, true, true);
            cluster.waitActive();
        }
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        Assert.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        Assert.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        Assert.assertEquals((long)cluster.getBlockFile(dn, newBlock.getBlock()).length(), (long)newBlock.getBlockSize());
        Assert.assertTrue((boolean)cluster.getBlockMetadataFile(dn, newBlock.getBlock()).getName().endsWith(newBlock.getBlock().getGenerationStamp() + ".meta"));
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLength));
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(parent, true);
    }

    @Test(timeout=60000L)
    public void testTruncateWithDataNodesRestartImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path parent = new Path("/test");
        Path p = new Path(parent, "testTruncateWithDataNodesRestartImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn0 = 0;
        int dn1 = 1;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        cluster.restartDataNode(dn0, true, true);
        cluster.restartDataNode(dn1, true, true);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        Assert.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        Assert.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        Assert.assertEquals((long)cluster.getBlockFile(dn0, newBlock.getBlock()).length(), (long)newBlock.getBlockSize());
        Assert.assertTrue((boolean)cluster.getBlockMetadataFile(dn0, newBlock.getBlock()).getName().endsWith(newBlock.getBlock().getGenerationStamp() + ".meta"));
        Assert.assertEquals((long)cluster.getBlockFile(dn1, newBlock.getBlock()).length(), (long)newBlock.getBlockSize());
        Assert.assertTrue((boolean)cluster.getBlockMetadataFile(dn1, newBlock.getBlock()).getName().endsWith(newBlock.getBlock().getGenerationStamp() + ".meta"));
        FileStatus fileStatus = fs.getFileStatus(p);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)newLength));
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(parent, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=60000L)
    public void testTruncateWithDataNodesShutdownImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path parent = new Path("/test");
        Path p = new Path(parent, "testTruncateWithDataNodesShutdownImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        Assert.assertFalse((boolean)isReady);
        cluster.shutdownDataNodes();
        cluster.setDataNodesDead();
        try {
            for (int i = 0; i < 300 && cluster.isDataNodeUp(); ++i) {
                Thread.sleep(100L);
            }
            Assert.assertFalse((String)"All DataNodes should be down.", (boolean)cluster.isDataNodeUp());
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p);
            Assert.assertTrue((boolean)blocks.isUnderConstruction());
        }
        finally {
            cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
            cluster.waitActive();
        }
        TestFileTruncate.checkBlockRecovery(p);
        fs.delete(parent, true);
    }

    @Test
    public void testTruncateRecovery() throws IOException {
        FSNamesystem fsn = cluster.getNamesystem();
        String client = "client";
        String clientMachine = "clientMachine";
        Path parent = new Path("/test");
        String src = "/test/testTruncateRecovery";
        Path srcPath = new Path(src);
        byte[] contents = AppendTestUtil.initBuffer(4);
        TestFileTruncate.writeContents(contents, 4, srcPath);
        INodesInPath iip = this.getINodesInPath(src, fsn, cluster);
        INodeFile file = iip.getLastINode().asFile();
        long initialGenStamp = this.getLastBlock(file, fsn).getGenerationStamp();
        BlockInfoContiguous oldBlock = this.getLastBlock(file, fsn);
        Block truncateBlock = this.prepareFileForTruncate(file, iip, client, clientMachine, fsn);
        Assert.assertThat((Object)truncateBlock.getBlockId(), (Matcher)Is.is((Matcher)CoreMatchers.equalTo((Object)oldBlock.getBlockId())));
        Assert.assertThat((Object)truncateBlock.getNumBytes(), (Matcher)Is.is((Object)oldBlock.getNumBytes()));
        Assert.assertThat((Object)truncateBlock.getGenerationStamp(), (Matcher)Is.is((Object)(oldBlock.getGenerationStamp() + 1L)));
        Assert.assertThat((Object)this.getLastBlock(file, fsn).getBlockUCState(), (Matcher)Is.is((Object)HdfsServerConstants.BlockUCState.UNDER_RECOVERY));
        long blockRecoveryId = ((BlockInfoContiguousUnderConstruction)this.getLastBlock(file, fsn)).getBlockRecoveryId();
        Assert.assertThat((Object)blockRecoveryId, (Matcher)Is.is((Object)(initialGenStamp + 1L)));
        fs.delete(parent, true);
    }

    public INodesInPath getINodesInPath(final String src, final FSNamesystem fsn, final MiniDFSCluster cluster) throws IOException {
        HopsTransactionalRequestHandler getInodeHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_INODE){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.READ, TransactionLockTypes.INodeResolveType.PATH, new String[]{src}).setNameNodeID(cluster.getNameNode().getId()).setActiveNameNodes((Collection)cluster.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il);
            }

            public Object performTask() throws IOException {
                return fsn.getFSDirectory().getINodesInPath4Write(src, true);
            }
        };
        return (INodesInPath)getInodeHandler.handle();
    }

    private BlockInfoContiguous getLastBlock(final INodeFile inode, final FSNamesystem fsn) throws IOException {
        return (BlockInfoContiguous)new HopsTransactionalRequestHandler(HDFSOperationType.TRUNCATE){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, inode.getId()).setNameNodeID(fsn.getNamenodeId()).setActiveNameNodes((Collection)fsn.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il).add(lf.getBlockLock()).add(lf.getBlockRelated(new LockFactory.BLK[]{LockFactory.BLK.RE, LockFactory.BLK.CR, LockFactory.BLK.ER, LockFactory.BLK.PE, LockFactory.BLK.UR, LockFactory.BLK.UC, LockFactory.BLK.IV}));
                locks.add(lf.getLeaseLock(TransactionLockTypes.LockType.WRITE)).add(lf.getLeasePathLock(TransactionLockTypes.LockType.WRITE));
                locks.add(lf.getAcesLock());
            }

            public Object performTask() throws IOException {
                return inode.getLastBlock();
            }
        }.handle();
    }

    private Block prepareFileForTruncate(final INodeFile inode, final INodesInPath iip, final String client, final String clientMachine, final FSNamesystem fsn) throws IOException {
        return (Block)new HopsTransactionalRequestHandler(HDFSOperationType.TRUNCATE){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, inode.getId()).setNameNodeID(fsn.getNamenodeId()).setActiveNameNodes((Collection)fsn.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il).add(lf.getBlockLock()).add(lf.getBlockRelated(new LockFactory.BLK[]{LockFactory.BLK.RE, LockFactory.BLK.CR, LockFactory.BLK.ER, LockFactory.BLK.PE, LockFactory.BLK.UR, LockFactory.BLK.UC, LockFactory.BLK.IV}));
                locks.add(lf.getLeaseLock(TransactionLockTypes.LockType.WRITE, client)).add(lf.getLeasePathLock(TransactionLockTypes.LockType.WRITE));
                locks.add(lf.getAcesLock());
            }

            public Object performTask() throws IOException {
                return fsn.prepareFileForTruncate(iip, client, clientMachine, 1L, null);
            }
        }.handle();
    }

    @Test
    public void testTruncate4Symlink() throws IOException {
        int fileLength = 12;
        Path parent = new Path("/test");
        fs.mkdirs(parent);
        byte[] contents = AppendTestUtil.initBuffer(12);
        Path file = new Path(parent, "testTruncate4Symlink");
        TestFileTruncate.writeContents(contents, 12, file);
        Path link = new Path(parent, "link");
        fs.createSymlink(file, link, false);
        int newLength = 4;
        boolean isReady = fs.truncate(link, 4L);
        Assert.assertTrue((String)"Recovery is not expected.", (boolean)isReady);
        FileStatus fileStatus = fs.getFileStatus(file);
        Assert.assertThat((Object)fileStatus.getLen(), (Matcher)Is.is((Object)4L));
        ContentSummary cs = fs.getContentSummary(parent);
        Assert.assertEquals((String)"Bad disk space usage", (long)cs.getSpaceConsumed(), (long)12L);
        TestFileTruncate.checkFullFile(file, 4, contents);
        fs.delete(parent, true);
    }

    static void writeContents(byte[] contents, int fileLength, Path p) throws IOException {
        FSDataOutputStream out = fs.create(p, true, 4, (short)3, 4L);
        out.write(contents, 0, fileLength);
        out.close();
    }

    static void checkBlockRecovery(Path p) throws IOException {
        TestFileTruncate.checkBlockRecovery(p, fs);
    }

    public static void checkBlockRecovery(Path p, DistributedFileSystem dfs) throws IOException {
        boolean success = false;
        for (int i = 0; i < 300; ++i) {
            boolean noLastBlock;
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p, dfs);
            boolean bl = noLastBlock = blocks.getLastLocatedBlock() == null;
            if (!blocks.isUnderConstruction() && (noLastBlock || blocks.isLastBlockComplete())) {
                success = true;
                break;
            }
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        Assert.assertThat((String)"inode should complete in ~30000 ms.", (Object)success, (Matcher)Is.is((Object)true));
    }

    static LocatedBlocks getLocatedBlocks(Path src) throws IOException {
        return TestFileTruncate.getLocatedBlocks(src, fs);
    }

    static LocatedBlocks getLocatedBlocks(Path src, DistributedFileSystem dfs) throws IOException {
        return dfs.getClient().getLocatedBlocks(src.toString(), 0L, Long.MAX_VALUE);
    }

    static void assertFileLength(Path file, long length) throws IOException {
        byte[] data = DFSTestUtil.readFileBuffer((FileSystem)fs, file);
        Assert.assertEquals((String)"Wrong data size in snapshot.", (long)length, (long)data.length);
    }

    static void checkFullFile(Path p, int newLength, byte[] contents) throws IOException {
        AppendTestUtil.checkFullFile((FileSystem)fs, p, newLength, contents, p.toString());
    }

    static {
        GenericTestUtils.setLogLevel((Logger)NameNode.stateChangeLog, (Level)Level.ALL);
        LOG = LogFactory.getLog(TestFileTruncate.class);
    }
}

