/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.common.CounterType;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.LeaseDataAccess;
import io.hops.metadata.hdfs.dal.LeasePathDataAccess;
import io.hops.metadata.hdfs.entity.LeasePath;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.INodeLock;
import io.hops.transaction.lock.Lock;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.Lease;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class LeaseManager {
    public static final Log LOG = LogFactory.getLog(LeaseManager.class);
    private final FSNamesystem fsnamesystem;
    private long softLimit = 60000L;
    private long hardLimit = 3600000L;
    private Daemon lmthread;
    private volatile boolean shouldRunMonitor;

    LeaseManager(FSNamesystem fsnamesystem) {
        this.fsnamesystem = fsnamesystem;
    }

    Lease getLease(String holder) throws StorageException, TransactionContextException {
        return (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolder, (Object[])new Object[]{holder, Lease.getHolderId(holder)});
    }

    SortedSet<Lease> getSortedLeases() throws IOException {
        HopsTransactionalRequestHandler getSortedLeasesHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_SORTED_LEASES){

            public void acquireLock(TransactionLocks locks) throws IOException {
            }

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return da.findAll();
            }
        };
        return new TreeSet<Lease>((Collection)getSortedLeasesHandler.handle(this.fsnamesystem));
    }

    public Lease getLeaseByPath(String src) throws StorageException, TransactionContextException {
        LeasePath leasePath = (LeasePath)EntityManager.find((FinderType)LeasePath.Finder.ByPath, (Object[])new Object[]{src});
        if (leasePath != null) {
            int holderID = leasePath.getHolderId();
            Lease lease = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolderId, (Object[])new Object[]{holderID});
            return lease;
        }
        return null;
    }

    public int countLease() throws IOException {
        return (Integer)new LightWeightRequestHandler(HDFSOperationType.COUNT_LEASE){

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return da.countAll();
            }
        }.handle((Object)this.fsnamesystem);
    }

    int countPath() throws StorageException, TransactionContextException {
        return EntityManager.count((CounterType)Lease.Counter.All, (Object[])new Object[0]);
    }

    Lease addLease(String holder, String src) throws StorageException, TransactionContextException {
        Lease lease = this.getLease(holder);
        if (lease == null) {
            lease = new Lease(holder, Lease.getHolderId(holder), Time.now());
            EntityManager.add((Object)lease);
        } else {
            this.renewLease(lease);
        }
        LeasePath lPath = new LeasePath(src, lease.getHolderID());
        lease.addFirstPath(lPath);
        EntityManager.add((Object)lPath);
        return lease;
    }

    void removeLease(Lease lease, LeasePath src) throws StorageException, TransactionContextException {
        if (lease.removePath(src)) {
            EntityManager.remove((Object)src);
        } else {
            LOG.error((Object)(src + " not found in lease.paths (=" + lease.getPaths() + ")"));
        }
        if (!lease.hasPath()) {
            EntityManager.remove((Object)lease);
        }
    }

    void removeLease(String holder, String src) throws StorageException, TransactionContextException {
        Lease lease = this.getLease(holder);
        if (lease != null) {
            this.removeLease(lease, new LeasePath(src, lease.getHolderID()));
        } else {
            LOG.warn((Object)("Removing non-existent lease! holder=" + holder + " src=" + src));
        }
    }

    void removeAllLeases() throws IOException {
        new LightWeightRequestHandler(HDFSOperationType.REMOVE_ALL_LEASES){

            public Object performTask() throws StorageException, IOException {
                LeaseDataAccess lda = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                LeasePathDataAccess lpda = (LeasePathDataAccess)HdfsStorageFactory.getDataAccess(LeasePathDataAccess.class);
                lda.removeAll();
                lpda.removeAll();
                return null;
            }
        }.handle((Object)this.fsnamesystem);
    }

    Lease reassignLease(Lease lease, String src, String newHolder) throws StorageException, TransactionContextException {
        assert (newHolder != null) : "new lease holder is null";
        if (lease != null) {
            LeasePath lp = new LeasePath(src, lease.getHolderID());
            if (!lease.removePath(lp)) {
                LOG.error((Object)(src + " not found in lease.paths (=" + lease.getPaths() + ")"));
            }
            EntityManager.remove((Object)lp);
            if (!lease.hasPath() && !lease.getHolder().equals(newHolder)) {
                EntityManager.remove((Object)lease);
            }
        }
        Lease newLease = this.getLease(newHolder);
        LeasePath lPath = null;
        if (newLease == null) {
            newLease = new Lease(newHolder, Lease.getHolderId(newHolder), Time.now());
            EntityManager.add((Object)newLease);
            lPath = new LeasePath(src, newLease.getHolderID());
            newLease.addFirstPath(lPath);
        } else {
            this.renewLease(newLease);
            lPath = new LeasePath(src, newLease.getHolderID());
            newLease.addPath(lPath);
        }
        EntityManager.update((Object)lPath);
        return newLease;
    }

    void renewLease(String holder) throws StorageException, TransactionContextException {
        this.renewLease(this.getLease(holder));
    }

    void renewLease(Lease lease) throws StorageException, TransactionContextException {
        if (lease != null) {
            lease.setLastUpdate(Time.now());
            EntityManager.update((Object)lease);
        }
    }

    void changeLease(String src, String dst) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".changelease:  src=" + src + ", dest=" + dst));
        }
        int len = src.length();
        Collection<LeasePath> paths = this.findLeasePathsWithPrefix(src);
        ArrayList<LeasePath> newLPs = new ArrayList<LeasePath>(paths.size());
        ArrayList<LeasePath> deletedLPs = new ArrayList<LeasePath>(paths.size());
        for (LeasePath oldPath : paths) {
            int holderId = oldPath.getHolderId();
            LeasePath newpath = new LeasePath(dst + oldPath.getPath().substring(len), holderId, oldPath.getLastBlockId(), oldPath.getPenultimateBlockId());
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("changeLease: replacing " + oldPath + " with " + newpath));
            }
            newLPs.add(newpath);
            deletedLPs.add(oldPath);
        }
        for (LeasePath newPath : newLPs) {
            EntityManager.add((Object)newPath);
        }
        for (LeasePath deletedLP : deletedLPs) {
            EntityManager.remove((Object)deletedLP);
        }
    }

    void removeLeaseWithPrefixPath(String prefix) throws StorageException, TransactionContextException {
        for (Map.Entry<LeasePath, Lease> entry : this.findLeaseWithPrefixPath(prefix).entrySet()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".removeLeaseWithPrefixPath: entry=" + entry));
            }
            this.removeLease(entry.getValue(), entry.getKey());
        }
    }

    private Map<LeasePath, Lease> findLeaseWithPrefixPath(String prefix) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        Collection leasePathSet = EntityManager.findList((FinderType)LeasePath.Finder.ByPrefix, (Object[])new Object[]{prefix});
        HashMap<LeasePath, Lease> entries = new HashMap<LeasePath, Lease>();
        int srclen = prefix.length();
        if (prefix.charAt(srclen - 1) == '/') {
            --srclen;
        }
        for (LeasePath lPath : leasePathSet) {
            if (!lPath.getPath().startsWith(prefix)) {
                LOG.warn((Object)("LeasePath fetched by prefix does not start with the prefix: \nLeasePath: " + lPath + "\t Prefix: " + prefix));
                return entries;
            }
            if (lPath.getPath().length() != srclen && lPath.getPath().charAt(srclen) != '/') continue;
            Lease lease = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolderId, (Object[])new Object[]{lPath.getHolderId()});
            entries.put(lPath, lease);
        }
        return entries;
    }

    private Collection<LeasePath> findLeasePathsWithPrefix(String prefix) throws StorageException, TransactionContextException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(LeaseManager.class.getSimpleName() + ".findLease: prefix=" + prefix));
        }
        Collection leasePathSet = EntityManager.findList((FinderType)LeasePath.Finder.ByPrefix, (Object[])new Object[]{prefix});
        return leasePathSet;
    }

    public void setLeasePeriod(long softLimit, long hardLimit) {
        this.softLimit = softLimit;
        this.hardLimit = hardLimit;
    }

    private boolean checkLeases(String holder) throws StorageException, TransactionContextException {
        boolean needSync = false;
        Lease oldest = (Lease)EntityManager.find((FinderType)Lease.Finder.ByHolder, (Object[])new Object[]{holder, Lease.getHolderId(holder)});
        if (oldest == null) {
            return needSync;
        }
        if (!this.expiredHardLimit(oldest)) {
            return needSync;
        }
        LOG.info((Object)("Lease " + oldest + " has expired hard limit"));
        ArrayList<LeasePath> removing = new ArrayList<LeasePath>();
        Collection<LeasePath> paths = oldest.getPaths();
        assert (paths != null) : "The lease " + oldest.toString() + " has no path.";
        LeasePath[] leasePaths = new LeasePath[paths.size()];
        paths.toArray(leasePaths);
        for (LeasePath lPath : leasePaths) {
            try {
                boolean leaseReleased = false;
                leaseReleased = this.fsnamesystem.internalReleaseLease(oldest, lPath.getPath(), "HDFS_NameNode");
                if (leaseReleased) {
                    LOG.info((Object)("Lease recovery for file " + lPath + " is complete. File closed."));
                    removing.add(lPath);
                } else {
                    LOG.info((Object)("Started block recovery for file " + lPath + " lease " + oldest));
                }
                if (needSync || leaseReleased) continue;
                needSync = true;
            }
            catch (IOException e) {
                LOG.error((Object)("Cannot release the path " + lPath + " in the lease " + oldest), (Throwable)e);
                removing.add(lPath);
            }
        }
        for (LeasePath lPath : removing) {
            if (!oldest.getPaths().contains(lPath)) continue;
            this.removeLease(oldest, lPath);
        }
        return needSync;
    }

    void startMonitor() {
        Preconditions.checkState((this.lmthread == null ? 1 : 0) != 0, (Object)"Lease Monitor already running");
        this.shouldRunMonitor = true;
        this.lmthread = new Daemon((Runnable)new Monitor());
        this.lmthread.start();
    }

    void stopMonitor() {
        if (this.lmthread != null) {
            this.shouldRunMonitor = false;
            try {
                this.lmthread.interrupt();
                this.lmthread.join(3000L);
            }
            catch (InterruptedException ie) {
                LOG.warn((Object)"Encountered exception ", (Throwable)ie);
            }
            this.lmthread = null;
        }
    }

    @VisibleForTesting
    void triggerMonitorCheckNow() {
        Preconditions.checkState((this.lmthread != null ? 1 : 0) != 0, (Object)"Lease monitor is not running");
        this.lmthread.interrupt();
    }

    private boolean expiredHardLimit(Lease lease) {
        return Time.now() - lease.getLastUpdate() > this.hardLimit;
    }

    public boolean expiredSoftLimit(Lease lease) {
        return Time.now() - lease.getLastUpdate() > this.softLimit;
    }

    class Monitor
    implements Runnable {
        final String name = this.getClass().getSimpleName();
        LightWeightRequestHandler findExpiredLeaseHandler = new LightWeightRequestHandler(HDFSOperationType.PREPARE_LEASE_MANAGER_MONITOR){

            public Object performTask() throws StorageException, IOException {
                long expiredTime = Time.now() - LeaseManager.this.hardLimit;
                LeaseDataAccess da = (LeaseDataAccess)HdfsStorageFactory.getDataAccess(LeaseDataAccess.class);
                return new TreeSet(da.findByTimeLimit(expiredTime));
            }
        };
        HopsTransactionalRequestHandler expiredLeaseHandler = new HopsTransactionalRequestHandler(HDFSOperationType.LEASE_MANAGER_MONITOR){
            private Set<String> leasePaths;
            {
                this.leasePaths = null;
            }

            @Override
            public void setUp() throws StorageException {
                String holder = (String)this.getParams()[0];
                this.leasePaths = INodeUtil.findPathsByLeaseHolder(holder);
                if (this.leasePaths != null) {
                    LOG.debug((Object)("Total Paths " + this.leasePaths.size() + " Paths: " + Arrays.toString(this.leasePaths.toArray())));
                }
            }

            public void acquireLock(TransactionLocks locks) throws IOException {
                String holder = (String)this.getParams()[0];
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, this.leasePaths.toArray(new String[this.leasePaths.size()])).setNameNodeID(LeaseManager.this.fsnamesystem.getNameNode().getId()).setActiveNameNodes(LeaseManager.this.fsnamesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il).add(lf.getNameNodeLeaseLock(TransactionLockTypes.LockType.WRITE)).add(lf.getLeaseLock(TransactionLockTypes.LockType.WRITE, holder)).add(lf.getLeasePathLock(TransactionLockTypes.LockType.WRITE, this.leasePaths.size())).add(lf.getBlockLock()).add(lf.getBlockRelated(LockFactory.BLK.RE, LockFactory.BLK.CR, LockFactory.BLK.ER, LockFactory.BLK.UC, LockFactory.BLK.UR));
            }

            public Object performTask() throws StorageException, IOException {
                String holder = (String)this.getParams()[0];
                if (holder != null) {
                    LeaseManager.this.checkLeases(holder);
                }
                return null;
            }
        };

        Monitor() {
        }

        @Override
        public void run() {
            while (LeaseManager.this.shouldRunMonitor && LeaseManager.this.fsnamesystem.isRunning()) {
                try {
                    if (LeaseManager.this.fsnamesystem.isLeader()) {
                        try {
                            SortedSet sortedLeases;
                            if (!LeaseManager.this.fsnamesystem.isInSafeMode() && (sortedLeases = (SortedSet)this.findExpiredLeaseHandler.handle((Object)LeaseManager.this.fsnamesystem)) != null) {
                                for (Lease expiredLease : sortedLeases) {
                                    this.expiredLeaseHandler.setParams(new Object[]{expiredLease.getHolder()}).handle((Object)LeaseManager.this.fsnamesystem);
                                }
                            }
                        }
                        catch (IOException ex) {
                            LOG.error((Object)ex);
                        }
                    }
                    Thread.sleep(2000L);
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(this.name + " is interrupted"), (Throwable)ie);
                }
            }
        }
    }
}

