/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import io.hops.common.IDsGeneratorFactory;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.CachePoolDataAccess;
import io.hops.metadata.hdfs.entity.CachedBlock;
import io.hops.metadata.hdfs.entity.INodeIdentifier;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.CacheFlag;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.CacheDirective;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveEntry;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveStats;
import org.apache.hadoop.hdfs.protocol.CachePoolEntry;
import org.apache.hadoop.hdfs.protocol.CachePoolInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.CacheReplicationMonitor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeManager;
import org.apache.hadoop.hdfs.server.namenode.CachePool;
import org.apache.hadoop.hdfs.server.namenode.CachedBlock;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FSPermissionChecker;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.util.Time;

@InterfaceAudience.LimitedPrivate(value={"HDFS"})
public final class CacheManager {
    public static final Log LOG = LogFactory.getLog(CacheManager.class);
    private static final float MIN_CACHED_BLOCKS_PERCENT = 0.001f;
    private final FSNamesystem namesystem;
    private final BlockManager blockManager;
    private final int maxListCachePoolsResponses;
    private final int maxListCacheDirectivesNumResponses;
    private final long scanIntervalMs;
    private final ReentrantLock crmLock = new ReentrantLock();
    private CacheReplicationMonitor monitor;

    CacheManager(FSNamesystem namesystem, Configuration conf, BlockManager blockManager) {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.maxListCachePoolsResponses = conf.getInt("dfs.namenode.list.cache.pools.num.responses", 100);
        this.maxListCacheDirectivesNumResponses = conf.getInt("dfs.namenode.list.cache.directives.num.responses", 100);
        this.scanIntervalMs = conf.getLong("dfs.namenode.path.based.cache.refresh.interval.ms", 30000L);
        float cachedBlocksPercent = conf.getFloat("dfs.namenode.path.based.cache.block.map.allocation.percent", 0.25f);
        if (cachedBlocksPercent < 0.001f) {
            LOG.info((Object)"Using minimum value 0.001 for dfs.namenode.path.based.cache.block.map.allocation.percent");
            cachedBlocksPercent = 0.001f;
        }
    }

    public void startMonitorThread() {
        this.crmLock.lock();
        try {
            if (this.monitor == null) {
                this.monitor = new CacheReplicationMonitor(this.namesystem, this, this.scanIntervalMs, this.crmLock);
                this.monitor.start();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    public void stopMonitorThread() {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                CacheReplicationMonitor prevMonitor = this.monitor;
                this.monitor = null;
                IOUtils.closeQuietly((Closeable)prevMonitor);
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    public Collection<CachePool> getCachePools() throws TransactionContextException, StorageException {
        return Collections.unmodifiableCollection(EntityManager.findList((FinderType)CachePool.Finder.All, (Object[])new Object[0]));
    }

    public Collection<CacheDirective> getCacheDirectives() throws TransactionContextException, StorageException {
        return Collections.unmodifiableCollection(EntityManager.findList((FinderType)CacheDirective.Finder.All, (Object[])new Object[0]));
    }

    @VisibleForTesting
    public Set<CachedBlock> getCachedBlocks(final DatanodeManager datanodeManager) throws TransactionContextException, StorageException, IOException {
        Collection all = (Collection)new HopsTransactionalRequestHandler(HDFSOperationType.TEST){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getAllCachedBlockLocks());
            }

            public Object performTask() throws IOException {
                return CachedBlock.getAll(datanodeManager);
            }
        }.handle();
        HashSet<CachedBlock> result = new HashSet<CachedBlock>();
        for (CachedBlock block : all) {
            if (!block.isCached()) continue;
            result.add(block);
        }
        return result;
    }

    public CachedBlock getCachedBlock(CachedBlock block) throws TransactionContextException, StorageException {
        Collection dalCachedBlocks = EntityManager.findList((FinderType)CachedBlock.Finder.ByBlockIdAndInodeId, (Object[])new Object[]{block.getBlockId(), block.getInodeId()});
        if (dalCachedBlocks == null || dalCachedBlocks.isEmpty()) {
            return null;
        }
        CachedBlock cachedBlock = null;
        for (io.hops.metadata.hdfs.entity.CachedBlock dalBlock : dalCachedBlocks) {
            if (dalBlock.getStatus().equals("")) continue;
            if (cachedBlock == null) {
                cachedBlock = new CachedBlock(dalBlock.getBlockId(), dalBlock.getInodeId(), dalBlock.getReplicationAndMark());
            }
            if (CachedBlock.Type.valueOf(dalBlock.getStatus()).equals((Object)CachedBlock.Type.INIT)) continue;
            cachedBlock.addDatanode(this.blockManager.getDatanodeManager().getDatanodeByUuid(dalBlock.getDatanodeId()), dalBlock.getStatus());
        }
        return cachedBlock;
    }

    public long getNextDirectiveId() throws IOException {
        long nextDirectiveId = IDsGeneratorFactory.getInstance().getUniqueCacheDirectiveID();
        if (nextDirectiveId >= 0x7FFFFFFFFFFFFFFEL) {
            throw new IOException("No more available IDs.");
        }
        return nextDirectiveId;
    }

    private static void checkWritePermission(FSPermissionChecker pc, CachePool pool) throws AccessControlException {
        if (pc != null) {
            pc.checkPermission(pool, FsAction.WRITE);
        }
    }

    public static String validatePoolName(CacheDirectiveInfo directive) throws InvalidRequestException {
        String pool = directive.getPool();
        if (pool == null) {
            throw new InvalidRequestException("No pool specified.");
        }
        if (pool.isEmpty()) {
            throw new InvalidRequestException("Invalid empty pool name.");
        }
        return pool;
    }

    public static String validatePath(CacheDirectiveInfo directive) throws InvalidRequestException {
        if (directive.getPath() == null) {
            throw new InvalidRequestException("No path specified.");
        }
        String path = directive.getPath().toUri().getPath();
        if (!DFSUtil.isValidName(path)) {
            throw new InvalidRequestException("Invalid path '" + path + "'.");
        }
        return path;
    }

    private static short validateReplication(CacheDirectiveInfo directive, short defaultValue) throws InvalidRequestException {
        short repl;
        short s = repl = directive.getReplication() != null ? directive.getReplication() : defaultValue;
        if (repl <= 0) {
            throw new InvalidRequestException("Invalid replication factor " + repl + " <= 0");
        }
        return repl;
    }

    private static long validateExpiryTime(CacheDirectiveInfo info, long maxRelativeExpiryTime) throws InvalidRequestException {
        long absExpiryTime;
        long relExpiryTime;
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Validating directive " + info + " pool maxRelativeExpiryTime " + maxRelativeExpiryTime));
        }
        long now = new Date().getTime();
        long maxAbsoluteExpiryTime = now + maxRelativeExpiryTime;
        if (info == null || info.getExpiration() == null) {
            return maxAbsoluteExpiryTime;
        }
        CacheDirectiveInfo.Expiration expiry = info.getExpiration();
        if (expiry.getMillis() < 0L) {
            throw new InvalidRequestException("Cannot set a negative expiration: " + expiry.getMillis());
        }
        if (expiry.isRelative()) {
            relExpiryTime = expiry.getMillis();
            absExpiryTime = now + relExpiryTime;
        } else {
            absExpiryTime = expiry.getMillis();
            relExpiryTime = absExpiryTime - now;
        }
        if (relExpiryTime > 0x1FFFFFFFFFFFFFFFL) {
            throw new InvalidRequestException("Expiration " + expiry.toString() + " is too far in the future!");
        }
        if (relExpiryTime > maxRelativeExpiryTime) {
            throw new InvalidRequestException("Expiration " + expiry.toString() + " exceeds the max relative expiration time of " + maxRelativeExpiryTime + " ms.");
        }
        return absExpiryTime;
    }

    private void checkLimit(CachePool pool, String path, short replication) throws InvalidRequestException, StorageException, TransactionContextException {
        CacheDirectiveStats stats = this.computeNeeded(path, replication);
        if (pool.getLimit() == Long.MAX_VALUE) {
            return;
        }
        if (pool.getBytesNeeded() + stats.getBytesNeeded() * (long)replication > pool.getLimit()) {
            throw new InvalidRequestException("Caching path " + path + " of size " + stats.getBytesNeeded() / (long)replication + " bytes at replication " + replication + " would exceed pool " + pool.getPoolName() + "'s remaining capacity of " + (pool.getLimit() - pool.getBytesNeeded()) + " bytes.");
        }
    }

    private CacheDirectiveStats computeNeeded(String path, short replication) throws StorageException, TransactionContextException {
        INode node;
        FSDirectory fsDir = this.namesystem.getFSDirectory();
        long requestedBytes = 0L;
        long requestedFiles = 0L;
        CacheDirectiveStats.Builder builder = new CacheDirectiveStats.Builder();
        try {
            node = fsDir.getINode(path);
        }
        catch (UnresolvedLinkException e) {
            return builder.build();
        }
        if (node == null) {
            return builder.build();
        }
        if (node.isFile()) {
            requestedFiles = 1L;
            INodeFile file = node.asFile();
            requestedBytes = file.computeFileSize();
        } else if (node.isDirectory()) {
            INodeDirectory dir = node.asDirectory();
            List<INode> children = dir.getChildrenList();
            requestedFiles = children.size();
            for (INode child : children) {
                if (!child.isFile()) continue;
                requestedBytes += child.asFile().computeFileSize();
            }
        }
        return new CacheDirectiveStats.Builder().setBytesNeeded(requestedBytes).setFilesCached(requestedFiles).build();
    }

    private CacheDirective getById(long id) throws InvalidRequestException, TransactionContextException, StorageException {
        if (id <= 0L) {
            throw new InvalidRequestException("Invalid negative ID.");
        }
        CacheDirective directive = (CacheDirective)EntityManager.find((FinderType)CacheDirective.Finder.ById, (Object[])new Object[]{id});
        if (directive == null) {
            throw new InvalidRequestException("No directive with ID " + id + " found.");
        }
        return directive;
    }

    private CachePool getCachePool(String poolName) throws InvalidRequestException, TransactionContextException, StorageException {
        CachePool pool = (CachePool)EntityManager.find((FinderType)CachePool.Finder.ByName, (Object[])new Object[]{poolName});
        if (pool == null) {
            throw new InvalidRequestException("Unknown pool " + poolName);
        }
        return pool;
    }

    private void addInternal(CacheDirective directive, CachePool pool) throws StorageException, TransactionContextException, IOException {
        directive.setPoolName(pool.getPoolName());
        CacheDirectiveStats stats = this.computeNeeded(directive.getPath(), directive.getReplication());
        directive.addBytesNeeded(stats.getBytesNeeded());
        directive.addFilesNeeded(directive.getFilesNeeded());
        EntityManager.update((Object)directive);
        this.setNeedsRescan();
    }

    public CacheDirectiveInfo addDirective(CacheDirectiveInfo info, FSPermissionChecker pc, EnumSet<CacheFlag> flags, long id) throws IOException {
        CacheDirective directive;
        try {
            CachePool pool = this.getCachePool(CacheManager.validatePoolName(info));
            CacheManager.checkWritePermission(pc, pool);
            String path = CacheManager.validatePath(info);
            short replication = CacheManager.validateReplication(info, (short)1);
            long expiryTime = CacheManager.validateExpiryTime(info, pool.getMaxRelativeExpiryMs());
            if (!flags.contains((Object)CacheFlag.FORCE)) {
                this.checkLimit(pool, path, replication);
            }
            directive = new CacheDirective(id, path, replication, expiryTime);
            this.addInternal(directive, pool);
        }
        catch (IOException e) {
            LOG.warn((Object)("addDirective of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("addDirective of " + info + " successful."));
        return directive.toInfo();
    }

    private static CacheDirectiveInfo createFromInfoAndDefaults(CacheDirectiveInfo info, CacheDirective defaults) {
        CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder(defaults.toInfo());
        if (info.getPath() != null) {
            builder.setPath(info.getPath());
        }
        if (info.getReplication() != null) {
            builder.setReplication(info.getReplication());
        }
        if (info.getPool() != null) {
            builder.setPool(info.getPool());
        }
        if (info.getExpiration() != null) {
            builder.setExpiration(info.getExpiration());
        }
        return builder.build();
    }

    public void modifyDirective(CacheDirectiveInfo info, FSPermissionChecker pc, EnumSet<CacheFlag> flags) throws IOException {
        String idString = info.getId() == null ? "(null)" : info.getId().toString();
        try {
            Long id = info.getId();
            if (id == null) {
                throw new InvalidRequestException("Must supply an ID.");
            }
            CacheDirective prevEntry = this.getById(id);
            CacheManager.checkWritePermission(pc, prevEntry.getPool());
            CacheDirectiveInfo infoWithDefaults = CacheManager.createFromInfoAndDefaults(info, prevEntry);
            CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder(infoWithDefaults);
            CacheManager.validatePath(infoWithDefaults);
            CacheManager.validateReplication(infoWithDefaults, (short)-1);
            CachePool srcPool = prevEntry.getPool();
            CachePool destPool = this.getCachePool(CacheManager.validatePoolName(infoWithDefaults));
            if (!srcPool.getPoolName().equals(destPool.getPoolName())) {
                CacheManager.checkWritePermission(pc, destPool);
                if (!flags.contains((Object)CacheFlag.FORCE)) {
                    this.checkLimit(destPool, infoWithDefaults.getPath().toUri().getPath(), infoWithDefaults.getReplication());
                }
            }
            CacheManager.validateExpiryTime(infoWithDefaults, destPool.getMaxRelativeExpiryMs());
            this.setNeedsRescan();
            this.removeInternal(prevEntry);
            this.addInternal(new CacheDirective(builder.build()), destPool);
        }
        catch (IOException e) {
            LOG.warn((Object)("modifyDirective of " + idString + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("modifyDirective of " + idString + " successfully applied " + info + "."));
    }

    private void removeInternal(CacheDirective directive) throws InvalidRequestException, StorageException, TransactionContextException, IOException {
        directive.addBytesNeeded(-directive.getBytesNeeded());
        directive.addFilesNeeded(-directive.getFilesNeeded());
        EntityManager.remove((Object)directive);
        this.setNeedsRescan();
    }

    public void removeDirective(long id, FSPermissionChecker pc) throws IOException {
        try {
            CacheDirective directive = this.getById(id);
            CacheManager.checkWritePermission(pc, directive.getPool());
            this.removeInternal(directive);
        }
        catch (IOException e) {
            LOG.warn((Object)("removeDirective of " + id + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("removeDirective of " + id + " successful."));
    }

    public BatchedRemoteIterator.BatchedListEntries<CacheDirectiveEntry> listCacheDirectives(long prevId1, final CacheDirectiveInfo filter, final FSPermissionChecker pc) throws IOException {
        int NUM_PRE_ALLOCATED_ENTRIES = 16;
        final String[] filterPath = new String[1];
        if (filter.getPath() != null) {
            filterPath[0] = CacheManager.validatePath(filter);
        }
        if (filter.getReplication() != null) {
            throw new InvalidRequestException("Filtering by replication is unsupported.");
        }
        final ArrayList replies = new ArrayList(16);
        final Long id = filter.getId();
        if (id != null) {
            prevId1 = id - 1L;
        }
        final long prevId = prevId1;
        HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(HDFSOperationType.LIST_CACHE_DIRECTIVE){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getCacheDirectiveLock(prevId + 1L, filterPath[0], filter.getPool(), CacheManager.this.maxListCacheDirectivesNumResponses + 1)).add(lf.getCachePoolLock(TransactionLockTypes.LockType.READ));
            }

            public Object performTask() throws IOException {
                if (id != null && EntityManager.find((FinderType)CacheDirective.Finder.ById, (Object[])new Object[]{id}) == null) {
                    throw new InvalidRequestException("Did not find requested id " + id);
                }
                int numReplies = 0;
                Map tailMap = CacheManager.this.getDirectivesByIdPathAndPool(prevId + 1L, filterPath[0], filter.getPool(), CacheManager.this.maxListCacheDirectivesNumResponses + 1);
                for (Map.Entry cur : tailMap.entrySet()) {
                    if (numReplies >= CacheManager.this.maxListCacheDirectivesNumResponses) {
                        return new BatchedRemoteIterator.BatchedListEntries((List)replies, true);
                    }
                    CacheDirective curDirective = (CacheDirective)cur.getValue();
                    CacheDirectiveInfo info = ((CacheDirective)cur.getValue()).toInfo();
                    if (id != null && !info.getId().equals(id)) break;
                    if (filter.getPool() != null && !info.getPool().equals(filter.getPool()) || filterPath[0] != null && !info.getPath().toUri().getPath().equals(filterPath[0])) continue;
                    boolean hasPermission = true;
                    if (pc != null) {
                        try {
                            pc.checkPermission(curDirective.getPool(), FsAction.READ);
                        }
                        catch (AccessControlException e) {
                            hasPermission = false;
                        }
                    }
                    if (!hasPermission) continue;
                    replies.add(new CacheDirectiveEntry(info, ((CacheDirective)cur.getValue()).toStats()));
                    ++numReplies;
                }
                return new BatchedRemoteIterator.BatchedListEntries((List)replies, false);
            }
        };
        return (BatchedRemoteIterator.BatchedListEntries)handler.handle();
    }

    private Map<Long, CacheDirective> getDirectivesByIdPathAndPool(long id, String path, String pool, int maxNumberResults) throws IOException {
        List directives = (List)EntityManager.findList((FinderType)CacheDirective.Finder.ByIdPoolAndPath, (Object[])new Object[]{id, pool, path, maxNumberResults});
        TreeMap<Long, CacheDirective> result = new TreeMap<Long, CacheDirective>();
        for (CacheDirective directive : directives) {
            result.put(directive.getId(), directive);
        }
        return result;
    }

    public CachePoolInfo addCachePool(CachePoolInfo info) throws IOException {
        CachePool pool;
        try {
            CachePoolInfo.validate(info);
            String poolName = info.getPoolName();
            pool = (CachePool)EntityManager.find((FinderType)CachePool.Finder.ByName, (Object[])new Object[]{poolName});
            if (pool != null) {
                throw new InvalidRequestException("Cache pool " + poolName + " already exists.");
            }
            pool = CachePool.createFromInfoAndDefaults(info);
            EntityManager.update((Object)pool);
        }
        catch (IOException e) {
            LOG.info((Object)("addCachePool of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("addCachePool of " + info + " successful."));
        return pool.getInfo(true);
    }

    public void modifyCachePool(CachePoolInfo info) throws IOException {
        StringBuilder bld = new StringBuilder();
        try {
            CachePoolInfo.validate(info);
            String poolName = info.getPoolName();
            CachePool pool = (CachePool)EntityManager.find((FinderType)CachePool.Finder.ByName, (Object[])new Object[]{poolName});
            if (pool == null) {
                throw new InvalidRequestException("Cache pool " + poolName + " does not exist.");
            }
            String prefix = "";
            if (info.getOwnerName() != null) {
                pool.setOwnerName(info.getOwnerName());
                bld.append(prefix).append("set owner to ").append(info.getOwnerName());
                prefix = "; ";
            }
            if (info.getGroupName() != null) {
                pool.setGroupName(info.getGroupName());
                bld.append(prefix).append("set group to ").append(info.getGroupName());
                prefix = "; ";
            }
            if (info.getMode() != null) {
                pool.setMode(info.getMode());
                bld.append(prefix).append("set mode to " + info.getMode());
                prefix = "; ";
            }
            if (info.getLimit() != null) {
                pool.setLimit(info.getLimit());
                bld.append(prefix).append("set limit to " + info.getLimit());
                prefix = "; ";
                this.setNeedsRescan();
            }
            if (info.getMaxRelativeExpiryMs() != null) {
                Long maxRelativeExpiry = info.getMaxRelativeExpiryMs();
                pool.setMaxRelativeExpiryMs(maxRelativeExpiry);
                bld.append(prefix).append("set maxRelativeExpiry to " + maxRelativeExpiry);
                prefix = "; ";
            }
            if (prefix.isEmpty()) {
                bld.append("no changes.");
            }
            EntityManager.update((Object)pool);
        }
        catch (IOException e) {
            LOG.info((Object)("modifyCachePool of " + info + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("modifyCachePool of " + info.getPoolName() + " successful; " + bld.toString()));
    }

    public void removeCachePool(String poolName) throws IOException {
        try {
            CachePoolInfo.validateName(poolName);
            CachePool pool = (CachePool)EntityManager.find((FinderType)CachePool.Finder.ByName, (Object[])new Object[]{poolName});
            if (pool == null) {
                throw new InvalidRequestException("Cannot remove non-existent cache pool " + poolName);
            }
            EntityManager.remove((Object)pool);
            Iterator<CacheDirective> iter = pool.getDirectiveList().iterator();
            while (iter.hasNext()) {
                CacheDirective directive = iter.next();
                EntityManager.remove((Object)directive);
                iter.remove();
            }
            this.setNeedsRescan();
        }
        catch (IOException e) {
            LOG.info((Object)("removeCachePool of " + poolName + " failed: "), (Throwable)e);
            throw e;
        }
        LOG.info((Object)("removeCachePool of " + poolName + " successful."));
    }

    public BatchedRemoteIterator.BatchedListEntries<CachePoolEntry> listCachePools(FSPermissionChecker pc, String prevKey) throws IOException {
        int NUM_PRE_ALLOCATED_ENTRIES = 16;
        ArrayList<CachePoolEntry> results = new ArrayList<CachePoolEntry>(16);
        Map<String, CachePool> tailMap = this.getCachePoolMap(prevKey);
        int numListed = 0;
        for (Map.Entry<String, CachePool> cur : tailMap.entrySet()) {
            if (numListed++ >= this.maxListCachePoolsResponses) {
                return new BatchedRemoteIterator.BatchedListEntries(results, true);
            }
            results.add(cur.getValue().getEntry(pc));
        }
        return new BatchedRemoteIterator.BatchedListEntries(results, false);
    }

    private Map<String, CachePool> getCachePoolMap(final String poolName) throws IOException {
        Collection cachePools = (Collection)new LightWeightRequestHandler(HDFSOperationType.LIST_CACHE_POOL){

            public Object performTask() throws IOException {
                CachePoolDataAccess da = (CachePoolDataAccess)HdfsStorageFactory.getDataAccess(CachePoolDataAccess.class);
                return da.findAboveName(poolName);
            }
        }.handle();
        TreeMap<String, CachePool> result = new TreeMap<String, CachePool>();
        for (CachePool dalPool : cachePools) {
            result.put(dalPool.getPoolName(), new CachePool(dalPool.getPoolName(), dalPool.getOwnerName(), dalPool.getGroupName(), new FsPermission(dalPool.getMode()), dalPool.getLimit(), dalPool.getMaxRelativeExpiryMs(), dalPool.getBytesNeeded(), dalPool.getBytesCached(), dalPool.getFilesNeeded(), dalPool.getFilesCached()));
        }
        return result;
    }

    public void setCachedLocations(LocatedBlock block, long inodeId) throws TransactionContextException, StorageException {
        CachedBlock cachedBlock = new CachedBlock(block.getBlock().getBlockId(), inodeId, 0, false);
        if ((cachedBlock = this.getCachedBlock(cachedBlock)) == null) {
            return;
        }
        List<DatanodeDescriptor> datanodes = cachedBlock.getDatanodes(CachedBlock.Type.CACHED);
        for (DatanodeDescriptor datanode : datanodes) {
            block.addCachedLoc(datanode);
        }
    }

    public final void processCacheReport(final DatanodeID datanodeID, final List<Long> blockIds, final long cacheCapacity, final long cacheUsed) throws IOException {
        long startTime = Time.monotonicNow();
        new HopsTransactionalRequestHandler(HDFSOperationType.PROCESS_CACHED_BLOCKS_REPORT){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getCachedBlockReportingLocks(blockIds, datanodeID));
            }

            public Object performTask() throws IOException {
                DatanodeDescriptor datanode = CacheManager.this.blockManager.getDatanodeManager().getDatanode(datanodeID);
                if (datanode == null || !datanode.isAlive) {
                    throw new IOException("processCacheReport from dead or unregistered datanode: " + datanode);
                }
                datanode.setCacheCapacity(cacheCapacity);
                datanode.setCacheUsed(cacheUsed);
                CacheManager.this.processCacheReportImpl(datanode, blockIds);
                return null;
            }
        }.handle();
        long endTime = Time.monotonicNow();
        NameNodeMetrics metrics = NameNode.getNameNodeMetrics();
        if (metrics != null) {
            metrics.addCacheBlockReport((int)(endTime - startTime));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Processed cache report from " + datanodeID + ", blocks: " + blockIds.size() + ", processing time: " + (endTime - startTime) + " msecs"));
        }
    }

    private void processCacheReportImpl(DatanodeDescriptor datanode, List<Long> blockIds) throws TransactionContextException, StorageException {
        Collection<CachedBlock> oldCached = datanode.getCached(this.blockManager.getDatanodeManager());
        HashSet<CachedBlock> pendingUncache = new HashSet<CachedBlock>(datanode.getPendingUncached(this.blockManager.getDatanodeManager()));
        HashSet<CachedBlock> cached = new HashSet<CachedBlock>();
        for (long blockId : blockIds) {
            CachedBlock cachedBlock;
            CachedBlock prevCachedBlock;
            INodeIdentifier inode = INodeUtil.resolveINodeFromBlockID(blockId);
            long inodeId = -1L;
            if (inode != null) {
                inodeId = inode.getInodeId();
            }
            if ((prevCachedBlock = this.getCachedBlock(cachedBlock = new CachedBlock(blockId, inodeId, 0, false, datanode, CachedBlock.Type.CACHED))) != null) {
                cachedBlock = prevCachedBlock;
                cachedBlock.switchPendingCachedToCached(datanode);
            }
            cached.add(cachedBlock);
            if (pendingUncache.contains(cachedBlock)) continue;
            cachedBlock.save();
        }
        for (CachedBlock block : oldCached) {
            if (cached.contains(block)) continue;
            block.remove(datanode);
        }
        for (CachedBlock block : pendingUncache) {
            if (cached.contains(block)) continue;
            block.remove(datanode);
        }
    }

    public void waitForRescanIfNeeded() throws StorageException, TransactionContextException, IOException {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                this.monitor.waitForRescanIfNeeded();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    private void setNeedsRescan() throws StorageException, TransactionContextException, IOException {
        this.crmLock.lock();
        try {
            if (this.monitor != null) {
                this.monitor.setNeedsRescan();
            }
        }
        finally {
            this.crmLock.unlock();
        }
    }

    @VisibleForTesting
    public Thread getCacheReplicationMonitor() {
        this.crmLock.lock();
        try {
            CacheReplicationMonitor cacheReplicationMonitor = this.monitor;
            return cacheReplicationMonitor;
        }
        finally {
            this.crmLock.unlock();
        }
    }
}

