/*
 * Decompiled with CFR 0.152.
 */
package io.hops.erasure_coding;

import io.hops.erasure_coding.BlockRepairManager;
import io.hops.erasure_coding.EncodingManager;
import io.hops.erasure_coding.Report;
import io.hops.exception.StorageException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.EncodingStatusDataAccess;
import io.hops.metadata.hdfs.entity.EncodingStatus;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.EncodingStatusOperationType;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.INodeLock;
import io.hops.transaction.lock.Lock;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.StringUtils;

public class ErasureCodingManager
extends Configured {
    static final Log LOG = LogFactory.getLog(ErasureCodingManager.class);
    private final FSNamesystem namesystem;
    private final Daemon erasureCodingMonitorThread = new Daemon((Runnable)new ErasureCodingMonitor());
    private EncodingManager encodingManager;
    private BlockRepairManager blockRepairManager;
    private String parityFolder;
    private final long recheckInterval;
    private final int activeEncodingLimit;
    private int activeEncodings = 0;
    private final int activeRepairLimit;
    private final int activeParityRepairLimit;
    private int activeRepairs = 0;
    private int activeParityRepairs = 0;
    private final int repairDelay;
    private final int parityRepairDelay;
    private final int deletionLimit;
    private static boolean enabled = false;

    public ErasureCodingManager(FSNamesystem namesystem, Configuration conf) {
        super(conf);
        this.namesystem = namesystem;
        this.parityFolder = conf.get("dfs.erasure_coding.parity_folder", "/parity");
        this.recheckInterval = conf.getInt("dfs.erasure_coding.recheck_interval", 300000);
        this.activeEncodingLimit = conf.getInt("dfs.erasure_coding.active_encoding_limit", 10);
        this.activeRepairLimit = conf.getInt("dfs.erasure_coding.active_repair_limit", 10);
        this.activeParityRepairLimit = conf.getInt("dfs.erasure_coding.active_parity_repair_limit", 10);
        this.repairDelay = conf.getInt("dfs.erasure_coding.repair_delay", 1800000);
        this.parityRepairDelay = conf.getInt("dfs.erasure_coding.parity_repair_delay", 1800000);
        this.deletionLimit = conf.getInt("dfs.erasure_coding.deletion_limit", 100);
        enabled = conf.getBoolean("dfs.erasure_coding.enabled", false);
    }

    private boolean loadRaidNodeClasses() {
        try {
            Class<?> encodingManagerClass = this.getConf().getClass("dfs.erasure_coding.encoding_manager", null);
            if (encodingManagerClass == null) {
                encodingManagerClass = Class.forName("io.hops.erasure_coding.MapReduceEncodingManager");
            }
            if (!EncodingManager.class.isAssignableFrom(encodingManagerClass)) {
                throw new ClassNotFoundException(encodingManagerClass + " is not an implementation of " + EncodingManager.class.getCanonicalName());
            }
            Constructor<?> encodingManagerConstructor = encodingManagerClass.getConstructor(Configuration.class);
            this.encodingManager = (EncodingManager)encodingManagerConstructor.newInstance(this.getConf());
            Class<?> blockRepairManagerClass = this.getConf().getClass("dfs.erasure_coding.block_rapair_manager", null);
            if (blockRepairManagerClass == null) {
                blockRepairManagerClass = Class.forName("io.hops.erasure_coding.MapReduceBlockRepairManager");
            }
            if (!BlockRepairManager.class.isAssignableFrom(blockRepairManagerClass)) {
                throw new ClassNotFoundException(blockRepairManagerClass + " is not an implementation of " + BlockRepairManager.class.getCanonicalName());
            }
            Constructor<?> blockRepairManagerConstructor = blockRepairManagerClass.getConstructor(Configuration.class);
            this.blockRepairManager = (BlockRepairManager)blockRepairManagerConstructor.newInstance(this.getConf());
        }
        catch (Exception e) {
            LOG.error((Object)"Could not load erasure coding classes", (Throwable)e);
            return false;
        }
        return true;
    }

    public void activate() {
        if (!this.loadRaidNodeClasses()) {
            LOG.error((Object)"ErasureCodingMonitor not started. An error occurred during the loading of the encoding library.");
            return;
        }
        this.erasureCodingMonitorThread.start();
        LOG.info((Object)"ErasureCodingMonitor started");
    }

    public void close() {
        try {
            if (this.erasureCodingMonitorThread != null) {
                this.erasureCodingMonitorThread.interrupt();
                this.erasureCodingMonitorThread.join(3000L);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        LOG.info((Object)"ErasureCodingMonitor stopped");
    }

    public static boolean isErasureCodingEnabled(Configuration conf) {
        return conf.getBoolean("dfs.erasure_coding.enabled", false);
    }

    private void checkActiveEncodings() throws IOException {
        LOG.info((Object)"Checking active encoding.");
        List<Report> reports = this.encodingManager.computeReports();
        for (Report report : reports) {
            switch (report.getStatus()) {
                case ACTIVE: {
                    break;
                }
                case FINISHED: {
                    LOG.info((Object)("Encoding finished for " + report.getFilePath()));
                    this.finalizeEncoding(report.getFilePath());
                    --this.activeEncodings;
                    break;
                }
                case FAILED: {
                    LOG.info((Object)("Encoding failed for " + report.getFilePath()));
                    this.updateEncodingStatus(report.getFilePath(), EncodingStatus.Status.ENCODING_FAILED, EncodingStatus.ParityStatus.REPAIR_FAILED);
                    --this.activeEncodings;
                    break;
                }
                case CANCELED: {
                    LOG.info((Object)("Encoding canceled for " + report.getFilePath()));
                    this.updateEncodingStatus(report.getFilePath(), EncodingStatus.Status.ENCODING_CANCELED);
                    --this.activeEncodings;
                }
            }
        }
    }

    private void finalizeEncoding(final String path) {
        LOG.info((Object)("Finilizing encoding for " + path));
        try {
            new HopsTransactionalRequestHandler(HDFSOperationType.GET_INODE){
                private String parityPath;

                @Override
                public void setUp() throws StorageException, IOException {
                    super.setUp();
                    EncodingStatus status = ErasureCodingManager.this.namesystem.getEncodingStatus(path);
                    this.parityPath = ErasureCodingManager.this.parityFolder + "/" + status.getParityFileName();
                }

                public void acquireLock(TransactionLocks locks) throws IOException {
                    LockFactory lf = LockFactory.getInstance();
                    INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, path, this.parityPath).setNameNodeID(ErasureCodingManager.this.namesystem.getNameNode().getId()).setActiveNameNodes(ErasureCodingManager.this.namesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                    locks.add((Lock)il).add(lf.getEncodingStatusLock(TransactionLockTypes.LockType.WRITE, path));
                }

                public Object performTask() throws StorageException, IOException {
                    INode sourceInode = ErasureCodingManager.this.namesystem.getINode(path);
                    INode parityInode = ErasureCodingManager.this.namesystem.getINode(this.parityPath);
                    if (sourceInode == null) {
                        return null;
                    }
                    EncodingStatus encodingStatus = (EncodingStatus)EntityManager.find((FinderType)EncodingStatus.Finder.ByInodeId, (Object[])new Object[]{sourceInode.getId()});
                    if (encodingStatus.getStatus() != EncodingStatus.Status.ENCODING_ACTIVE) {
                        return null;
                    }
                    if (parityInode == null) {
                        encodingStatus.setStatus(EncodingStatus.Status.ENCODING_FAILED);
                        encodingStatus.setStatusModificationTime(Long.valueOf(System.currentTimeMillis()));
                    } else {
                        encodingStatus.setStatus(EncodingStatus.Status.ENCODED);
                        encodingStatus.setStatusModificationTime(Long.valueOf(System.currentTimeMillis()));
                        encodingStatus.setParityInodeId(Long.valueOf(parityInode.getId()));
                        encodingStatus.setParityStatus(EncodingStatus.ParityStatus.HEALTHY);
                        encodingStatus.setParityStatusModificationTime(Long.valueOf(System.currentTimeMillis()));
                    }
                    EntityManager.update((Object)encodingStatus);
                    return null;
                }
            }.handle((Object)this);
        }
        catch (IOException e) {
            LOG.error((Object)StringUtils.stringifyException((Throwable)e));
        }
    }

    private void updateEncodingStatus(String filePath, EncodingStatus.Status status, EncodingStatus.ParityStatus parityStatus) {
        try {
            this.namesystem.updateEncodingStatus(filePath, status, parityStatus, null);
        }
        catch (IOException e) {
            LOG.error((Object)StringUtils.stringifyException((Throwable)e));
        }
    }

    private void updateEncodingStatus(String filePath, EncodingStatus.Status status) {
        this.updateEncodingStatus(filePath, status, null);
    }

    private void updateEncodingStatus(String filePath, EncodingStatus.ParityStatus status) {
        this.updateEncodingStatus(filePath, null, status);
    }

    private void scheduleEncodings() throws IOException {
        LOG.info((Object)"Schedule encodings.");
        final int limit = this.activeEncodingLimit - this.activeEncodings;
        if (limit <= 0) {
            return;
        }
        LightWeightRequestHandler findHandler = new LightWeightRequestHandler(EncodingStatusOperationType.FIND_REQUESTED_ENCODINGS){

            public Object performTask() throws StorageException, IOException {
                EncodingStatusDataAccess dataAccess = (EncodingStatusDataAccess)HdfsStorageFactory.getDataAccess(EncodingStatusDataAccess.class);
                return dataAccess.findRequestedEncodings(limit);
            }
        };
        Collection requestedEncodings = (Collection)findHandler.handle();
        for (EncodingStatus encodingStatus : requestedEncodings) {
            try {
                LOG.info((Object)("Trying to schedule encoding for " + encodingStatus));
                INode iNode = this.namesystem.findInode(encodingStatus.getInodeId());
                if (iNode == null) {
                    LOG.error((Object)("findInode returned null for id " + encodingStatus.getInodeId()));
                    continue;
                }
                if (iNode.isUnderConstruction()) {
                    LOG.info((Object)("Still under construction. Encoding not scheduled for " + iNode.getId()));
                    continue;
                }
                String path = this.namesystem.getPath(iNode.getId(), iNode.isInTree());
                if (iNode == null) continue;
                LOG.info((Object)("Schedule encoding for " + path));
                UUID parityFileName = UUID.randomUUID();
                this.encodingManager.encodeFile(encodingStatus.getEncodingPolicy(), new Path(path), new Path(this.parityFolder + "/" + parityFileName.toString()), encodingStatus.getStatus() == EncodingStatus.Status.COPY_ENCODING_REQUESTED);
                this.namesystem.updateEncodingStatus(path, EncodingStatus.Status.ENCODING_ACTIVE, parityFileName.toString());
                ++this.activeEncodings;
            }
            catch (IOException e) {
                LOG.error((Object)StringUtils.stringifyException((Throwable)e));
            }
        }
    }

    private void checkActiveRepairs() throws IOException {
        LOG.info((Object)"Checking active repairs.");
        List<Report> reports = this.blockRepairManager.computeReports();
        for (Report report : reports) {
            switch (report.getStatus()) {
                case ACTIVE: {
                    break;
                }
                case FINISHED: {
                    LOG.info((Object)("Repair finished for " + report.getFilePath()));
                    if (this.isParityFile(report.getFilePath())) {
                        this.checkFixedParity(report.getFilePath());
                        --this.activeParityRepairs;
                        break;
                    }
                    this.checkFixedSource(report.getFilePath());
                    --this.activeRepairs;
                    break;
                }
                case FAILED: {
                    LOG.info((Object)("Repair failed for " + report.getFilePath()));
                    if (this.isParityFile(report.getFilePath())) {
                        this.updateEncodingStatus(report.getFilePath(), EncodingStatus.ParityStatus.REPAIR_FAILED);
                        --this.activeParityRepairs;
                        break;
                    }
                    this.updateEncodingStatus(report.getFilePath(), EncodingStatus.Status.REPAIR_FAILED);
                    --this.activeRepairs;
                    break;
                }
                case CANCELED: {
                    LOG.info((Object)("Repair canceled for " + report.getFilePath()));
                    if (this.isParityFile(report.getFilePath())) {
                        this.updateEncodingStatus(report.getFilePath(), EncodingStatus.ParityStatus.REPAIR_CANCELED);
                        --this.activeParityRepairs;
                        break;
                    }
                    this.updateEncodingStatus(report.getFilePath(), EncodingStatus.Status.REPAIR_CANCELED);
                    --this.activeRepairs;
                }
            }
        }
    }

    private void checkFixedSource(final String path) throws IOException {
        new HopsTransactionalRequestHandler(HDFSOperationType.CHECK_FIXED_SOURCE){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, path).setNameNodeID(ErasureCodingManager.this.namesystem.getNameNode().getId()).setActiveNameNodes(ErasureCodingManager.this.namesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il).add(lf.getEncodingStatusLock(TransactionLockTypes.LockType.WRITE, path));
            }

            public Object performTask() throws IOException {
                INode targetNode = ErasureCodingManager.this.namesystem.getINode(path);
                EncodingStatus status = (EncodingStatus)EntityManager.find((FinderType)EncodingStatus.Finder.ByInodeId, (Object[])new Object[]{targetNode.getId()});
                if (status.getLostBlocks() == 0) {
                    status.setStatus(EncodingStatus.Status.ENCODED);
                } else {
                    status.setStatus(EncodingStatus.Status.REPAIR_REQUESTED);
                }
                status.setStatusModificationTime(Long.valueOf(System.currentTimeMillis()));
                EntityManager.update((Object)status);
                return null;
            }
        }.handle();
    }

    private void checkFixedParity(final String path) throws IOException {
        new HopsTransactionalRequestHandler(HDFSOperationType.CHECK_FIXED_PARITY){

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                INodeLock il = lf.getINodeLock(TransactionLockTypes.INodeLockType.WRITE, TransactionLockTypes.INodeResolveType.PATH, path).setNameNodeID(ErasureCodingManager.this.namesystem.getNameNode().getId()).setActiveNameNodes(ErasureCodingManager.this.namesystem.getNameNode().getActiveNameNodes().getActiveNodes());
                locks.add((Lock)il).add(lf.getEncodingStatusLock(TransactionLockTypes.LockType.WRITE, path));
            }

            public Object performTask() throws IOException {
                INode targetNode = ErasureCodingManager.this.namesystem.getINode(path);
                EncodingStatus status = (EncodingStatus)EntityManager.find((FinderType)EncodingStatus.Finder.ByParityInodeId, (Object[])new Object[]{targetNode.getId()});
                if (status.getLostParityBlocks() == 0) {
                    status.setParityStatus(EncodingStatus.ParityStatus.HEALTHY);
                } else {
                    status.setParityStatus(EncodingStatus.ParityStatus.REPAIR_REQUESTED);
                }
                status.setParityStatusModificationTime(Long.valueOf(System.currentTimeMillis()));
                EntityManager.update((Object)status);
                return null;
            }
        }.handle();
    }

    private void scheduleSourceRepairs() throws IOException {
        LOG.info((Object)"Scheduling repairs");
        final int limit = this.activeRepairLimit - this.activeRepairs;
        if (limit <= 0) {
            return;
        }
        LightWeightRequestHandler findHandler = new LightWeightRequestHandler(EncodingStatusOperationType.FIND_REQUESTED_REPAIRS){

            public Object performTask() throws IOException {
                EncodingStatusDataAccess dataAccess = (EncodingStatusDataAccess)HdfsStorageFactory.getDataAccess(EncodingStatusDataAccess.class);
                return dataAccess.findRequestedRepairs(limit);
            }
        };
        Collection requestedRepairs = (Collection)findHandler.handle();
        for (EncodingStatus encodingStatus : requestedRepairs) {
            try {
                LOG.info((Object)("Scheduling source repair  for " + encodingStatus));
                if (System.currentTimeMillis() - encodingStatus.getStatusModificationTime() < (long)this.repairDelay) {
                    LOG.info((Object)("Skipping source repair. Delay not reached: " + this.repairDelay));
                    continue;
                }
                if (encodingStatus.isParityRepairActive()) {
                    LOG.info((Object)"Skipping source repair. Parity repair is active");
                    continue;
                }
                String path = this.namesystem.getPath(encodingStatus.getInodeId(), encodingStatus.isInTree());
                this.namesystem.updateEncodingStatus(path, EncodingStatus.Status.REPAIR_ACTIVE);
                LOG.info((Object)("Status set to source repair active " + encodingStatus));
                this.blockRepairManager.repairSourceBlocks(encodingStatus.getEncodingPolicy().getCodec(), new Path(path), new Path(this.parityFolder + "/" + encodingStatus.getParityFileName()));
                LOG.info((Object)("Scheduled job for source repair " + encodingStatus));
                ++this.activeRepairs;
            }
            catch (IOException e) {
                LOG.error((Object)StringUtils.stringifyException((Throwable)e));
            }
        }
    }

    private void scheduleParityRepairs() {
        LOG.info((Object)"Scheduling parity repairs");
        final int limit = this.activeParityRepairLimit - this.activeParityRepairs;
        if (limit <= 0) {
            return;
        }
        LightWeightRequestHandler findHandler = new LightWeightRequestHandler(EncodingStatusOperationType.FIND_REQUESTED_PARITY_REPAIRS){

            public Object performTask() throws IOException {
                EncodingStatusDataAccess dataAccess = (EncodingStatusDataAccess)HdfsStorageFactory.getDataAccess(EncodingStatusDataAccess.class);
                return dataAccess.findRequestedParityRepairs(limit);
            }
        };
        try {
            Collection requestedRepairs = (Collection)findHandler.handle();
            for (EncodingStatus encodingStatus : requestedRepairs) {
                LOG.info((Object)("Scheduling parity repair for " + encodingStatus));
                if (System.currentTimeMillis() - encodingStatus.getParityStatusModificationTime() < (long)this.parityRepairDelay) {
                    LOG.info((Object)("Skipping  parity repair. Delay not reached: " + this.parityRepairDelay));
                    continue;
                }
                if (!encodingStatus.getStatus().equals((Object)EncodingStatus.Status.ENCODED)) {
                    LOG.info((Object)"Skipping parity repair. Source file not healthy.");
                    continue;
                }
                String path = this.namesystem.getPath(encodingStatus.getInodeId(), encodingStatus.isInTree());
                this.namesystem.updateEncodingStatus(path, EncodingStatus.ParityStatus.REPAIR_ACTIVE);
                LOG.info((Object)("Status set to parity repair active " + encodingStatus));
                this.blockRepairManager.repairParityBlocks(encodingStatus.getEncodingPolicy().getCodec(), new Path(path), new Path(this.parityFolder + "/" + encodingStatus.getParityFileName()));
                LOG.info((Object)("Scheduled job for parity repair " + encodingStatus));
                ++this.activeRepairs;
            }
        }
        catch (IOException e) {
            LOG.error((Object)StringUtils.stringifyException((Throwable)e));
        }
    }

    private void garbageCollect() throws IOException {
        LOG.info((Object)"Starting garbage collection");
        LightWeightRequestHandler findHandler = new LightWeightRequestHandler(EncodingStatusOperationType.FIND_DELETED){

            public Object performTask() throws IOException {
                EncodingStatusDataAccess dataAccess = (EncodingStatusDataAccess)HdfsStorageFactory.getDataAccess(EncodingStatusDataAccess.class);
                return dataAccess.findDeleted(ErasureCodingManager.this.deletionLimit);
            }
        };
        Collection markedAsDeleted = (Collection)findHandler.handle();
        for (EncodingStatus status : markedAsDeleted) {
            LOG.info((Object)("Trying to collect " + status));
            try {
                this.namesystem.deleteWithTransaction(this.parityFolder + "/" + status.getParityFileName(), false);
                this.namesystem.removeEncodingStatus(status);
            }
            catch (IOException e) {
                LOG.error((Object)StringUtils.stringifyException((Throwable)e));
            }
        }
    }

    private void checkRevoked() throws IOException {
        LOG.info((Object)"Checking replication for revocations");
        LightWeightRequestHandler findHandler = new LightWeightRequestHandler(EncodingStatusOperationType.FIND_REVOKED){

            public Object performTask() throws IOException {
                EncodingStatusDataAccess dataAccess = (EncodingStatusDataAccess)HdfsStorageFactory.getDataAccess(EncodingStatusDataAccess.class);
                return dataAccess.findRevoked();
            }
        };
        Collection markedAsRevoked = (Collection)findHandler.handle();
        for (EncodingStatus status : markedAsRevoked) {
            LOG.info((Object)("Checking replication for revoked status: " + status));
            String path = this.namesystem.getPath(status.getInodeId(), status.isInTree());
            short replication = this.namesystem.getFileInfo(path, true).getReplication();
            LocatedBlocks blocks = this.namesystem.getBlockLocations(path, 0L, Long.MAX_VALUE, false, true, true);
            if (!this.checkReplication(blocks, replication)) continue;
            LOG.info((Object)("Revocation successful for " + status));
            this.namesystem.deleteWithTransaction(this.parityFolder + "/" + status.getParityFileName(), false);
            this.namesystem.removeEncodingStatus(path, status);
        }
    }

    private boolean checkReplication(LocatedBlocks blocks, int replication) {
        for (LocatedBlock locatedBlock : blocks.getLocatedBlocks()) {
            if (locatedBlock.getLocations().length == replication) continue;
            return false;
        }
        return true;
    }

    public boolean isParityFile(String path) {
        Pattern pattern = Pattern.compile(this.parityFolder + ".*");
        Matcher matcher = pattern.matcher(path);
        return matcher.matches();
    }

    public static boolean isEnabled() {
        return enabled;
    }

    private class ErasureCodingMonitor
    implements Runnable {
        private ErasureCodingMonitor() {
        }

        @Override
        public void run() {
            while (ErasureCodingManager.this.namesystem.isRunning()) {
                try {
                    try {
                        if (ErasureCodingManager.this.namesystem.isInSafeMode()) {
                            Thread.sleep(ErasureCodingManager.this.recheckInterval);
                            continue;
                        }
                    }
                    catch (IOException e) {
                        LOG.info((Object)"In safe mode skipping this round");
                    }
                    if (ErasureCodingManager.this.namesystem.isLeader()) {
                        ErasureCodingManager.this.checkActiveEncodings();
                        ErasureCodingManager.this.scheduleEncodings();
                        ErasureCodingManager.this.checkActiveRepairs();
                        ErasureCodingManager.this.scheduleSourceRepairs();
                        ErasureCodingManager.this.scheduleParityRepairs();
                        ErasureCodingManager.this.garbageCollect();
                        ErasureCodingManager.this.checkRevoked();
                    }
                    try {
                        Thread.sleep(ErasureCodingManager.this.recheckInterval);
                    }
                    catch (InterruptedException ie) {
                        LOG.warn((Object)"ErasureCodingMonitor thread received InterruptedException.", (Throwable)ie);
                        break;
                    }
                }
                catch (Throwable e) {
                    LOG.error((Object)e);
                }
            }
        }
    }
}

