/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.protobuf.InvalidProtocolBufferException;
import io.hops.common.IDsGeneratorFactory;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.DirectoryWithQuotaFeatureDataAccess;
import io.hops.metadata.hdfs.dal.INodeDataAccess;
import io.hops.metadata.hdfs.entity.INodeIdentifier;
import io.hops.resolvingcache.Cache;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import org.apache.commons.io.Charsets;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockStoragePolicy;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.FSLimitException;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelper;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguousUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockStoragePolicySuite;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.DirectoryWithQuotaFeature;
import org.apache.hadoop.hdfs.server.namenode.EncryptionZoneManager;
import org.apache.hadoop.hdfs.server.namenode.FSDirStatAndListingOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirXAttrOp;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FSPermissionChecker;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeWithAdditionalFields;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameCache;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.QuotaCounts;
import org.apache.hadoop.hdfs.server.namenode.XAttrFeature;
import org.apache.hadoop.hdfs.server.namenode.XAttrStorage;
import org.apache.hadoop.hdfs.util.ByteArray;
import org.apache.hadoop.hdfs.util.EnumCounters;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FSDirectory
implements Closeable {
    static final Logger LOG = LoggerFactory.getLogger(FSDirectory.class);
    @VisibleForTesting
    static boolean CHECK_RESERVED_FILE_NAMES = true;
    public static final String DOT_RESERVED_STRING = ".reserved";
    public static final String DOT_RESERVED_PATH_PREFIX = "/.reserved";
    public static final byte[] DOT_RESERVED = DFSUtil.string2Bytes(".reserved");
    private static final String RAW_STRING = "raw";
    private static final byte[] RAW = DFSUtil.string2Bytes("raw");
    public static final String DOT_INODES_STRING = ".inodes";
    public static final byte[] DOT_INODES = DFSUtil.string2Bytes(".inodes");
    private final FSNamesystem namesystem;
    private final int maxComponentLength;
    private final int maxDirItems;
    private final int lsLimit;
    private final int contentCountLimit;
    private final long contentSleepMicroSec;
    private long yieldCount = 0L;
    private final int inodeXAttrsLimit;
    private boolean quotaEnabled;
    private final boolean isPermissionEnabled;
    private final boolean aclsEnabled;
    private final boolean xattrsEnabled;
    private final int xattrMaxSize;
    private final long accessTimePrecision;
    private final boolean storagePolicyEnabled;
    private final boolean quotaByStorageTypeEnabled;
    private final String fsOwnerShortUserName;
    private final String supergroup;
    @VisibleForTesting
    public final EncryptionZoneManager ezManager;
    private final NameCache<ByteArray> nameCache;

    FSDirectory(FSNamesystem ns, Configuration conf) throws IOException {
        this.quotaEnabled = conf.getBoolean("dfs.namenode.quota.enabled", true);
        this.namesystem = ns;
        this.createRoot(ns.createFsOwnerPermissions(new FsPermission(493)), false);
        this.isPermissionEnabled = conf.getBoolean("dfs.permissions.enabled", true);
        this.fsOwnerShortUserName = UserGroupInformation.getCurrentUser().getShortUserName();
        this.supergroup = conf.get("dfs.permissions.superusergroup", "supergroup");
        this.aclsEnabled = conf.getBoolean("dfs.namenode.acls.enabled", false);
        LOG.info("ACLs enabled? " + this.aclsEnabled);
        this.xattrsEnabled = conf.getBoolean("dfs.namenode.xattrs.enabled", true);
        LOG.info("XAttrs enabled? " + this.xattrsEnabled);
        this.xattrMaxSize = conf.getInt("dfs.namenode.fs-limits.max-xattr-size", DFSConfigKeys.DFS_NAMENODE_MAX_XATTR_SIZE_DEFAULT);
        Preconditions.checkArgument((this.xattrMaxSize > 0 ? 1 : 0) != 0, (String)"The maximum size of an xattr should be > 0: (%s).", (Object)"dfs.namenode.fs-limits.max-xattr-size");
        Preconditions.checkArgument((this.xattrMaxSize <= DFSConfigKeys.DFS_NAMENODE_MAX_XATTR_SIZE_HARD_LIMIT ? 1 : 0) != 0, (String)("The maximum size of an xattr should be <= maximum size hard limit " + DFSConfigKeys.DFS_NAMENODE_MAX_XATTR_SIZE_HARD_LIMIT + ": (%s)."), (Object)"dfs.namenode.fs-limits.max-xattr-size");
        LOG.info("Maximum size of an xattr: " + this.xattrMaxSize);
        int configuredLimit = conf.getInt("dfs.ls.limit", Integer.MAX_VALUE);
        this.lsLimit = configuredLimit > 0 ? configuredLimit : Integer.MAX_VALUE;
        this.accessTimePrecision = conf.getLong("dfs.namenode.accesstime.precision", 3600000L);
        this.storagePolicyEnabled = conf.getBoolean("dfs.storage.policy.enabled", true);
        this.quotaByStorageTypeEnabled = conf.getBoolean("dfs.quota.by.storage.type.enabled", true);
        this.contentCountLimit = conf.getInt("dfs.content-summary.limit", 5000);
        this.contentSleepMicroSec = conf.getLong("dfs.content-summary.sleep-microsec", 500L);
        this.maxComponentLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 255);
        this.maxDirItems = conf.getInt("dfs.namenode.fs-limits.max-directory-items", 10240);
        Preconditions.checkArgument((this.maxDirItems >= 0 ? 1 : 0) != 0, (Object)"Cannot set dfs.namenode.fs-limits.max-directory-items to a value less than 0");
        int inodeXAttrs = conf.getInt("dfs.namenode.fs-limits.max-xattrs-per-inode", 32);
        Preconditions.checkArgument((inodeXAttrs >= 0 ? 1 : 0) != 0, (String)"Cannot set a negative limit on the number of xattrs per inode (%s).", (Object)"dfs.namenode.fs-limits.max-xattrs-per-inode");
        if (inodeXAttrs > XAttrStorage.getMaxNumberOfUserXAttrPerInode()) {
            inodeXAttrs = XAttrStorage.getMaxNumberOfUserXAttrPerInode();
        }
        this.inodeXAttrsLimit = inodeXAttrs;
        NameNode.LOG.info("The maximum number of xattrs per inode is set to " + this.inodeXAttrsLimit);
        int threshold = conf.getInt("dfs.namenode.name.cache.threshold", 10);
        NameNode.LOG.info("Caching file names occuring more than " + threshold + " times");
        this.nameCache = new NameCache(threshold);
        this.ezManager = new EncryptionZoneManager(this, conf);
    }

    public FSNamesystem getFSNamesystem() {
        return this.namesystem;
    }

    private BlockManager getBlockManager() {
        return this.getFSNamesystem().getBlockManager();
    }

    public BlockStoragePolicySuite getBlockStoragePolicySuite() {
        return this.getBlockManager().getStoragePolicySuite();
    }

    boolean isPermissionEnabled() {
        return this.isPermissionEnabled;
    }

    boolean isAclsEnabled() {
        return this.aclsEnabled;
    }

    boolean isStoragePolicyEnabled() {
        return this.storagePolicyEnabled;
    }

    boolean isAccessTimeSupported() {
        return this.accessTimePrecision > 0L;
    }

    boolean isQuotaByStorageTypeEnabled() {
        return this.quotaByStorageTypeEnabled;
    }

    boolean isXattrsEnabled() {
        return this.xattrsEnabled;
    }

    int getXattrMaxSize() {
        return this.xattrMaxSize;
    }

    int getLsLimit() {
        return this.lsLimit;
    }

    int getInodeXAttrsLimit() {
        return this.inodeXAttrsLimit;
    }

    int getContentCountLimit() {
        return this.contentCountLimit;
    }

    long getContentSleepMicroSec() {
        return this.contentSleepMicroSec;
    }

    @Override
    public void close() throws IOException {
    }

    void markNameCacheInitialized() {
        this.nameCache.initialized();
    }

    INodesInPath addFile(INodesInPath existing, String localName, PermissionStatus permissions, short replication, long preferredBlockSize, String clientName, String clientMachine) throws IOException {
        long modTime = Time.now();
        INodeFile newNode = new INodeFile(IDsGeneratorFactory.getInstance().getUniqueINodeID(), permissions, BlockInfoContiguous.EMPTY_ARRAY, replication, modTime, modTime, preferredBlockSize, 0);
        newNode.setLocalNameNoPersistance(localName.getBytes(Charsets.UTF_8));
        newNode.toUnderConstruction(clientName, clientMachine);
        INodesInPath newiip = this.addINode(existing, newNode);
        if (newiip == null) {
            NameNode.stateChangeLog.info("DIR* addFile: failed to add " + existing.getPath() + "/" + localName);
            return null;
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug("DIR* addFile: " + localName + " is added");
        }
        return newiip;
    }

    BlockInfoContiguous addBlock(String path, INodesInPath inodesInPath, Block block, DatanodeStorageInfo[] targets) throws QuotaExceededException, StorageException, TransactionContextException, IOException {
        INodeFile fileINode = inodesInPath.getLastINode().asFile();
        Preconditions.checkState((boolean)fileINode.isUnderConstruction());
        long diskspaceTobeConsumed = fileINode.getPreferredBlockSize();
        if (fileINode.isFileStoredInDB()) {
            diskspaceTobeConsumed -= fileINode.getSize();
        }
        this.updateCount(inodesInPath, 0L, diskspaceTobeConsumed, fileINode.getBlockReplication(), true);
        BlockInfoContiguousUnderConstruction blockInfo = new BlockInfoContiguousUnderConstruction(block, fileINode.getId(), HdfsServerConstants.BlockUCState.UNDER_CONSTRUCTION, targets);
        this.getBlockManager().addBlockCollection(blockInfo, fileINode);
        fileINode.addBlock(blockInfo);
        fileINode.getFileUnderConstructionFeature().setLastBlockId(blockInfo.getBlockId());
        fileINode.setHasBlocks(true);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug("DIR* FSDirectory.addBlock: " + path + " with " + block + " block is added to the in-memory file system");
        }
        return blockInfo;
    }

    boolean removeBlock(String path, INodesInPath iip, INodeFile fileNode, Block block) throws IOException, StorageException {
        Preconditions.checkArgument((boolean)fileNode.isUnderConstruction());
        return this.unprotectedRemoveBlock(path, iip, fileNode, block);
    }

    boolean unprotectedRemoveBlock(String path, INodesInPath iip, INodeFile fileNode, Block block) throws IOException, StorageException {
        Preconditions.checkArgument((boolean)fileNode.isUnderConstruction());
        boolean removed = fileNode.removeLastBlock(block);
        if (!removed) {
            return false;
        }
        this.getBlockManager().addToInvalidates(block);
        this.getBlockManager().removeBlockFromMap(block);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug("DIR* FSDirectory.removeReplica: " + path + " with " + block + " block is removed from the file system");
        }
        this.updateCount(iip, 0L, -fileNode.getPreferredBlockSize(), fileNode.getBlockReplication(), true);
        return true;
    }

    String resolvePath(FSPermissionChecker pc, String path, byte[][] pathComponents) throws FileNotFoundException, AccessControlException, IOException {
        if (FSDirectory.isReservedRawName(path) && this.isPermissionEnabled) {
            pc.checkSuperuserPrivilege();
        }
        return FSDirectory.resolvePath(path, pathComponents, this);
    }

    boolean isNonEmptyDirectory(INodesInPath inodesInPath) throws UnresolvedLinkException, StorageException, TransactionContextException {
        INode inode = inodesInPath.getLastINode();
        if (inode == null || !inode.isDirectory()) {
            return false;
        }
        return ((INodeDirectory)inode).getChildrenList().size() != 0;
    }

    boolean isValidToCreate(String src, INodesInPath iip) throws UnresolvedLinkException, StorageException, TransactionContextException {
        String srcs = FSDirectory.normalizePath(src);
        return srcs.startsWith("/") && !srcs.endsWith("/") && iip.getLastINode() == null;
    }

    boolean isDir(String src) throws UnresolvedLinkException, StorageException, TransactionContextException {
        INode node = this.getINode(src = FSDirectory.normalizePath(src), false);
        return node != null && node.isDirectory();
    }

    void updateSpaceConsumed(INodesInPath iip, long nsDelta, long ssDelta, short replication) throws QuotaExceededException, FileNotFoundException, UnresolvedLinkException, StorageException, TransactionContextException {
        if (iip.getLastINode() == null) {
            throw new FileNotFoundException("Path not found: " + iip.getPath());
        }
        this.updateCount(iip, nsDelta, ssDelta, replication, true);
    }

    void updateCount(INodesInPath iip, long nsDelta, long ssDelta, short replication, boolean checkQuota) throws QuotaExceededException, StorageException, TransactionContextException {
        INodeFile fileINode = iip.getLastINode().asFile();
        EnumCounters<StorageType> typeSpaceDeltas = this.getStorageTypeDeltas(fileINode.getStoragePolicyID(), ssDelta, replication, replication);
        this.updateCount(iip, iip.length() - 1, new QuotaCounts.Builder().nameSpace(nsDelta).storageSpace(ssDelta * (long)replication).typeSpaces(typeSpaceDeltas).build(), checkQuota);
    }

    void updateCount(INodesInPath iip, long nsDelta, long ssDelta, short oldRep, short newRep, boolean checkQuota) throws QuotaExceededException, StorageException, TransactionContextException {
        INodeFile fileINode = iip.getLastINode().asFile();
        EnumCounters<StorageType> typeSpaceDeltas = this.getStorageTypeDeltas(fileINode.getStoragePolicyID(), ssDelta, oldRep, newRep);
        this.updateCount(iip, iip.length() - 1, new QuotaCounts.Builder().nameSpace(nsDelta).storageSpace(ssDelta * (long)(newRep - oldRep)).typeSpaces(typeSpaceDeltas).build(), checkQuota);
    }

    private void updateCount(INodesInPath iip, int numOfINodes, QuotaCounts counts, boolean checkQuota) throws QuotaExceededException, StorageException, TransactionContextException {
        if (!this.isQuotaEnabled()) {
            return;
        }
        if (!this.namesystem.isImageLoaded()) {
            return;
        }
        if (numOfINodes > iip.length()) {
            numOfINodes = iip.length();
        }
        if (checkQuota) {
            FSDirectory.verifyQuota(iip, numOfINodes, counts, null);
        }
        INode iNode = iip.getINode(numOfINodes - 1);
        this.namesystem.getQuotaUpdateManager().addUpdate(iNode.getId(), counts);
    }

    void updateCountNoQuotaCheck(INodesInPath inodesInPath, int numOfINodes, QuotaCounts counts) throws StorageException, TransactionContextException {
        try {
            this.updateCount(inodesInPath, numOfINodes, counts, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn("FSDirectory.updateCountNoQuotaCheck - unexpected ", (Throwable)e);
        }
    }

    void unprotectedUpdateCount(INodesInPath inodesInPath, int numOfINodes, QuotaCounts counts) throws StorageException, TransactionContextException {
        if (!this.isQuotaEnabled()) {
            return;
        }
        INode iNode = inodesInPath.getINode(numOfINodes - 1);
        this.namesystem.getQuotaUpdateManager().addUpdate(iNode.getId(), counts);
    }

    public EnumCounters<StorageType> getStorageTypeDeltas(byte storagePolicyID, long dsDelta, short oldRep, short newRep) {
        EnumCounters<StorageType> typeSpaceDeltas = new EnumCounters<StorageType>(StorageType.class);
        if (dsDelta == 0L) {
            return typeSpaceDeltas;
        }
        if (storagePolicyID != 0) {
            BlockStoragePolicy storagePolicy = this.getBlockManager().getStoragePolicy(storagePolicyID);
            if (oldRep != newRep) {
                List oldChosenStorageTypes = storagePolicy.chooseStorageTypes(oldRep);
                for (StorageType t : oldChosenStorageTypes) {
                    if (!t.supportTypeQuota()) continue;
                    Preconditions.checkArgument((dsDelta > 0L ? 1 : 0) != 0);
                    typeSpaceDeltas.add(t, -dsDelta);
                }
            }
            List newChosenStorageTypes = storagePolicy.chooseStorageTypes(newRep);
            for (StorageType t : newChosenStorageTypes) {
                if (!t.supportTypeQuota()) continue;
                typeSpaceDeltas.add(t, dsDelta);
            }
        }
        return typeSpaceDeltas;
    }

    static String getFullPathName(INode[] inodes, int pos) {
        StringBuilder fullPathName = new StringBuilder();
        if (inodes[0].isRoot()) {
            if (pos == 0) {
                return "/";
            }
        } else {
            fullPathName.append(inodes[0].getLocalName());
        }
        for (int i = 1; i <= pos; ++i) {
            fullPathName.append('/').append(inodes[i].getLocalName());
        }
        return fullPathName.toString();
    }

    static String getFullPathName(INode inode) throws StorageException, TransactionContextException {
        int depth = 0;
        for (INode i = inode; i != null; i = i.getParent()) {
            ++depth;
        }
        INode[] inodes = new INode[depth];
        for (int i = 0; i < depth; ++i) {
            if (inode == null) {
                NameNode.stateChangeLog.warn("Could not get full path. Corresponding file might have deleted already.");
                return null;
            }
            inodes[depth - i - 1] = inode;
            inode = inode.getParent();
        }
        return FSDirectory.getFullPathName(inodes, depth - 1);
    }

    INodesInPath addINode(INodesInPath existing, INode child) throws IOException {
        this.cacheName(child);
        return this.addLastINode(existing, child, true);
    }

    static void verifyQuota(INodesInPath iip, int pos, QuotaCounts deltas, INode commonAncestor) throws QuotaExceededException, StorageException, TransactionContextException {
        if (deltas.getNameSpace() <= 0L && deltas.getStorageSpace() <= 0L && deltas.getTypeSpaces().allLessOrEqual(0L)) {
            return;
        }
        for (int i = (pos > iip.length() ? iip.length() : pos) - 1; i >= 0; --i) {
            if (commonAncestor != null && commonAncestor.equals(iip.getINode(i))) {
                return;
            }
            DirectoryWithQuotaFeature q = iip.getINode(i).asDirectory().getDirectoryWithQuotaFeature();
            if (q == null) continue;
            try {
                q.verifyQuota(deltas);
                continue;
            }
            catch (QuotaExceededException e) {
                List<INode> inodes = iip.getReadOnlyINodes();
                String path = FSDirectory.getFullPathName(inodes.toArray(new INode[inodes.size()]), i);
                e.setPathName(path);
                throw e;
            }
        }
    }

    void verifyMaxComponentLength(byte[] childName, String parentPath) throws FSLimitException.PathComponentTooLongException {
        if (this.maxComponentLength == 0) {
            return;
        }
        int length = childName.length;
        if (length > this.maxComponentLength) {
            FSLimitException.PathComponentTooLongException e = new FSLimitException.PathComponentTooLongException(this.maxComponentLength, length, parentPath, DFSUtil.bytes2String(childName));
            if (this.namesystem.isImageLoaded()) {
                throw e;
            }
            NameNode.LOG.error("ERROR in FSDirectory.verifyINodeName", (Throwable)((Object)e));
        }
    }

    void verifyMaxDirItems(INodeDirectory parent, String parentPath) throws FSLimitException.MaxDirectoryItemsExceededException, StorageException, TransactionContextException {
        if (this.maxDirItems <= 0) {
            return;
        }
        int count = parent.getChildrenNum();
        if (count >= this.maxDirItems) {
            FSLimitException.MaxDirectoryItemsExceededException e = new FSLimitException.MaxDirectoryItemsExceededException(this.maxDirItems, count);
            if (this.namesystem.isImageLoaded()) {
                e.setPathName(parentPath);
                throw e;
            }
            NameNode.LOG.error("FSDirectory.verifyMaxDirItems: " + e.getLocalizedMessage());
        }
    }

    protected <T extends INode> void verifyFsLimits(byte[] childName, String parentPath, INodeDirectory parent) throws FSLimitException, StorageException, TransactionContextException {
        boolean includeChildName = false;
        try {
            int count;
            int length;
            if (this.maxComponentLength != 0 && (length = childName.length) > this.maxComponentLength) {
                includeChildName = true;
                throw new FSLimitException.PathComponentTooLongException(this.maxComponentLength, length, parentPath, DFSUtil.bytes2String(childName));
            }
            if (this.maxDirItems != 0 && (count = parent.getChildrenNum()) >= this.maxDirItems) {
                throw new FSLimitException.MaxDirectoryItemsExceededException(this.maxDirItems, count);
            }
        }
        catch (FSLimitException e) {
            String badPath = parentPath;
            if (includeChildName) {
                badPath = badPath + "/" + childName;
            }
            e.setPathName(badPath);
            if (this.namesystem.isImageLoaded()) {
                throw e;
            }
            NameNode.LOG.error("FSDirectory.verifyFsLimits - " + e.getLocalizedMessage());
        }
    }

    INodesInPath addLastINode(INodesInPath existing, INode inode, boolean checkQuota) throws QuotaExceededException, StorageException, IOException {
        assert (existing.getLastINode() != null && existing.getLastINode().isDirectory());
        QuotaCounts counts = new QuotaCounts.Builder().build();
        if (this.isQuotaEnabled()) {
            inode.computeQuotaUsage(this.getBlockStoragePolicySuite(), counts);
        }
        return this.addLastINode(existing, inode, counts, checkQuota, true);
    }

    INodesInPath addLastINode(INodesInPath existing, INode inode, QuotaCounts counts, boolean checkQuota, boolean logMetadataEvent) throws QuotaExceededException, StorageException, IOException {
        assert (existing.getLastINode() != null && existing.getLastINode().isDirectory());
        int pos = existing.length();
        if (pos == 1 && existing.getINode(0) == this.getRootDir() && FSDirectory.isReservedName(inode)) {
            throw new HadoopIllegalArgumentException("File name \"" + inode.getLocalName() + "\" is reserved and cannot be created. If this is during upgrade change the name of the existing file or directory to another name before upgrading to the new release.");
        }
        INodeDirectory parent = existing.getINode(pos - 1).asDirectory();
        if (checkQuota) {
            String parentPath = existing.getPath();
            this.verifyMaxComponentLength(inode.getLocalNameBytes(), parentPath);
            this.verifyMaxDirItems(parent, parentPath);
        }
        this.updateCount(existing, pos, counts, checkQuota);
        boolean added = parent.addChild(inode, true, logMetadataEvent, this.getFSNamesystem().getNamenodeId());
        if (!added) {
            this.updateCountNoQuotaCheck(existing, pos, counts.negation());
            return null;
        }
        this.addToInodeMap(inode);
        INodesInPath iip = INodesInPath.append(existing, inode, inode.getLocalNameBytes());
        if (added && !inode.isDirectory()) {
            ArrayList<INode> pathInodes = new ArrayList<INode>(pos + 1);
            int i = 0;
            for (INode node : existing.getReadOnlyINodes()) {
                pathInodes.add(node);
                if (++i != pos) continue;
                pathInodes.add(inode);
                break;
            }
            String path = iip.getPath();
            Cache.getInstance().set(path, pathInodes);
        }
        return iip;
    }

    INodesInPath addLastINodeNoQuotaCheck(INodesInPath existing, INode inode, QuotaCounts counts) throws IOException {
        try {
            return this.addLastINode(existing, inode, counts, false, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn("FSDirectory.addChildNoQuotaCheck - unexpected", (Throwable)e);
            return null;
        }
    }

    long removeLastINode(INodesInPath iip) throws IOException {
        QuotaCounts counts = new QuotaCounts.Builder().build();
        if (this.isQuotaEnabled()) {
            iip.getLastINode().computeQuotaUsage(this.getBlockStoragePolicySuite(), counts);
        }
        return this.removeLastINode(iip, false, counts);
    }

    long removeLastINode(INodesInPath iip, boolean forRename, QuotaCounts counts) throws IOException {
        INode last = iip.getLastINode();
        INodeDirectory parent = iip.getINode(-2).asDirectory();
        if (!forRename) {
            if (!parent.removeChild(last)) {
                return -1L;
            }
        } else {
            parent.decreaseChildrenNum();
        }
        if (this.isQuotaEnabled()) {
            this.updateCountNoQuotaCheck(iip, iip.length() - 1, counts.negation());
            return counts.getNameSpace();
        }
        return 1L;
    }

    static String normalizePath(String src) {
        if (src.length() > 1 && src.endsWith("/")) {
            src = src.substring(0, src.length() - 1);
        }
        return src;
    }

    @VisibleForTesting
    public long getYieldCount() {
        return this.yieldCount;
    }

    void addYieldCount(long value) {
        this.yieldCount += value;
    }

    boolean truncate(INodesInPath iip, long newLength, INode.BlocksMapUpdateInfo collectedBlocks, long mtime, QuotaCounts delta) throws IOException {
        return this.unprotectedTruncate(iip, newLength, collectedBlocks, mtime, delta);
    }

    boolean unprotectedTruncate(INodesInPath iip, long newLength, INode.BlocksMapUpdateInfo collectedBlocks, long mtime, QuotaCounts delta) throws IOException {
        INodeFile file = iip.getLastINode().asFile();
        this.verifyQuotaForTruncate(iip, file, newLength, delta);
        long remainingLength = file.collectBlocksBeyondMax(newLength, collectedBlocks);
        file.setModificationTime(mtime);
        return remainingLength - newLength == 0L;
    }

    private void verifyQuotaForTruncate(INodesInPath iip, INodeFile file, long newLength, QuotaCounts delta) throws QuotaExceededException, TransactionContextException, StorageException {
        if (!this.getFSNamesystem().isImageLoaded()) {
            return;
        }
        long diff = file.computeQuotaDeltaForTruncate(newLength);
        short repl = file.getBlockReplication();
        delta.addStorageSpace(diff * (long)repl);
        BlockStoragePolicy policy = this.getBlockStoragePolicySuite().getPolicy(file.getStoragePolicyID());
        List types = policy.chooseStorageTypes(repl);
        for (StorageType t : types) {
            if (!t.supportTypeQuota()) continue;
            delta.addTypeSpace(t, diff);
        }
        if (diff > 0L) {
            FSDirectory.verifyQuota(iip, iip.length() - 1, delta, null);
        }
    }

    long totalInodes() throws IOException {
        LightWeightRequestHandler totalInodesHandler = new LightWeightRequestHandler(HDFSOperationType.TOTAL_FILES){

            public Object performTask() throws StorageException, IOException {
                INodeDataAccess da = (INodeDataAccess)HdfsStorageFactory.getDataAccess(INodeDataAccess.class);
                return da.countAll();
            }
        };
        return ((Integer)totalInodesHandler.handle()).intValue();
    }

    void reset() throws IOException {
        this.createRoot(this.namesystem.createFsOwnerPermissions(new FsPermission(493)), true);
        this.nameCache.reset();
    }

    boolean isInAnEZ(INodesInPath iip) throws UnresolvedLinkException, TransactionContextException, StorageException, InvalidProtocolBufferException {
        return this.ezManager.isInAnEZ(iip);
    }

    String getKeyName(INodesInPath iip) throws TransactionContextException, StorageException, InvalidProtocolBufferException {
        return this.ezManager.getKeyName(iip);
    }

    XAttr createEncryptionZone(String src, CipherSuite suite, CryptoProtocolVersion version, String keyName) throws IOException {
        return this.ezManager.createEncryptionZone(src, suite, version, keyName);
    }

    EncryptionZone getEZForPath(INodesInPath iip) throws IOException {
        return this.ezManager.getEZINodeForPath(iip);
    }

    BatchedRemoteIterator.BatchedListEntries<EncryptionZone> listEncryptionZones(long prevId) throws IOException {
        return this.ezManager.listEncryptionZones(prevId);
    }

    void setFileEncryptionInfo(String src, FileEncryptionInfo info) throws IOException {
        HdfsProtos.PerFileEncryptionInfoProto proto = PBHelper.convertPerFileEncInfo(info);
        byte[] protoBytes = proto.toByteArray();
        XAttr fileEncryptionAttr = XAttrHelper.buildXAttr("raw.hdfs.crypto.file.encryption.info", protoBytes);
        ArrayList xAttrs = Lists.newArrayListWithCapacity((int)1);
        xAttrs.add(fileEncryptionAttr);
        FSDirXAttrOp.unprotectedSetXAttrs(this, src, xAttrs, EnumSet.of(XAttrSetFlag.CREATE));
    }

    FileEncryptionInfo getFileEncryptionInfo(INode inode, INodesInPath iip) throws IOException {
        EncryptionZone encryptionZone;
        if (!inode.isFile()) {
            return null;
        }
        if (iip == null) {
            iip = this.getINodesInPath(inode.getFullPathName(), true);
        }
        if ((encryptionZone = this.getEZForPath(iip)) == null) {
            return null;
        }
        if ((encryptionZone.getPath() == null || encryptionZone.getPath().isEmpty()) && NameNode.LOG.isDebugEnabled()) {
            NameNode.LOG.debug("Encryption zone " + encryptionZone.getPath() + " does not have a valid path.");
        }
        CryptoProtocolVersion version = encryptionZone.getVersion();
        CipherSuite suite = encryptionZone.getSuite();
        String keyName = encryptionZone.getKeyName();
        XAttr fileXAttr = FSDirXAttrOp.unprotectedGetXAttrByName(inode, "raw.hdfs.crypto.file.encryption.info");
        if (fileXAttr == null) {
            NameNode.LOG.warn("Could not find encryption XAttr for file " + inode.getFullPathName() + " in encryption zone " + encryptionZone.getPath());
            return null;
        }
        try {
            HdfsProtos.PerFileEncryptionInfoProto fileProto = HdfsProtos.PerFileEncryptionInfoProto.parseFrom(fileXAttr.getValue());
            return PBHelper.convert(fileProto, suite, version, keyName);
        }
        catch (InvalidProtocolBufferException e) {
            throw new IOException("Could not parse file encryption info for inode " + inode, e);
        }
    }

    void cacheName(INode inode) throws StorageException, TransactionContextException {
        if (!inode.isFile()) {
            return;
        }
        ByteArray name = new ByteArray(inode.getLocalNameBytes());
        if ((name = this.nameCache.put(name)) != null) {
            inode.setLocalName(name.getBytes());
        }
    }

    static byte[][] getPathComponents(INode inode) throws StorageException, TransactionContextException {
        ArrayList<byte[]> components = new ArrayList<byte[]>();
        components.add(0, inode.getLocalNameBytes());
        while (inode.getParent() != null) {
            components.add(0, inode.getParent().getLocalNameBytes());
            inode = inode.getParent();
        }
        return (byte[][])components.toArray((T[])new byte[components.size()][]);
    }

    static byte[][] getPathComponentsForReservedPath(String src) {
        return !FSDirectory.isReservedName(src) ? (byte[][])null : INode.getPathComponents(src);
    }

    public static boolean isReservedName(INode inode) {
        return CHECK_RESERVED_FILE_NAMES && Arrays.equals(inode.getLocalNameBytes(), DOT_RESERVED);
    }

    public static boolean isReservedName(String src) {
        return src.startsWith("/.reserved/");
    }

    static boolean isReservedRawName(String src) {
        return src.startsWith("/.reserved/raw");
    }

    static String resolvePath(String src, byte[][] pathComponents, FSDirectory fsd) throws FileNotFoundException, IOException {
        int nComponents;
        int n = nComponents = pathComponents == null ? 0 : pathComponents.length;
        if (nComponents <= 2) {
            return src;
        }
        if (!Arrays.equals(DOT_RESERVED, pathComponents[1])) {
            return src;
        }
        if (Arrays.equals(DOT_INODES, pathComponents[2])) {
            if (nComponents > 3) {
                return FSDirectory.resolveDotInodesPath(src, pathComponents, fsd);
            }
            return src;
        }
        if (Arrays.equals(RAW, pathComponents[2])) {
            if (nComponents == 3) {
                return "/";
            }
            return FSDirectory.constructRemainingPath("", pathComponents, 3);
        }
        return src;
    }

    private static String resolveDotInodesPath(String src, byte[][] pathComponents, FSDirectory fsd) throws FileNotFoundException, IOException {
        int id;
        String inodeId = DFSUtil.bytes2String(pathComponents[3]);
        try {
            id = Integer.valueOf(inodeId);
        }
        catch (NumberFormatException e) {
            throw new FileNotFoundException("Invalid inode path: " + src);
        }
        if ((long)id == 1L && pathComponents.length == 4) {
            return "/";
        }
        if (pathComponents.length > 4 && DFSUtil.bytes2String(pathComponents[4]).equals("..")) {
            INode parent = fsd.getParent(id, src);
            if (parent == null || parent.getId() == 1L) {
                return "/";
            }
            return fsd.getFullPathName(parent.getId(), src);
        }
        String path = "";
        if ((long)id != 1L) {
            path = fsd.getFullPathName(id, src);
        }
        return FSDirectory.constructRemainingPath(path, pathComponents, 4);
    }

    private static String constructRemainingPath(String pathPrefix, byte[][] pathComponents, int startAt) {
        StringBuilder path = new StringBuilder(pathPrefix);
        for (int i = startAt; i < pathComponents.length; ++i) {
            path.append("/").append(DFSUtil.bytes2String(pathComponents[i]));
        }
        if (NameNode.LOG.isDebugEnabled()) {
            NameNode.LOG.debug("Resolved path is " + path);
        }
        return path.toString();
    }

    String getFullPathName(final long id, final String src) throws IOException {
        HopsTransactionalRequestHandler getFullPathNameHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_INODE){
            INodeIdentifier inodeIdentifier;

            @Override
            public void setUp() throws StorageException {
                this.inodeIdentifier = INodeUtil.resolveINodeFromId(id);
            }

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.READ_COMMITTED, this.inodeIdentifier, true));
            }

            public Object performTask() throws IOException {
                INode inode = (INode)EntityManager.find((FinderType)INode.Finder.ByINodeIdFTIS, (Object[])new Object[]{id});
                if (inode == null) {
                    throw new FileNotFoundException("File for given inode path does not exist: " + src);
                }
                return inode.getFullPathName();
            }
        };
        return (String)getFullPathNameHandler.handle();
    }

    INode getParent(final long id, final String src) throws IOException {
        HopsTransactionalRequestHandler getParentHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_INODE){
            INodeIdentifier inodeIdentifier;

            @Override
            public void setUp() throws StorageException {
                this.inodeIdentifier = INodeUtil.resolveINodeFromId(id);
            }

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.READ_COMMITTED, this.inodeIdentifier, true));
            }

            public Object performTask() throws IOException {
                INode inode = (INode)EntityManager.find((FinderType)INode.Finder.ByINodeIdFTIS, (Object[])new Object[]{id});
                if (inode == null) {
                    throw new FileNotFoundException("File for given inode path does not exist: " + src);
                }
                return inode.getParent();
            }
        };
        return (INode)getParentHandler.handle();
    }

    public final void addToInodeMap(INode inode) throws TransactionContextException, StorageException {
        XAttrFeature xaf;
        if (inode instanceof INodeWithAdditionalFields && !inode.isSymlink() && (xaf = inode.getXAttrFeature()) != null) {
            ImmutableList<XAttr> xattrs = xaf.getXAttrs();
            for (XAttr xattr : xattrs) {
                String xaName = XAttrHelper.getPrefixName(xattr);
                if (!"raw.hdfs.crypto.encryption.zone".equals(xaName)) continue;
                try {
                    HdfsProtos.ZoneEncryptionInfoProto ezProto = HdfsProtos.ZoneEncryptionInfoProto.parseFrom(xattr.getValue());
                    this.ezManager.unprotectedAddEncryptionZone(inode.getId(), PBHelper.convert(ezProto.getSuite()), PBHelper.convert(ezProto.getCryptoProtocolVersion()), ezProto.getKeyName());
                }
                catch (InvalidProtocolBufferException e) {
                    NameNode.LOG.warn("Error parsing protocol buffer of EZ XAttr " + xattr.getName());
                }
            }
        }
    }

    public final void removeFromInodeMap(List<? extends INode> inodes) throws IOException {
        if (inodes != null) {
            for (INode iNode : inodes) {
                if (iNode == null) continue;
                iNode.remove();
                if (!(iNode instanceof INodeWithAdditionalFields)) continue;
                this.ezManager.removeEncryptionZone(iNode.getId());
            }
        }
    }

    INode getInodeTX(final long id) throws IOException {
        HopsTransactionalRequestHandler getInodeHandler = new HopsTransactionalRequestHandler(HDFSOperationType.GET_INODE){
            INodeIdentifier inodeIdentifier;

            @Override
            public void setUp() throws StorageException {
                this.inodeIdentifier = INodeUtil.resolveINodeFromId(id);
            }

            public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.READ_COMMITTED, this.inodeIdentifier, true));
            }

            public Object performTask() throws IOException {
                return FSDirectory.this.getInode(id);
            }
        };
        return (INode)getInodeHandler.handle();
    }

    INode getInode(long id) throws IOException {
        return (INode)EntityManager.find((FinderType)INode.Finder.ByINodeIdFTIS, (Object[])new Object[]{id});
    }

    static INode resolveLastINode(INodesInPath iip) throws FileNotFoundException {
        INode inode = iip.getLastINode();
        if (inode == null) {
            throw new FileNotFoundException("cannot find " + iip.getPath());
        }
        return inode;
    }

    INodesInPath getExistingPathINodes(byte[][] components) throws UnresolvedLinkException, StorageException, TransactionContextException {
        return INodesInPath.resolve(this.getRootDir(), components, false);
    }

    public INodesInPath getINodesInPath4Write(String src) throws UnresolvedLinkException, StorageException, TransactionContextException {
        return this.getINodesInPath4Write(src, true);
    }

    public INode getINode4Write(String src) throws UnresolvedLinkException, StorageException, TransactionContextException {
        return this.getINodesInPath4Write(src, true).getLastINode();
    }

    public INodesInPath getINodesInPath(String path, boolean resolveLink) throws UnresolvedLinkException, StorageException, TransactionContextException {
        byte[][] components = INode.getPathComponents(path);
        return INodesInPath.resolve(this.getRootDir(), components, resolveLink);
    }

    INode getINode(String path, boolean resolveLink) throws UnresolvedLinkException, StorageException, TransactionContextException {
        return this.getINodesInPath(path, resolveLink).getLastINode();
    }

    public INode getINode(String src) throws UnresolvedLinkException, StorageException, TransactionContextException {
        return this.getINode(src, true);
    }

    INodesInPath getINodesInPath4Write(String src, boolean resolveLink) throws UnresolvedLinkException, StorageException, TransactionContextException {
        byte[][] components = INode.getPathComponents(src);
        INodesInPath inodesInPath = INodesInPath.resolve(this.getRootDir(), components, resolveLink);
        return inodesInPath;
    }

    public INodeDirectory getRootDir() throws StorageException, TransactionContextException {
        return INodeDirectory.getRootDir();
    }

    public boolean isQuotaEnabled() {
        return this.quotaEnabled;
    }

    public INodeDirectory createRoot(final PermissionStatus ps, final boolean overwrite) throws IOException {
        LightWeightRequestHandler addRootINode = new LightWeightRequestHandler(HDFSOperationType.SET_ROOT){

            public Object performTask() throws IOException {
                INodeDirectory newRootINode = null;
                INodeDataAccess da = (INodeDataAccess)HdfsStorageFactory.getDataAccess(INodeDataAccess.class);
                INodeDirectory rootInode = (INodeDirectory)da.findInodeByNameParentIdAndPartitionIdPK("", 0L, INodeDirectory.getRootDirPartitionKey());
                if (rootInode == null || overwrite) {
                    newRootINode = INodeDirectory.createRootDir(ps);
                    DirectoryWithQuotaFeature quotaFeature = new DirectoryWithQuotaFeature.Builder(newRootINode.getId()).nameSpaceQuota(Long.MAX_VALUE).storageSpaceQuota(-1L).build();
                    DirectoryWithQuotaFeatureDataAccess ida = (DirectoryWithQuotaFeatureDataAccess)HdfsStorageFactory.getDataAccess(DirectoryWithQuotaFeatureDataAccess.class);
                    newRootINode.addDirectoryWithQuotaFeature(quotaFeature);
                    ArrayList<INodeDirectory> newINodes = new ArrayList<INodeDirectory>();
                    newINodes.add(newRootINode);
                    da.prepare(INode.EMPTY_LIST, newINodes, INode.EMPTY_LIST);
                    ArrayList<DirectoryWithQuotaFeature> attrList = new ArrayList<DirectoryWithQuotaFeature>();
                    attrList.add(quotaFeature);
                    ida.prepare(attrList, null);
                    LOG.info("Added new root inode");
                }
                return newRootINode;
            }
        };
        return (INodeDirectory)addRootINode.handle();
    }

    public boolean hasChildren(final long parentId, final boolean areChildrenRandomlyPartitioned) throws IOException {
        LightWeightRequestHandler hasChildrenHandler = new LightWeightRequestHandler(HDFSOperationType.HAS_CHILDREN){

            public Object performTask() throws IOException {
                INodeDataAccess ida = (INodeDataAccess)HdfsStorageFactory.getDataAccess(INodeDataAccess.class);
                return ida.hasChildren(parentId, areChildrenRandomlyPartitioned);
            }
        };
        return (Boolean)hasChildrenHandler.handle();
    }

    FSPermissionChecker getPermissionChecker() throws AccessControlException {
        try {
            return new FSPermissionChecker(this.fsOwnerShortUserName, this.supergroup, NameNode.getRemoteUser());
        }
        catch (IOException ioe) {
            throw new AccessControlException((Throwable)ioe);
        }
    }

    void checkOwner(FSPermissionChecker pc, INodesInPath iip) throws AccessControlException, IOException {
        this.checkPermission(pc, iip, true, null, null, null, null);
    }

    void checkPathAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException, IOException {
        this.checkPermission(pc, iip, false, null, null, access, null);
    }

    void checkParentAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException, IOException {
        this.checkPermission(pc, iip, false, null, access, null, null);
    }

    void checkAncestorAccess(FSPermissionChecker pc, INodesInPath iip, FsAction access) throws AccessControlException, IOException {
        this.checkPermission(pc, iip, false, access, null, null, null);
    }

    void checkTraverse(FSPermissionChecker pc, INodesInPath iip) throws AccessControlException, IOException {
        this.checkPermission(pc, iip, false, null, null, null, null);
    }

    void checkPermission(FSPermissionChecker pc, INodesInPath iip, boolean doCheckOwner, FsAction ancestorAccess, FsAction parentAccess, FsAction access, FsAction subAccess) throws AccessControlException, UnresolvedLinkException, IOException {
        this.checkPermission(pc, iip, doCheckOwner, ancestorAccess, parentAccess, access, subAccess, false);
    }

    void checkPermission(FSPermissionChecker pc, INodesInPath iip, boolean doCheckOwner, FsAction ancestorAccess, FsAction parentAccess, FsAction access, FsAction subAccess, boolean ignoreEmptyDir) throws AccessControlException, UnresolvedLinkException, TransactionContextException, IOException {
        if (!pc.isSuperUser()) {
            pc.checkPermission(iip, doCheckOwner, ancestorAccess, parentAccess, access, subAccess, ignoreEmptyDir);
        }
    }

    HdfsFileStatus getAuditFileInfo(INodesInPath iip) throws IOException {
        return this.namesystem.isAuditEnabled() && this.namesystem.isExternalInvocation() ? FSDirStatAndListingOp.getFileInfo(this, iip, false, false, false, false) : null;
    }

    void verifyParentDir(INodesInPath iip, String src) throws FileNotFoundException, ParentNotDirectoryException, StorageException, TransactionContextException {
        Path parent = new Path(src).getParent();
        if (parent != null) {
            INode parentNode = iip.getINode(-2);
            if (parentNode == null) {
                throw new FileNotFoundException("Parent directory doesn't exist: " + parent);
            }
            if (!parentNode.isDirectory() && !parentNode.isSymlink()) {
                throw new ParentNotDirectoryException("Parent path is not a directory: " + parent);
            }
        }
    }
}

