/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import io.hops.TestUtil;
import java.io.IOException;
import java.util.EnumSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.DSQuotaExceededException;
import org.apache.hadoop.hdfs.protocol.QuotaByStorageTypeExceededException;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.QuotaCounts;
import org.apache.hadoop.hdfs.server.namenode.TestLeaseManager;
import org.apache.hadoop.ipc.RemoteException;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestDiskspaceQuotaUpdate {
    private static final int BLOCKSIZE = 131072;
    private static final short REPLICATION = 1;
    static final long seed = 0L;
    private static final Path dir = new Path("/TestQuotaUpdate");
    private Configuration conf;
    private MiniDFSCluster cluster;
    private FSDirectory fsdir;
    private DistributedFileSystem dfs;
    private int leaseCreationLockRows;

    @Before
    public void setUp() throws Exception {
        this.conf = new Configuration();
        this.conf.setLong("dfs.blocksize", 131072L);
        this.conf.setInt("dfs.db.file.max.size", 0);
        this.leaseCreationLockRows = this.conf.getInt("dfs.lease.creation.locks.count.key", 1000);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(1).build();
        this.cluster.waitActive();
        this.fsdir = this.cluster.getNamesystem().getFSDirectory();
        this.dfs = this.cluster.getFileSystem();
    }

    @After
    public void tearDown() throws Exception {
        if (this.cluster != null) {
            this.cluster.shutdown();
        }
    }

    @Test(timeout=60000L)
    public void testQuotaUpdateWithFileCreate() throws Exception {
        Path foo = new Path(dir, "foo");
        Path createdFile = new Path(foo, "created_file.data");
        this.dfs.mkdirs(foo);
        this.dfs.setQuota(foo, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        long fileLen = 327680L;
        DFSTestUtil.createFile((FileSystem)this.dfs, createdFile, 8192, fileLen, 131072L, (short)1, 0L);
        Thread.sleep(1000L);
        QuotaCounts cnt = this.getSpaceConsumed(foo);
        Assert.assertEquals((long)2L, (long)cnt.getNameSpace());
        Assert.assertEquals((long)(fileLen * 1L), (long)cnt.getStorageSpace());
    }

    @Test
    public void testUpdateQuotaForAppend() throws Exception {
        Path foo = new Path(dir, "foo");
        Path bar = new Path(foo, "bar");
        long currentFileLen = 131072L;
        DFSTestUtil.createFile((FileSystem)this.dfs, bar, currentFileLen, (short)1, 0L);
        this.dfs.setQuota(foo, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        DFSTestUtil.appendFile((FileSystem)this.dfs, bar, 65536);
        currentFileLen += 65536L;
        Thread.sleep(1000L);
        QuotaCounts quota = this.getSpaceConsumed(foo);
        long ns = quota.getNameSpace();
        long ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)(currentFileLen * 1L), (long)ds);
        ContentSummary c = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)c.getSpaceConsumed(), (long)ds);
        DFSTestUtil.appendFile((FileSystem)this.dfs, bar, 131072);
        currentFileLen += 131072L;
        Thread.sleep(1000L);
        quota = this.getSpaceConsumed(foo);
        ns = quota.getNameSpace();
        ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)(currentFileLen * 1L), (long)ds);
        c = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)c.getSpaceConsumed(), (long)ds);
        DFSTestUtil.appendFile((FileSystem)this.dfs, bar, 409600);
        currentFileLen += 409600L;
        Thread.sleep(1000L);
        quota = this.getSpaceConsumed(foo);
        ns = quota.getNameSpace();
        ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)(currentFileLen * 1L), (long)ds);
        c = this.dfs.getContentSummary(foo);
        Assert.assertEquals((long)c.getSpaceConsumed(), (long)ds);
    }

    @Test(timeout=60000L)
    public void testUpdateQuotaForFSync() throws Exception {
        Path foo = new Path("/foo");
        Path bar = new Path(foo, "bar");
        DFSTestUtil.createFile((FileSystem)this.dfs, bar, 131072L, (short)1, 0L);
        this.dfs.setQuota(foo, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        FSDataOutputStream out = this.dfs.append(bar);
        out.write(new byte[32768]);
        ((DFSOutputStream)out.getWrappedStream()).hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        QuotaCounts quota = this.getSpaceConsumed(foo, false);
        long ns = quota.getNameSpace();
        long ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)163840L, (long)ds);
        Assert.assertEquals((long)262144L, (long)this.dfs.getLastUpdatedContentSummary(foo).getSpaceConsumed());
        out.write(new byte[32768]);
        out.close();
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        quota = this.getSpaceConsumed(foo);
        ns = quota.getNameSpace();
        ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)196608L, (long)ds);
        DFSTestUtil.appendFile((FileSystem)this.dfs, bar, 131072);
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        quota = this.getSpaceConsumed(foo);
        ns = quota.getNameSpace();
        ds = quota.getStorageSpace();
        Assert.assertEquals((long)2L, (long)ns);
        Assert.assertEquals((long)327680L, (long)ds);
    }

    @Test
    public void testAppendOverStorageQuota() throws Exception {
        Path dir = new Path("/TestAppendOverQuota");
        Path file = new Path(dir, "file");
        this.dfs.mkdirs(dir);
        DFSTestUtil.createFile((FileSystem)this.dfs, file, 65536L, (short)1, 0L);
        this.dfs.setQuota(dir, 0x7FFFFFFFFFFFFFFEL, 1L);
        long spaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        try {
            DFSTestUtil.appendFile((FileSystem)this.dfs, file, 131072);
            Assert.fail((String)"append didn't fail");
        }
        catch (DSQuotaExceededException dSQuotaExceededException) {
            // empty catch block
        }
        INodeFile inode = this.getINodeFile(file);
        Assert.assertNotNull((Object)inode);
        Assert.assertFalse((String)"should not be UC", (boolean)inode.isUnderConstruction());
        Assert.assertNull((String)"should not have a lease", (Object)TestLeaseManager.getLeaseByPath(this.cluster.getNamesystem().getLeaseManager(), file.toString()));
        long newSpaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        Assert.assertEquals((long)spaceUsed, (long)newSpaceUsed);
        this.dfs.recoverLease(file);
        this.cluster.restartNameNodes();
    }

    @Test(timeout=60000L)
    public void testAppendOverTypeQuota() throws Exception {
        Path dir = new Path("/TestAppendOverTypeQuota");
        Path file = new Path(dir, "file");
        this.dfs.mkdirs(dir);
        this.dfs.setStoragePolicy(dir, "ONE_SSD");
        DFSTestUtil.createFile((FileSystem)this.dfs, file, 65536L, (short)1, 0L);
        this.dfs.setQuotaByStorageType(dir, StorageType.SSD, 1L);
        long spaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        try {
            DFSTestUtil.appendFile((FileSystem)this.dfs, file, 131072);
            Assert.fail((String)"append didn't fail");
        }
        catch (QuotaByStorageTypeExceededException quotaByStorageTypeExceededException) {
            // empty catch block
        }
        INodeFile inode = this.getINodeFile(file);
        Assert.assertNotNull((Object)inode);
        Assert.assertFalse((String)"should not be UC", (boolean)inode.isUnderConstruction());
        Assert.assertNull((String)"should not have a lease", (Object)TestLeaseManager.getLeaseByPath(this.cluster.getNamesystem().getLeaseManager(), file.toString()));
        long newSpaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        Assert.assertEquals((long)spaceUsed, (long)newSpaceUsed);
        this.dfs.recoverLease(file);
        this.cluster.restartNameNodes();
    }

    @Test(timeout=60000L)
    public void testTruncateOverQuota() throws Exception {
        Path dir = new Path("/TestTruncateOverquota");
        Path file = new Path(dir, "file");
        this.dfs.mkdirs(dir);
        DFSTestUtil.createFile((FileSystem)this.dfs, file, 65536L, (short)1, 0L);
        this.dfs.setQuota(dir, 0x7FFFFFFFFFFFFFFEL, 1L);
        long spaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        try {
            this.dfs.truncate(file, 65535L);
            Assert.fail((String)"truncate didn't fail");
        }
        catch (RemoteException e) {
            Assert.assertTrue((boolean)e.getClassName().contains("DSQuotaExceededException"));
        }
        INodeFile inode = this.getINodeFile(file);
        Assert.assertNotNull((Object)inode);
        Assert.assertFalse((String)"should not be UC", (boolean)inode.isUnderConstruction());
        Assert.assertNull((String)"should not have a lease", (Object)TestLeaseManager.getLeaseByPath(this.cluster.getNamesystem().getLeaseManager(), file.toString()));
        long newSpaceUsed = this.getSpaceConsumed(dir).getStorageSpace();
        Assert.assertEquals((long)spaceUsed, (long)newSpaceUsed);
        this.dfs.recoverLease(file);
        this.cluster.restartNameNodes();
    }

    private QuotaCounts getSpaceConsumed(Path foo) throws IOException, InterruptedException {
        return this.getSpaceConsumed(foo, true);
    }

    private QuotaCounts getSpaceConsumed(Path foo, boolean checkLastUpdated) throws IOException, InterruptedException {
        ContentSummary c = DFSTestUtil.getContentSummary(this.dfs, foo, checkLastUpdated);
        QuotaCounts counts = new QuotaCounts.Builder().nameSpace(c.getFileAndDirectoryCount()).storageSpace(c.getSpaceConsumed()).build();
        return counts;
    }

    private INodeFile getINodeFile(Path foo) throws IOException {
        return (INodeFile)TestUtil.getINode(this.cluster.getNameNode(), foo);
    }
}

