/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.lib.input;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.mapred.SplitLocationInfo;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.lib.input.FileSplit;
import org.apache.hadoop.mapreduce.lib.input.InvalidInputException;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@RunWith(value=Parameterized.class)
public class TestFileInputFormat {
    private static final Logger LOG = LoggerFactory.getLogger(TestFileInputFormat.class);
    private static String testTmpDir = System.getProperty("test.build.data", "/tmp");
    private static final Path TEST_ROOT_DIR = new Path(testTmpDir, "TestFIF");
    private static FileSystem localFs;
    private int numThreads;

    public TestFileInputFormat(int numThreads) {
        this.numThreads = numThreads;
        LOG.info("Running with numThreads: " + numThreads);
    }

    @Parameterized.Parameters
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{{1}, {5}};
        return Arrays.asList(data);
    }

    @Before
    public void setup() throws IOException {
        LOG.info("Using Test Dir: " + TEST_ROOT_DIR);
        localFs = FileSystem.getLocal((Configuration)new Configuration());
        localFs.delete(TEST_ROOT_DIR, true);
        localFs.mkdirs(TEST_ROOT_DIR);
    }

    @After
    public void cleanup() throws IOException {
        localFs.delete(TEST_ROOT_DIR, true);
    }

    @Test
    public void testNumInputFilesRecursively() throws Exception {
        Configuration conf = this.getConfiguration();
        conf.set("mapreduce.input.fileinputformat.input.dir.recursive", "true");
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assert.assertEquals((String)"Input splits are not correct", (long)3L, (long)splits.size());
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2/file2", "test:/a1/a2/file3", "test:/a1/file1"}), splits);
        conf = this.getConfiguration();
        conf.set("mapred.input.dir.recursive", "true");
        job = Job.getInstance((Configuration)conf);
        splits = fileInputFormat.getSplits((JobContext)job);
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2/file2", "test:/a1/a2/file3", "test:/a1/file1"}), splits);
    }

    @Test
    public void testNumInputFilesWithoutRecursively() throws Exception {
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assert.assertEquals((String)"Input splits are not correct", (long)2L, (long)splits.size());
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2", "test:/a1/file1"}), splits);
    }

    @Test
    public void testNumInputFilesIgnoreDirs() throws Exception {
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.nonrecursive.ignore.subdirs", true);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assert.assertEquals((String)"Input splits are not correct", (long)1L, (long)splits.size());
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/file1"}), splits);
    }

    @Test
    public void testListLocatedStatus() throws Exception {
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        conf.setBoolean("fs.test.impl.disable.cache", false);
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1/a2");
        MockFileSystem mockFs = (MockFileSystem)new Path("test:///").getFileSystem(conf);
        Assert.assertEquals((String)"listLocatedStatus already called", (long)0L, (long)mockFs.numListLocatedStatusCalls);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assert.assertEquals((String)"Input splits are not correct", (long)2L, (long)splits.size());
        Assert.assertEquals((String)"listLocatedStatuss calls", (long)1L, (long)mockFs.numListLocatedStatusCalls);
        FileSystem.closeAll();
    }

    @Test
    public void testSplitLocationInfo() throws Exception {
        Configuration conf = this.getConfiguration();
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1/a2");
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        String[] locations = ((InputSplit)splits.get(0)).getLocations();
        Assert.assertEquals((long)2L, (long)locations.length);
        SplitLocationInfo[] locationInfo = ((InputSplit)splits.get(0)).getLocationInfo();
        Assert.assertEquals((long)2L, (long)locationInfo.length);
        SplitLocationInfo localhostInfo = locations[0].equals("localhost") ? locationInfo[0] : locationInfo[1];
        SplitLocationInfo otherhostInfo = locations[0].equals("otherhost") ? locationInfo[0] : locationInfo[1];
        Assert.assertTrue((boolean)localhostInfo.isOnDisk());
        Assert.assertTrue((boolean)localhostInfo.isInMemory());
        Assert.assertTrue((boolean)otherhostInfo.isOnDisk());
        Assert.assertFalse((boolean)otherhostInfo.isInMemory());
    }

    @Test
    public void testListStatusSimple() throws IOException {
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestSimple(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @Test
    public void testListStatusNestedRecursive() throws IOException {
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestNestedRecursive(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @Test
    public void testListStatusNestedNonRecursive() throws IOException {
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestNestedNonRecursive(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @Test
    public void testListStatusErrorOnNonExistantDir() throws IOException {
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        TestFileInputFormat.configureTestErrorOnNonExistantDir(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        try {
            fif.listStatus((JobContext)job);
            Assert.fail((String)"Expecting an IOException for a missing Input path");
        }
        catch (IOException e) {
            Path expectedExceptionPath = new Path(TEST_ROOT_DIR, "input2");
            expectedExceptionPath = localFs.makeQualified(expectedExceptionPath);
            Assert.assertTrue((boolean)(e instanceof InvalidInputException));
            Assert.assertEquals((Object)("Input path does not exist: " + expectedExceptionPath.toString()), (Object)e.getMessage());
        }
    }

    public static List<Path> configureTestSimple(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        Path base2 = new Path(TEST_ROOT_DIR, "input2");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1) + "," + localFs.makeQualified(base2));
        localFs.mkdirs(base1);
        localFs.mkdirs(base2);
        Path in1File1 = new Path(base1, "file1");
        Path in1File2 = new Path(base1, "file2");
        localFs.createNewFile(in1File1);
        localFs.createNewFile(in1File2);
        Path in2File1 = new Path(base2, "file1");
        Path in2File2 = new Path(base2, "file2");
        localFs.createNewFile(in2File1);
        localFs.createNewFile(in2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{in1File1, in1File2, in2File1, in2File2});
        return expectedPaths;
    }

    public static List<Path> configureTestNestedRecursive(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1).toString());
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", true);
        localFs.mkdirs(base1);
        Path inDir1 = new Path(base1, "dir1");
        Path inDir2 = new Path(base1, "dir2");
        Path inFile1 = new Path(base1, "file1");
        Path dir1File1 = new Path(inDir1, "file1");
        Path dir1File2 = new Path(inDir1, "file2");
        Path dir2File1 = new Path(inDir2, "file1");
        Path dir2File2 = new Path(inDir2, "file2");
        localFs.mkdirs(inDir1);
        localFs.mkdirs(inDir2);
        localFs.createNewFile(inFile1);
        localFs.createNewFile(dir1File1);
        localFs.createNewFile(dir1File2);
        localFs.createNewFile(dir2File1);
        localFs.createNewFile(dir2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{inFile1, dir1File1, dir1File2, dir2File1, dir2File2});
        return expectedPaths;
    }

    public static List<Path> configureTestNestedNonRecursive(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1).toString());
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", false);
        localFs.mkdirs(base1);
        Path inDir1 = new Path(base1, "dir1");
        Path inDir2 = new Path(base1, "dir2");
        Path inFile1 = new Path(base1, "file1");
        Path dir1File1 = new Path(inDir1, "file1");
        Path dir1File2 = new Path(inDir1, "file2");
        Path dir2File1 = new Path(inDir2, "file1");
        Path dir2File2 = new Path(inDir2, "file2");
        localFs.mkdirs(inDir1);
        localFs.mkdirs(inDir2);
        localFs.createNewFile(inFile1);
        localFs.createNewFile(dir1File1);
        localFs.createNewFile(dir1File2);
        localFs.createNewFile(dir2File1);
        localFs.createNewFile(dir2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{inFile1, inDir1, inDir2});
        return expectedPaths;
    }

    public static List<Path> configureTestErrorOnNonExistantDir(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        Path base2 = new Path(TEST_ROOT_DIR, "input2");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1) + "," + localFs.makeQualified(base2));
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", true);
        localFs.mkdirs(base1);
        Path inFile1 = new Path(base1, "file1");
        Path inFile2 = new Path(base1, "file2");
        localFs.createNewFile(inFile1);
        localFs.createNewFile(inFile2);
        ArrayList expectedPaths = Lists.newArrayList();
        return expectedPaths;
    }

    public static void verifyFileStatuses(List<Path> expectedPaths, List<FileStatus> fetchedStatuses, final FileSystem localFs) {
        Assert.assertEquals((long)expectedPaths.size(), (long)fetchedStatuses.size());
        Iterable fqExpectedPaths = Iterables.transform(expectedPaths, (Function)new Function<Path, Path>(){

            public Path apply(Path input) {
                return localFs.makeQualified(input);
            }
        });
        HashSet expectedPathSet = Sets.newHashSet((Iterable)fqExpectedPaths);
        for (FileStatus fileStatus : fetchedStatuses) {
            if (expectedPathSet.remove(localFs.makeQualified(fileStatus.getPath()))) continue;
            Assert.fail((String)("Found extra fetched status: " + fileStatus.getPath()));
        }
        Assert.assertEquals((String)("Not all expectedPaths matched: " + ((Object)expectedPathSet).toString()), (long)0L, (long)expectedPathSet.size());
    }

    private void verifySplits(List<String> expected, List<InputSplit> splits) {
        Iterable pathsFromSplits = Iterables.transform(splits, (Function)new Function<InputSplit, String>(){

            public String apply(@Nullable InputSplit input) {
                return ((FileSplit)input).getPath().toString();
            }
        });
        HashSet expectedSet = Sets.newHashSet(expected);
        for (String splitPathString : pathsFromSplits) {
            if (expectedSet.remove(splitPathString)) continue;
            Assert.fail((String)("Found extra split: " + splitPathString));
        }
        Assert.assertEquals((String)("Not all expectedPaths matched: " + ((Object)expectedSet).toString()), (long)0L, (long)expectedSet.size());
    }

    private Configuration getConfiguration() {
        Configuration conf = new Configuration();
        conf.set("fs.test.impl.disable.cache", "true");
        conf.setClass("fs.test.impl", MockFileSystem.class, FileSystem.class);
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1");
        return conf;
    }

    static class MockFileSystem
    extends RawLocalFileSystem {
        int numListLocatedStatusCalls = 0;

        MockFileSystem() {
        }

        public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
            if (f.toString().equals("test:/a1")) {
                return new FileStatus[]{new FileStatus(0L, true, 1, 150L, 150L, new Path("test:/a1/a2")), new FileStatus(10L, false, 1, 150L, 150L, new Path("test:/a1/file1"))};
            }
            if (f.toString().equals("test:/a1/a2")) {
                return new FileStatus[]{new FileStatus(10L, false, 1, 150L, 150L, new Path("test:/a1/a2/file2")), new FileStatus(10L, false, 1, 151L, 150L, new Path("test:/a1/a2/file3"))};
            }
            return new FileStatus[0];
        }

        public FileStatus[] globStatus(Path pathPattern, PathFilter filter) throws IOException {
            return new FileStatus[]{new FileStatus(10L, true, 1, 150L, 150L, pathPattern)};
        }

        public FileStatus[] listStatus(Path f, PathFilter filter) throws FileNotFoundException, IOException {
            return this.listStatus(f);
        }

        public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
            return new BlockLocation[]{new BlockLocation(new String[]{"localhost:9866", "otherhost:9866"}, new String[]{"localhost", "otherhost"}, new String[]{"localhost"}, new String[0], 0L, len, false)};
        }

        protected RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f, PathFilter filter) throws FileNotFoundException, IOException {
            ++this.numListLocatedStatusCalls;
            return super.listLocatedStatus(f, filter);
        }
    }
}

