/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources;

import com.google.common.annotations.VisibleForTesting;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.privileged.PrivilegedOperation;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.privileged.PrivilegedOperationException;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.privileged.PrivilegedOperationExecutor;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.CGroupsHandler;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.ResourceHandlerException;
import org.apache.hadoop.yarn.util.Clock;
import org.apache.hadoop.yarn.util.SystemClock;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class CGroupsHandlerImpl
implements CGroupsHandler {
    private static final Log LOG = LogFactory.getLog(CGroupsHandlerImpl.class);
    private static final String MTAB_FILE = "/proc/mounts";
    private static final String CGROUPS_FSTYPE = "cgroup";
    private final String cGroupPrefix;
    private final boolean enableCGroupMount;
    private final String cGroupMountPath;
    private final long deleteCGroupTimeout;
    private final long deleteCGroupDelay;
    private Map<CGroupsHandler.CGroupController, String> controllerPaths;
    private final ReadWriteLock rwLock;
    private final PrivilegedOperationExecutor privilegedOperationExecutor;
    private final Clock clock;
    private static final Pattern MTAB_FILE_FORMAT = Pattern.compile("^[^\\s]+\\s([^\\s]+)\\s([^\\s]+)\\s([^\\s]+)\\s[^\\s]+\\s[^\\s]+$");

    public CGroupsHandlerImpl(Configuration conf, PrivilegedOperationExecutor privilegedOperationExecutor) throws ResourceHandlerException {
        this.cGroupPrefix = conf.get("yarn.nodemanager.linux-container-executor.cgroups.hierarchy", "/hadoop-yarn").replaceAll("^/", "").replaceAll("$/", "");
        this.enableCGroupMount = conf.getBoolean("yarn.nodemanager.linux-container-executor.cgroups.mount", false);
        this.cGroupMountPath = conf.get("yarn.nodemanager.linux-container-executor.cgroups.mount-path", null);
        this.deleteCGroupTimeout = conf.getLong("yarn.nodemanager.linux-container-executor.cgroups.delete-timeout-ms", 1000L);
        this.deleteCGroupDelay = conf.getLong("yarn.nodemanager.linux-container-executor.cgroups.delete-delay-ms", 20L);
        this.controllerPaths = new HashMap<CGroupsHandler.CGroupController, String>();
        this.rwLock = new ReentrantReadWriteLock();
        this.privilegedOperationExecutor = privilegedOperationExecutor;
        this.clock = new SystemClock();
        this.init();
    }

    private void init() throws ResourceHandlerException {
        this.initializeControllerPaths();
    }

    private String getControllerPath(CGroupsHandler.CGroupController controller) {
        try {
            this.rwLock.readLock().lock();
            String string = this.controllerPaths.get((Object)controller);
            return string;
        }
        finally {
            this.rwLock.readLock().unlock();
        }
    }

    private void initializeControllerPaths() throws ResourceHandlerException {
        if (this.enableCGroupMount) {
            LOG.info((Object)"CGroup controller mounting enabled.");
        } else {
            Map<CGroupsHandler.CGroupController, String> cPaths = CGroupsHandlerImpl.initializeControllerPathsFromMtab(MTAB_FILE, this.cGroupPrefix);
            try {
                this.rwLock.writeLock().lock();
                this.controllerPaths = cPaths;
            }
            finally {
                this.rwLock.writeLock().unlock();
            }
        }
    }

    @VisibleForTesting
    static Map<CGroupsHandler.CGroupController, String> initializeControllerPathsFromMtab(String mtab, String cGroupPrefix) throws ResourceHandlerException {
        try {
            Map<String, List<String>> parsedMtab = CGroupsHandlerImpl.parseMtab(mtab);
            HashMap<CGroupsHandler.CGroupController, String> ret = new HashMap<CGroupsHandler.CGroupController, String>();
            for (CGroupsHandler.CGroupController controller : CGroupsHandler.CGroupController.values()) {
                String name = controller.getName();
                String controllerPath = CGroupsHandlerImpl.findControllerInMtab(name, parsedMtab);
                if (controllerPath != null) {
                    File f = new File(controllerPath + "/" + cGroupPrefix);
                    if (FileUtil.canWrite((File)f)) {
                        ret.put(controller, controllerPath);
                        continue;
                    }
                    String error = new StringBuffer("Mount point Based on mtab file: ").append(mtab).append(". Controller mount point not writable for: ").append(name).toString();
                    LOG.error((Object)error);
                    throw new ResourceHandlerException(error);
                }
                LOG.warn((Object)("Controller not mounted but automount disabled: " + name));
            }
            return ret;
        }
        catch (IOException e) {
            LOG.warn((Object)("Failed to initialize controller paths! Exception: " + e));
            throw new ResourceHandlerException("Failed to initialize controller paths!");
        }
    }

    private static Map<String, List<String>> parseMtab(String mtab) throws IOException {
        HashMap<String, List<String>> ret = new HashMap<String, List<String>>();
        BufferedReader in = null;
        try {
            FileInputStream fis = new FileInputStream(new File(mtab));
            in = new BufferedReader(new InputStreamReader((InputStream)fis, "UTF-8"));
            String str = in.readLine();
            while (str != null) {
                Matcher m = MTAB_FILE_FORMAT.matcher(str);
                boolean mat = m.find();
                if (mat) {
                    String path = m.group(1);
                    String type = m.group(2);
                    String options = m.group(3);
                    if (type.equals(CGROUPS_FSTYPE)) {
                        List<String> value = Arrays.asList(options.split(","));
                        ret.put(path, value);
                    }
                }
                str = in.readLine();
            }
        }
        catch (IOException e) {
            try {
                throw new IOException("Error while reading " + mtab, e);
            }
            catch (Throwable throwable) {
                IOUtils.cleanup((Log)LOG, (Closeable[])new Closeable[]{in});
                throw throwable;
            }
        }
        IOUtils.cleanup((Log)LOG, (Closeable[])new Closeable[]{in});
        return ret;
    }

    private static String findControllerInMtab(String controller, Map<String, List<String>> entries) {
        for (Map.Entry<String, List<String>> e : entries.entrySet()) {
            if (!e.getValue().contains(controller)) continue;
            return e.getKey();
        }
        return null;
    }

    @Override
    public void mountCGroupController(CGroupsHandler.CGroupController controller) throws ResourceHandlerException {
        if (!this.enableCGroupMount) {
            LOG.warn((Object)("CGroup mounting is disabled - ignoring mount request for: " + controller.getName()));
            return;
        }
        String path = this.getControllerPath(controller);
        if (path == null) {
            try {
                this.rwLock.writeLock().lock();
                String hierarchy = this.cGroupPrefix;
                StringBuffer controllerPath = new StringBuffer().append(this.cGroupMountPath).append('/').append(controller.getName());
                StringBuffer cGroupKV = new StringBuffer().append(controller.getName()).append('=').append(controllerPath);
                PrivilegedOperation.OperationType opType = PrivilegedOperation.OperationType.MOUNT_CGROUPS;
                PrivilegedOperation op = new PrivilegedOperation(opType);
                op.appendArgs(hierarchy, cGroupKV.toString());
                LOG.info((Object)("Mounting controller " + controller.getName() + " at " + controllerPath));
                this.privilegedOperationExecutor.executePrivilegedOperation(op, false);
                this.controllerPaths.put(controller, controllerPath.toString());
                return;
            }
            catch (PrivilegedOperationException e) {
                LOG.error((Object)("Failed to mount controller: " + controller.getName()));
                throw new ResourceHandlerException("Failed to mount controller: " + controller.getName());
            }
            finally {
                this.rwLock.writeLock().unlock();
            }
        }
        LOG.info((Object)("CGroup controller already mounted at: " + path));
    }

    @Override
    public String getRelativePathForCGroup(String cGroupId) {
        return new StringBuffer(this.cGroupPrefix).append("/").append(cGroupId).toString();
    }

    @Override
    public String getPathForCGroup(CGroupsHandler.CGroupController controller, String cGroupId) {
        return new StringBuffer(this.getControllerPath(controller)).append('/').append(this.cGroupPrefix).append("/").append(cGroupId).toString();
    }

    @Override
    public String getPathForCGroupTasks(CGroupsHandler.CGroupController controller, String cGroupId) {
        return new StringBuffer(this.getPathForCGroup(controller, cGroupId)).append('/').append("tasks").toString();
    }

    @Override
    public String getPathForCGroupParam(CGroupsHandler.CGroupController controller, String cGroupId, String param) {
        return new StringBuffer(this.getPathForCGroup(controller, cGroupId)).append('/').append(controller.getName()).append('.').append(param).toString();
    }

    @Override
    public String createCGroup(CGroupsHandler.CGroupController controller, String cGroupId) throws ResourceHandlerException {
        String path = this.getPathForCGroup(controller, cGroupId);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("createCgroup: " + path));
        }
        if (!new File(path).mkdir()) {
            throw new ResourceHandlerException("Failed to create cgroup at " + path);
        }
        return path;
    }

    private void logLineFromTasksFile(File cgf) {
        if (LOG.isDebugEnabled()) {
            try (BufferedReader inl = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(cgf + "/tasks"), "UTF-8"));){
                String str = inl.readLine();
                if (str != null) {
                    LOG.debug((Object)("First line in cgroup tasks file: " + cgf + " " + str));
                }
            }
            catch (IOException e) {
                LOG.warn((Object)"Failed to read cgroup tasks file. ", (Throwable)e);
            }
        }
    }

    boolean checkAndDeleteCgroup(File cgf) throws InterruptedException {
        boolean deleted = false;
        try (FileInputStream in = new FileInputStream(cgf + "/tasks");){
            if (in.read() == -1) {
                Thread.sleep(this.deleteCGroupDelay);
                deleted = cgf.delete();
                if (!deleted) {
                    LOG.warn((Object)("Failed attempt to delete cgroup: " + cgf));
                }
            } else {
                this.logLineFromTasksFile(cgf);
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"Failed to read cgroup tasks file. ", (Throwable)e);
        }
        return deleted;
    }

    @Override
    public void deleteCGroup(CGroupsHandler.CGroupController controller, String cGroupId) throws ResourceHandlerException {
        boolean deleted = false;
        String cGroupPath = this.getPathForCGroup(controller, cGroupId);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("deleteCGroup: " + cGroupPath));
        }
        long start = this.clock.getTime();
        do {
            try {
                deleted = this.checkAndDeleteCgroup(new File(cGroupPath));
                if (deleted) continue;
                Thread.sleep(this.deleteCGroupDelay);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        } while (!deleted && this.clock.getTime() - start < this.deleteCGroupTimeout);
        if (!deleted) {
            LOG.warn((Object)("Unable to delete  " + cGroupPath + ", tried to delete for " + this.deleteCGroupTimeout + "ms"));
        }
    }

    @Override
    public void updateCGroupParam(CGroupsHandler.CGroupController controller, String cGroupId, String param, String value) throws ResourceHandlerException {
        String cGroupParamPath = this.getPathForCGroupParam(controller, cGroupId, param);
        PrintWriter pw = null;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("updateCGroupParam for path: " + cGroupParamPath + " with value " + value));
        }
        try {
            File file = new File(cGroupParamPath);
            OutputStreamWriter w = new OutputStreamWriter((OutputStream)new FileOutputStream(file), "UTF-8");
            pw = new PrintWriter(w);
            pw.write(value);
        }
        catch (IOException e) {
            throw new ResourceHandlerException(new StringBuffer("Unable to write to ").append(cGroupParamPath).append(" with value: ").append(value).toString(), e);
        }
        finally {
            if (pw != null) {
                boolean hasError = pw.checkError();
                pw.close();
                if (hasError) {
                    throw new ResourceHandlerException(new StringBuffer("Unable to write to ").append(cGroupParamPath).append(" with value: ").append(value).toString());
                }
                if (pw.checkError()) {
                    throw new ResourceHandlerException("Error while closing cgroup file " + cGroupParamPath);
                }
            }
        }
    }

    @Override
    public String getCGroupParam(CGroupsHandler.CGroupController controller, String cGroupId, String param) throws ResourceHandlerException {
        String cGroupParamPath = this.getPathForCGroupParam(controller, cGroupId, param);
        try {
            byte[] contents = Files.readAllBytes(Paths.get(cGroupParamPath, new String[0]));
            return new String(contents, "UTF-8").trim();
        }
        catch (IOException e) {
            throw new ResourceHandlerException("Unable to read from " + cGroupParamPath);
        }
    }
}

