/*
 * Decompiled with CFR 0.152.
 */
package io.hops.hopsworks.common.hdfs;

import io.hops.hopsworks.common.hdfs.Utils;
import io.hops.metadata.hdfs.entity.EncodingPolicy;
import io.hops.metadata.hdfs.entity.EncodingStatus;
import io.hops.metadata.hdfs.entity.MetaStatus;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.LastUpdatedContentSummary;
import org.apache.hadoop.security.UserGroupInformation;

public class DistributedFileSystemOps {
    private static final Logger logger = Logger.getLogger(DistributedFileSystemOps.class.getName());
    private static final long MB = 0x100000L;
    private final DistributedFileSystem dfs;
    private Configuration conf;
    private String hadoopConfDir;
    private final String effectiveUser;

    public DistributedFileSystemOps(UserGroupInformation ugi, Configuration conf, URI uri) {
        this.dfs = this.getDfs(ugi, conf, uri);
        this.conf = conf;
        this.effectiveUser = ugi.getUserName();
    }

    public DistributedFileSystemOps(UserGroupInformation ugi, Configuration conf) {
        this(ugi, conf, null);
    }

    private DistributedFileSystem getDfs(UserGroupInformation ugi, final Configuration conf, final URI uri) {
        FileSystem fs = null;
        try {
            fs = (FileSystem)ugi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

                @Override
                public FileSystem run() throws IOException {
                    if (null != uri) {
                        return FileSystem.get((URI)uri, (Configuration)conf);
                    }
                    return FileSystem.get((URI)FileSystem.getDefaultUri((Configuration)conf), (Configuration)conf);
                }
            });
        }
        catch (IOException | InterruptedException ex) {
            logger.log(Level.SEVERE, "Unable to initialize FileSystem", ex);
        }
        return (DistributedFileSystem)fs;
    }

    public DistributedFileSystem getFilesystem() {
        return this.dfs;
    }

    public String getEffectiveUser() {
        return this.effectiveUser;
    }

    public String cat(Path file) throws IOException {
        StringBuilder out = new StringBuilder();
        try (BufferedReader br = new BufferedReader(new InputStreamReader((InputStream)this.dfs.open(file)));){
            String line;
            while ((line = br.readLine()) != null) {
                out.append(line).append("\n");
            }
            String string = out.toString();
            return string;
        }
    }

    public String cat(String file) throws IOException {
        Path path = new Path(file);
        return this.cat(path);
    }

    public void copyToLocal(String src, String dest) throws IOException {
        this.dfs.copyToLocalFile(new Path(src), new Path(dest));
    }

    public boolean mkdir(Path location, FsPermission filePermission) throws IOException {
        return this.dfs.mkdir(location, filePermission);
    }

    public boolean mkdir(String path) throws IOException {
        Path location = new Path(path);
        return this.dfs.mkdir(location, FsPermission.getDefault());
    }

    public boolean mkdirs(Path location, FsPermission filePermission) throws IOException {
        return this.dfs.mkdirs(location, filePermission);
    }

    public void touchz(Path location) throws IOException {
        this.dfs.create(location).close();
    }

    public FileStatus[] listStatus(Path location) throws IOException {
        return this.dfs.listStatus(location);
    }

    public FileStatus getFileStatus(Path location) throws IOException {
        return this.dfs.getFileStatus(location);
    }

    public boolean rm(Path location, boolean recursive) throws IOException {
        logger.log(Level.INFO, "Deleting {0} as {1}", new Object[]{location.toString(), this.dfs.toString()});
        if (this.dfs.exists(location)) {
            return this.dfs.delete(location, recursive);
        }
        return true;
    }

    public void copyFromLocal(boolean deleteSource, Path source, Path destination) throws IOException {
        this.dfs.copyFromLocalFile(deleteSource, source, destination);
    }

    public void copyToHDFSFromLocal(boolean deleteSource, String src, String destination) throws IOException {
        Path dirs = new Path(Utils.getDirectoryPart(destination));
        this.mkdirs(dirs, this.getParentPermission(dirs));
        Path destp = new Path(destination);
        Path srcp = new Path(src);
        this.copyFromLocal(deleteSource, srcp, destp);
    }

    public void moveWithinHdfs(Path source, Path destination) throws IOException {
        this.dfs.rename(source, destination);
    }

    public void renameInHdfs(String source, String destination) throws IOException {
        if (source.equals(destination)) {
            return;
        }
        Path src = new Path(source);
        Path dst = new Path(destination);
        if (!this.dfs.exists(dst.getParent())) {
            this.dfs.mkdirs(dst.getParent());
        }
        this.moveWithinHdfs(src, dst);
    }

    public boolean exists(String path) throws IOException {
        Path location = new Path(path);
        return this.dfs.exists(location);
    }

    public boolean exists(Path path) throws IOException {
        return this.dfs.exists(path);
    }

    public void copyInHdfs(Path src, Path dst) throws IOException {
        Path[] srcs = FileUtil.stat2Paths((FileStatus[])this.dfs.globStatus(src), (Path)src);
        if (srcs.length > 1 && !this.dfs.isDirectory(dst)) {
            throw new IOException("When copying multiple files, destination should be a directory.");
        }
        for (Path src1 : srcs) {
            FileUtil.copy((FileSystem)this.dfs, (Path)src1, (FileSystem)this.dfs, (Path)dst, (boolean)false, (Configuration)this.conf);
        }
    }

    public FSDataOutputStream create(String path) throws IOException {
        Path dstPath = new Path(path);
        String dirs = Utils.getDirectoryPart(path);
        Path dirsPath = new Path(dirs);
        if (!this.exists(dirs)) {
            this.dfs.mkdirs(dirsPath);
        }
        return this.dfs.create(dstPath);
    }

    public FSDataOutputStream create(Path path) throws IOException {
        return this.create(path.toString());
    }

    public void setPermission(Path path, FsPermission permission) throws IOException {
        this.dfs.setPermission(path, permission);
    }

    public void setPermission(Set<Path> paths, FsPermission permission) throws IOException {
        for (Path path : paths) {
            this.setPermission(path, permission);
        }
    }

    public void setOwner(Path path, String username, String groupname) throws IOException {
        this.dfs.setOwner(path, username, groupname);
    }

    public void setHdfsSpaceQuotaInMBs(Path src, long diskspaceQuotaInMB) throws IOException {
        this.setHdfsQuotaBytes(src, Long.MAX_VALUE, 0x100000L * diskspaceQuotaInMB);
    }

    public void setHdfsQuotaBytes(Path src, long numberOfFiles, long diskspaceQuotaInBytes) throws IOException {
        this.dfs.setQuota(src, numberOfFiles, diskspaceQuotaInBytes);
    }

    public long getHdfsSpaceQuotaInMbs(Path path) throws IOException {
        return this.dfs.getContentSummary(path).getSpaceQuota() / 0x100000L;
    }

    public long getHdfsNumFilesQuota(Path path) throws IOException {
        return this.dfs.getContentSummary(path).getQuota();
    }

    public long getUsedQuotaInMbs(Path path) throws IOException {
        return this.dfs.getContentSummary(path).getSpaceConsumed() / 0x100000L;
    }

    public FSDataInputStream open(Path location) throws IOException {
        return this.dfs.open(location);
    }

    public FSDataInputStream open(String location) throws IOException {
        Path path = new Path(location);
        return this.dfs.open(path);
    }

    public Configuration getConf() {
        return this.conf;
    }

    public void setConf(Configuration conf) {
        this.conf = conf;
    }

    public boolean compress(String p) throws IOException, IllegalStateException {
        EncodingStatus encodingStatus;
        Path location = new Path(p);
        File erasureCodingConfFile = new File(this.hadoopConfDir, "erasure-coding-site.xml");
        if (!erasureCodingConfFile.exists()) {
            logger.log(Level.SEVERE, "Unable to locate configuration file in {0}", erasureCodingConfFile);
            throw new IllegalStateException("No erasure coding conf file: erasure-coding-site.xml");
        }
        this.conf.addResource(new Path(erasureCodingConfFile.getAbsolutePath()));
        DistributedFileSystem localDfs = this.dfs;
        localDfs.setConf(this.conf);
        EncodingPolicy policy = new EncodingPolicy("src", 1);
        String path = location.toUri().getPath();
        localDfs.encodeFile(path, policy);
        while (!(encodingStatus = localDfs.getEncodingStatus(path)).isEncoded()) {
            try {
                Thread.sleep(1000L);
                logger.log(Level.INFO, "ongoing file compression of {0} ", path);
            }
            catch (InterruptedException e) {
                logger.log(Level.SEVERE, "Wait for encoding thread was interrupted.");
                return false;
            }
        }
        return true;
    }

    public FsPermission getParentPermission(Path path) throws IOException {
        Path location = new Path(path.toUri());
        if (this.dfs.exists(location)) {
            location = location.getParent();
            return this.dfs.getFileStatus(location).getPermission();
        }
        while (!this.dfs.exists(location)) {
            location = location.getParent();
        }
        return this.dfs.getFileStatus(location).getPermission();
    }

    public boolean isDir(String path) {
        Path location = new Path(path);
        try {
            return this.dfs.isDirectory(location);
        }
        catch (IOException ex) {
            logger.log(Level.SEVERE, null, ex);
            return false;
        }
    }

    public void setMetaEnabled(String location) throws IOException {
        Path path = new Path(location);
        this.setMetaEnabled(path);
    }

    public void setMetaEnabled(Path path) throws IOException {
        this.dfs.setMetaStatus(path, MetaStatus.META_ENABLED);
    }

    public void unsetMetaEnabled(Path path) throws IOException {
        this.dfs.setMetaStatus(path, MetaStatus.DISABLED);
    }

    public String getFileBlocks(String location) throws IOException {
        Path path = new Path(location);
        if (this.dfs.isFile(path)) {
            FileStatus filestatus = this.dfs.getFileStatus(path);
            long filesize = filestatus.getLen();
            long noOfBlocks = (long)Math.ceil(filesize / filestatus.getBlockSize());
            logger.log(Level.INFO, "File: {0}, Num of blocks: {1}", new Object[]{path, noOfBlocks});
            return "" + noOfBlocks;
        }
        return "-1";
    }

    public void close() {
        try {
            this.dfs.close();
        }
        catch (IOException ex) {
            logger.log(Level.SEVERE, "Error while closing file system.", ex);
        }
    }

    public long getlength(String path) {
        try {
            return this.dfs.getLength(new Path(path));
        }
        catch (IOException ex) {
            logger.log(Level.SEVERE, "Error while getting length of file", ex);
            return -1L;
        }
    }

    public long getLength(Path path) throws IOException {
        return this.dfs.getLength(path);
    }

    public long getDatasetSize(Path datasetPath) throws IOException {
        ContentSummary cs = this.dfs.getContentSummary(datasetPath);
        return cs.getLength();
    }

    public long getLastUpdatedDatasetSize(Path datasetPath) throws IOException {
        LastUpdatedContentSummary cs = this.dfs.getLastUpdatedContentSummary(datasetPath);
        return cs.getSpaceConsumed();
    }

    public void addUser(String userName) throws IOException {
        this.dfs.addUser(userName);
    }

    public void removeUser(String userName) throws IOException {
        this.dfs.removeUser(userName);
    }

    public void addGroup(String groupName) throws IOException {
        this.dfs.addGroup(groupName);
    }

    public void removeGroup(String groupName) throws IOException {
        this.dfs.removeGroup(groupName);
    }

    public void addUserToGroup(String userName, String groupName) throws IOException {
        this.dfs.addUserToGroup(userName, groupName);
    }

    public void removeUserFromGroup(String userName, String groupName) throws IOException {
        this.dfs.removeUserFromGroup(userName, groupName);
    }

    public void setStoragePolicy(Path path, StoragePolicy storagePolicy) throws IOException {
        this.dfs.setStoragePolicy(path, storagePolicy.toString());
    }

    public void setXAttr(Path path, String name, byte[] value) throws IOException {
        this.dfs.setXAttr(path, name, value);
    }

    public void removeXAttr(Path path, String name) throws IOException {
        this.dfs.removeXAttr(path, name);
    }

    public byte[] getXAttr(Path path, String name) throws IOException {
        return this.dfs.getXAttr(path, name);
    }

    public Map<String, byte[]> getXAttrs(Path path) throws IOException {
        return this.dfs.getXAttrs(path);
    }

    public static enum StoragePolicy {
        SMALL_FILES("DB"),
        DEFAULT("HOT");

        private String policyName;

        private StoragePolicy(String policyName) {
            this.policyName = policyName;
        }

        public String toString() {
            return this.policyName;
        }
    }
}

