/*
 * Decompiled with CFR 0.152.
 */
package io.hops.hopsworks.common.security.secrets;

import com.google.common.base.Strings;
import io.hops.hopsworks.common.dao.project.ProjectFacade;
import io.hops.hopsworks.common.dao.project.team.ProjectTeamFacade;
import io.hops.hopsworks.common.dao.user.UserFacade;
import io.hops.hopsworks.common.dao.user.security.secrets.SecretPlaintext;
import io.hops.hopsworks.common.dao.user.security.secrets.SecretsFacade;
import io.hops.hopsworks.common.security.CertificatesMgmService;
import io.hops.hopsworks.common.security.SymmetricEncryptionDescriptor;
import io.hops.hopsworks.common.security.SymmetricEncryptionService;
import io.hops.hopsworks.common.util.DateUtils;
import io.hops.hopsworks.exceptions.ProjectException;
import io.hops.hopsworks.exceptions.ServiceException;
import io.hops.hopsworks.exceptions.UserException;
import io.hops.hopsworks.persistence.entity.project.Project;
import io.hops.hopsworks.persistence.entity.project.team.ProjectTeam;
import io.hops.hopsworks.persistence.entity.user.Users;
import io.hops.hopsworks.persistence.entity.user.security.secrets.Secret;
import io.hops.hopsworks.persistence.entity.user.security.secrets.SecretId;
import io.hops.hopsworks.persistence.entity.user.security.secrets.VisibilityType;
import io.hops.hopsworks.restutils.RESTCodes;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.ejb.EJB;
import javax.ejb.EJBException;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;

@Stateless
@TransactionAttribute(value=TransactionAttributeType.SUPPORTS)
public class SecretsController {
    private static final Logger LOG = Logger.getLogger(SecretsController.class.getName());
    @EJB
    private SecretsFacade secretsFacade;
    @EJB
    private SymmetricEncryptionService symmetricEncryptionService;
    @EJB
    private CertificatesMgmService certificatesMgmService;
    @EJB
    private UserFacade userFacade;
    @EJB
    private ProjectFacade projectFacade;
    @EJB
    private ProjectTeamFacade projectTeamFacade;

    public Secret add(Users user, String secretName, String secret, VisibilityType visibilityType, Integer projectIdScope) throws UserException {
        SecretId secretId = new SecretId(user.getUid(), secretName);
        if (this.secretsFacade.findById(secretId) != null) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_EXISTS, Level.FINE, "Secret already exists", "Secret with name " + secretName + " already exists for user " + user.getUsername());
        }
        Secret storedSecret = this.validateAndCreateSecret(secretId, user, secret, visibilityType, projectIdScope);
        this.secretsFacade.persist(storedSecret);
        return storedSecret;
    }

    public Secret createSecretForProject(Users user, String secretName, String secret, Integer projectIdScope) throws UserException, ProjectException {
        Project project = this.projectFacade.find(projectIdScope);
        if (project == null) {
            throw new ProjectException(RESTCodes.ProjectErrorCode.PROJECT_NOT_FOUND, Level.FINE, "Project with ID " + projectIdScope + " does not exist!", "User " + user.getUsername() + " requested shared Secret " + secretName + " but Project with ID " + projectIdScope + "does not exist");
        }
        if (!this.projectTeamFacade.isUserMemberOfProject(project, user)) {
            throw new ProjectException(RESTCodes.ProjectErrorCode.TEAM_MEMBER_NOT_FOUND, Level.FINE, "User not a member of project with ID " + projectIdScope + ".");
        }
        SecretId secretId = new SecretId(user.getUid(), secretName);
        if (this.secretsFacade.findById(secretId) != null) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_EXISTS, Level.FINE, "Secret already exists", "Secret with name " + secretName + " already exists for user " + user.getUsername());
        }
        return this.validateAndCreateSecret(secretId, user, secret, VisibilityType.PROJECT, projectIdScope);
    }

    public Secret validateAndCreateSecret(SecretId secretId, Users user, String secret, VisibilityType visibilityType, Integer projectIdScope) throws UserException {
        this.checkIfUserIsNull(user);
        this.checkIfNameIsNullOrEmpty(secretId.getName());
        if (Strings.isNullOrEmpty((String)secretId.getName()) || Strings.isNullOrEmpty((String)secret)) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_EMPTY, Level.FINE, "Secret value is either null or empty", "Secret name or value is empty or null");
        }
        try {
            Secret storedSecret = new Secret(secretId, this.encryptSecret(secret), DateUtils.localDateTime2Date(DateUtils.getNow()));
            storedSecret.setVisibilityType(visibilityType);
            if (visibilityType.equals((Object)VisibilityType.PRIVATE)) {
                storedSecret.setProjectIdScope(null);
            } else {
                if (projectIdScope == null) {
                    throw new UserException(RESTCodes.UserErrorCode.SECRET_EMPTY, Level.FINE, "Secret visibility is PROJECT but there is not Project ID scope", "Project scope for shared secret " + secretId.getName() + " is null");
                }
                storedSecret.setProjectIdScope(projectIdScope);
            }
            return storedSecret;
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_ENCRYPTION_ERROR, Level.SEVERE, "Error encrypting secret", "Could not encrypt Secret " + secretId.getName(), (Throwable)ex);
        }
    }

    public List<SecretPlaintext> getAllForUser(Users user) throws UserException {
        this.checkIfUserIsNull(user);
        List<Secret> secrets = this.secretsFacade.findAllForUser(user);
        return secrets.stream().map(c -> this.constructSecretView(user, (Secret)c)).collect(Collectors.toList());
    }

    public void delete(Users user, String secretName) throws UserException {
        this.checkIfUserIsNull(user);
        this.checkIfNameIsNullOrEmpty(secretName);
        SecretId secretId = new SecretId(user.getUid(), secretName);
        try {
            this.secretsFacade.deleteSecret(secretId);
        }
        catch (EJBException de) {
            Throwable rootCause = this.getRootCause(de);
            if (rootCause instanceof SQLIntegrityConstraintViolationException) {
                throw new UserException(RESTCodes.UserErrorCode.SECRET_DELETION_FAILED, Level.FINE, "Cannot delete secret. Secret is in use by a connector. Try deleting the connector first. ", rootCause.getMessage());
            }
            throw de;
        }
    }

    private Throwable getRootCause(Throwable throwable) {
        Throwable rootCause = throwable;
        while (throwable != null) {
            rootCause = throwable;
            throwable = throwable.getCause();
        }
        return rootCause;
    }

    public void deleteAll(Users user) throws UserException {
        this.checkIfUserIsNull(user);
        try {
            this.secretsFacade.deleteSecretsForUser(user);
        }
        catch (EJBException de) {
            Throwable rootCause = this.getRootCause(de);
            if (rootCause instanceof SQLIntegrityConstraintViolationException) {
                throw new UserException(RESTCodes.UserErrorCode.SECRET_DELETION_FAILED, Level.FINE, "Cannot delete secrets. One or more secrets are in use by a connector. Try deleting the connectors first. ", rootCause.getMessage());
            }
            throw de;
        }
    }

    public List<Secret> getAllCiphered() {
        return this.secretsFacade.findAll();
    }

    public SecretPlaintext get(Users user, String secretName) throws UserException {
        this.checkIfUserIsNull(user);
        this.checkIfNameIsNullOrEmpty(secretName);
        SecretId id = new SecretId(user.getUid(), secretName);
        Secret storedSecret = this.secretsFacade.findById(id);
        this.checkIfSecretIsNull(storedSecret, secretName, user);
        try {
            return this.decrypt(user, storedSecret);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_ENCRYPTION_ERROR, Level.SEVERE, "Error decrypting Secret", "Could not decrypt Secret " + secretName, (Throwable)ex);
        }
    }

    public SecretPlaintext getShared(Users caller, String ownerUsername, String secretName) throws UserException, ServiceException, ProjectException {
        Users ownerUser = this.userFacade.findByUsername(ownerUsername);
        return this.getShared(caller, ownerUser, secretName);
    }

    public SecretPlaintext getShared(Users caller, Users ownerUser, String secretName) throws UserException, ServiceException, ProjectException {
        this.checkIfUserIsNull(caller);
        this.checkIfNameIsNullOrEmpty(secretName);
        this.checkIfUserIsNull(ownerUser);
        Secret storedSecret = this.secretsFacade.findById(new SecretId(ownerUser.getUid(), secretName));
        this.checkIfSecretIsNull(storedSecret, secretName, ownerUser);
        if (storedSecret.getVisibilityType() == null || storedSecret.getVisibilityType().equals((Object)VisibilityType.PRIVATE)) {
            throw new UserException(RESTCodes.UserErrorCode.ACCESS_CONTROL, Level.FINE, "Secret is Private", "User " + caller.getUsername() + " requested PRIVATE secret <" + ownerUser.getUid() + ", " + secretName + ">");
        }
        Integer projectId = storedSecret.getProjectIdScope();
        if (projectId == null) {
            throw new ServiceException(RESTCodes.ServiceErrorCode.SERVICE_GENERIC_ERROR, Level.WARNING, "Visibility's Project ID is empty", "Secret " + secretName + " visibility is PROJECT but Project ID is null");
        }
        Project project = this.projectFacade.find(projectId);
        if (project == null) {
            throw new ProjectException(RESTCodes.ProjectErrorCode.PROJECT_NOT_FOUND, Level.FINE, "Project with ID " + projectId + " does not exist!", "User " + caller.getUsername() + " requested shared Secret " + secretName + " but Project with ID " + projectId + "does not exist");
        }
        for (ProjectTeam projectTeam : project.getProjectTeamCollection()) {
            if (!caller.getUid().equals(projectTeam.getUser().getUid())) continue;
            try {
                return this.decrypt(ownerUser, storedSecret);
            }
            catch (IOException | GeneralSecurityException ex) {
                throw new UserException(RESTCodes.UserErrorCode.SECRET_ENCRYPTION_ERROR, Level.SEVERE, "Error decrypting Secret", "Could not decrypt Secret " + secretName, (Throwable)ex);
            }
        }
        throw new UserException(RESTCodes.UserErrorCode.ACCESS_CONTROL, Level.FINE, "Not authorized to access Secret " + secretName, "User " + caller.getUsername() + " tried to access shared Secret " + secretName + " but they are not member of Project " + project.getName());
    }

    private void checkIfUserIsNull(Users user) throws UserException {
        if (user == null) {
            throw new UserException(RESTCodes.UserErrorCode.USER_DOES_NOT_EXIST, Level.FINE);
        }
    }

    private void checkIfNameIsNullOrEmpty(String name) throws UserException {
        if (Strings.isNullOrEmpty((String)name)) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_EMPTY, Level.FINE, "Secret is either null or empty", "Secret name or key is empty or null");
        }
    }

    private void checkIfSecretIsNull(Secret secret, String secretName, Users user) throws UserException {
        if (secret == null) {
            throw new UserException(RESTCodes.UserErrorCode.SECRET_EMPTY, Level.FINE, "Could not find Secret for user", "Could not find Secret with name " + secretName + " for user " + user.getUsername());
        }
    }

    private SecretPlaintext constructSecretView(Users user, Secret ciphered) {
        return SecretPlaintext.newInstance(user, ciphered.getId().getName(), "", ciphered.getAddedOn(), ciphered.getVisibilityType());
    }

    private SecretPlaintext decrypt(Users user, Secret ciphered) throws IOException, GeneralSecurityException {
        String password = this.certificatesMgmService.getMasterEncryptionPassword();
        byte[][] split = this.symmetricEncryptionService.splitPayloadFromCryptoPrimitives(ciphered.getSecret());
        SymmetricEncryptionDescriptor descriptor = new SymmetricEncryptionDescriptor.Builder().setPassword(password).setSalt(split[0]).setIV(split[1]).setInput(split[2]).build();
        descriptor = this.symmetricEncryptionService.decrypt(descriptor);
        byte[] plaintext = descriptor.getOutput();
        return SecretPlaintext.newInstance(user, ciphered.getId().getName(), this.bytes2string(plaintext), ciphered.getAddedOn());
    }

    public byte[] encryptSecret(String secret) throws IOException, GeneralSecurityException {
        String password = this.certificatesMgmService.getMasterEncryptionPassword();
        SymmetricEncryptionDescriptor descriptor = new SymmetricEncryptionDescriptor.Builder().setInput(this.string2bytes(secret)).setPassword(password).build();
        descriptor = this.symmetricEncryptionService.encrypt(descriptor);
        return this.symmetricEncryptionService.mergePayloadWithCryptoPrimitives(descriptor.getSalt(), descriptor.getIv(), descriptor.getOutput());
    }

    private byte[] string2bytes(String str) {
        return str.getBytes(Charset.defaultCharset());
    }

    private String bytes2string(byte[] bytes) {
        return new String(bytes, Charset.defaultCharset());
    }

    @TransactionAttribute(value=TransactionAttributeType.SUPPORTS)
    public void checkCanAccessSecret(Secret secret, Users user) throws ProjectException {
        if (secret != null && !this.projectTeamFacade.isUserMemberOfProject(this.projectFacade.find(secret.getProjectIdScope()), user)) {
            throw new ProjectException(RESTCodes.ProjectErrorCode.TEAM_MEMBER_NOT_FOUND, Level.FINE, "User not a member of project with ID " + secret.getProjectIdScope() + ". Can not delete secret in project with id " + secret.getProjectIdScope());
        }
    }
}

