/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object containing configuration for a compute environment (such as Spark, Python or Athena) returned by the
 * <code>DescribeConnectionType</code> API.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComputeEnvironmentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ComputeEnvironmentConfiguration.Builder, ComputeEnvironmentConfiguration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ComputeEnvironmentConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ComputeEnvironmentConfiguration::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> COMPUTE_ENVIRONMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComputeEnvironment").getter(getter(ComputeEnvironmentConfiguration::computeEnvironmentAsString))
            .setter(setter(Builder::computeEnvironment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeEnvironment").build())
            .build();

    private static final SdkField<List<String>> SUPPORTED_AUTHENTICATION_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedAuthenticationTypes")
            .getter(getter(ComputeEnvironmentConfiguration::supportedAuthenticationTypesAsStrings))
            .setter(setter(Builder::supportedAuthenticationTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedAuthenticationTypes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, Property>> CONNECTION_OPTIONS_FIELD = SdkField
            .<Map<String, Property>> builder(MarshallingType.MAP)
            .memberName("ConnectionOptions")
            .getter(getter(ComputeEnvironmentConfiguration::connectionOptions))
            .setter(setter(Builder::connectionOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Property> builder(MarshallingType.SDK_POJO)
                                            .constructor(Property::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CONNECTION_PROPERTY_NAME_OVERRIDES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ConnectionPropertyNameOverrides")
            .getter(getter(ComputeEnvironmentConfiguration::connectionPropertyNameOverrides))
            .setter(setter(Builder::connectionPropertyNameOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionPropertyNameOverrides")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CONNECTION_OPTION_NAME_OVERRIDES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ConnectionOptionNameOverrides")
            .getter(getter(ComputeEnvironmentConfiguration::connectionOptionNameOverrides))
            .setter(setter(Builder::connectionOptionNameOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionOptionNameOverrides")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> CONNECTION_PROPERTIES_REQUIRED_OVERRIDES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ConnectionPropertiesRequiredOverrides")
            .getter(getter(ComputeEnvironmentConfiguration::connectionPropertiesRequiredOverrides))
            .setter(setter(Builder::connectionPropertiesRequiredOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("ConnectionPropertiesRequiredOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PHYSICAL_CONNECTION_PROPERTIES_REQUIRED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PhysicalConnectionPropertiesRequired")
            .getter(getter(ComputeEnvironmentConfiguration::physicalConnectionPropertiesRequired))
            .setter(setter(Builder::physicalConnectionPropertiesRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PhysicalConnectionPropertiesRequired").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            COMPUTE_ENVIRONMENT_FIELD, SUPPORTED_AUTHENTICATION_TYPES_FIELD, CONNECTION_OPTIONS_FIELD,
            CONNECTION_PROPERTY_NAME_OVERRIDES_FIELD, CONNECTION_OPTION_NAME_OVERRIDES_FIELD,
            CONNECTION_PROPERTIES_REQUIRED_OVERRIDES_FIELD, PHYSICAL_CONNECTION_PROPERTIES_REQUIRED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String computeEnvironment;

    private final List<String> supportedAuthenticationTypes;

    private final Map<String, Property> connectionOptions;

    private final Map<String, String> connectionPropertyNameOverrides;

    private final Map<String, String> connectionOptionNameOverrides;

    private final List<String> connectionPropertiesRequiredOverrides;

    private final Boolean physicalConnectionPropertiesRequired;

    private ComputeEnvironmentConfiguration(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.computeEnvironment = builder.computeEnvironment;
        this.supportedAuthenticationTypes = builder.supportedAuthenticationTypes;
        this.connectionOptions = builder.connectionOptions;
        this.connectionPropertyNameOverrides = builder.connectionPropertyNameOverrides;
        this.connectionOptionNameOverrides = builder.connectionOptionNameOverrides;
        this.connectionPropertiesRequiredOverrides = builder.connectionPropertiesRequiredOverrides;
        this.physicalConnectionPropertiesRequired = builder.physicalConnectionPropertiesRequired;
    }

    /**
     * <p>
     * A name for the compute environment configuration.
     * </p>
     * 
     * @return A name for the compute environment configuration.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description of the compute environment.
     * </p>
     * 
     * @return A description of the compute environment.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The type of compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #computeEnvironment} will return {@link ComputeEnvironment#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #computeEnvironmentAsString}.
     * </p>
     * 
     * @return The type of compute environment.
     * @see ComputeEnvironment
     */
    public final ComputeEnvironment computeEnvironment() {
        return ComputeEnvironment.fromValue(computeEnvironment);
    }

    /**
     * <p>
     * The type of compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #computeEnvironment} will return {@link ComputeEnvironment#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #computeEnvironmentAsString}.
     * </p>
     * 
     * @return The type of compute environment.
     * @see ComputeEnvironment
     */
    public final String computeEnvironmentAsString() {
        return computeEnvironment;
    }

    /**
     * <p>
     * The supported authentication types for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedAuthenticationTypes} method.
     * </p>
     * 
     * @return The supported authentication types for the compute environment.
     */
    public final List<AuthenticationType> supportedAuthenticationTypes() {
        return AuthenticationTypesCopier.copyStringToEnum(supportedAuthenticationTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedAuthenticationTypes property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedAuthenticationTypes() {
        return supportedAuthenticationTypes != null && !(supportedAuthenticationTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The supported authentication types for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedAuthenticationTypes} method.
     * </p>
     * 
     * @return The supported authentication types for the compute environment.
     */
    public final List<String> supportedAuthenticationTypesAsStrings() {
        return supportedAuthenticationTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ConnectionOptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConnectionOptions() {
        return connectionOptions != null && !(connectionOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters used as connection options for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionOptions} method.
     * </p>
     * 
     * @return The parameters used as connection options for the compute environment.
     */
    public final Map<String, Property> connectionOptions() {
        return connectionOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ConnectionPropertyNameOverrides
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasConnectionPropertyNameOverrides() {
        return connectionPropertyNameOverrides != null && !(connectionPropertyNameOverrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The connection property name overrides for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionPropertyNameOverrides}
     * method.
     * </p>
     * 
     * @return The connection property name overrides for the compute environment.
     */
    public final Map<String, String> connectionPropertyNameOverrides() {
        return connectionPropertyNameOverrides;
    }

    /**
     * For responses, this returns true if the service returned a value for the ConnectionOptionNameOverrides property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasConnectionOptionNameOverrides() {
        return connectionOptionNameOverrides != null && !(connectionOptionNameOverrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The connection option name overrides for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionOptionNameOverrides} method.
     * </p>
     * 
     * @return The connection option name overrides for the compute environment.
     */
    public final Map<String, String> connectionOptionNameOverrides() {
        return connectionOptionNameOverrides;
    }

    /**
     * For responses, this returns true if the service returned a value for the ConnectionPropertiesRequiredOverrides
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasConnectionPropertiesRequiredOverrides() {
        return connectionPropertiesRequiredOverrides != null
                && !(connectionPropertiesRequiredOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The connection properties that are required as overrides for the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConnectionPropertiesRequiredOverrides}
     * method.
     * </p>
     * 
     * @return The connection properties that are required as overrides for the compute environment.
     */
    public final List<String> connectionPropertiesRequiredOverrides() {
        return connectionPropertiesRequiredOverrides;
    }

    /**
     * <p>
     * Indicates whether <code>PhysicalConnectionProperties</code> are required for the compute environment.
     * </p>
     * 
     * @return Indicates whether <code>PhysicalConnectionProperties</code> are required for the compute environment.
     */
    public final Boolean physicalConnectionPropertiesRequired() {
        return physicalConnectionPropertiesRequired;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(computeEnvironmentAsString());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasSupportedAuthenticationTypes() ? supportedAuthenticationTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasConnectionOptions() ? connectionOptions() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasConnectionPropertyNameOverrides() ? connectionPropertyNameOverrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasConnectionOptionNameOverrides() ? connectionOptionNameOverrides() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasConnectionPropertiesRequiredOverrides() ? connectionPropertiesRequiredOverrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(physicalConnectionPropertiesRequired());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeEnvironmentConfiguration)) {
            return false;
        }
        ComputeEnvironmentConfiguration other = (ComputeEnvironmentConfiguration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(computeEnvironmentAsString(), other.computeEnvironmentAsString())
                && hasSupportedAuthenticationTypes() == other.hasSupportedAuthenticationTypes()
                && Objects.equals(supportedAuthenticationTypesAsStrings(), other.supportedAuthenticationTypesAsStrings())
                && hasConnectionOptions() == other.hasConnectionOptions()
                && Objects.equals(connectionOptions(), other.connectionOptions())
                && hasConnectionPropertyNameOverrides() == other.hasConnectionPropertyNameOverrides()
                && Objects.equals(connectionPropertyNameOverrides(), other.connectionPropertyNameOverrides())
                && hasConnectionOptionNameOverrides() == other.hasConnectionOptionNameOverrides()
                && Objects.equals(connectionOptionNameOverrides(), other.connectionOptionNameOverrides())
                && hasConnectionPropertiesRequiredOverrides() == other.hasConnectionPropertiesRequiredOverrides()
                && Objects.equals(connectionPropertiesRequiredOverrides(), other.connectionPropertiesRequiredOverrides())
                && Objects.equals(physicalConnectionPropertiesRequired(), other.physicalConnectionPropertiesRequired());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("ComputeEnvironmentConfiguration")
                .add("Name", name())
                .add("Description", description())
                .add("ComputeEnvironment", computeEnvironmentAsString())
                .add("SupportedAuthenticationTypes",
                        hasSupportedAuthenticationTypes() ? supportedAuthenticationTypesAsStrings() : null)
                .add("ConnectionOptions", hasConnectionOptions() ? connectionOptions() : null)
                .add("ConnectionPropertyNameOverrides",
                        hasConnectionPropertyNameOverrides() ? connectionPropertyNameOverrides() : null)
                .add("ConnectionOptionNameOverrides", hasConnectionOptionNameOverrides() ? connectionOptionNameOverrides() : null)
                .add("ConnectionPropertiesRequiredOverrides",
                        hasConnectionPropertiesRequiredOverrides() ? connectionPropertiesRequiredOverrides() : null)
                .add("PhysicalConnectionPropertiesRequired", physicalConnectionPropertiesRequired()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ComputeEnvironment":
            return Optional.ofNullable(clazz.cast(computeEnvironmentAsString()));
        case "SupportedAuthenticationTypes":
            return Optional.ofNullable(clazz.cast(supportedAuthenticationTypesAsStrings()));
        case "ConnectionOptions":
            return Optional.ofNullable(clazz.cast(connectionOptions()));
        case "ConnectionPropertyNameOverrides":
            return Optional.ofNullable(clazz.cast(connectionPropertyNameOverrides()));
        case "ConnectionOptionNameOverrides":
            return Optional.ofNullable(clazz.cast(connectionOptionNameOverrides()));
        case "ConnectionPropertiesRequiredOverrides":
            return Optional.ofNullable(clazz.cast(connectionPropertiesRequiredOverrides()));
        case "PhysicalConnectionPropertiesRequired":
            return Optional.ofNullable(clazz.cast(physicalConnectionPropertiesRequired()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ComputeEnvironment", COMPUTE_ENVIRONMENT_FIELD);
        map.put("SupportedAuthenticationTypes", SUPPORTED_AUTHENTICATION_TYPES_FIELD);
        map.put("ConnectionOptions", CONNECTION_OPTIONS_FIELD);
        map.put("ConnectionPropertyNameOverrides", CONNECTION_PROPERTY_NAME_OVERRIDES_FIELD);
        map.put("ConnectionOptionNameOverrides", CONNECTION_OPTION_NAME_OVERRIDES_FIELD);
        map.put("ConnectionPropertiesRequiredOverrides", CONNECTION_PROPERTIES_REQUIRED_OVERRIDES_FIELD);
        map.put("PhysicalConnectionPropertiesRequired", PHYSICAL_CONNECTION_PROPERTIES_REQUIRED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComputeEnvironmentConfiguration, T> g) {
        return obj -> g.apply((ComputeEnvironmentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComputeEnvironmentConfiguration> {
        /**
         * <p>
         * A name for the compute environment configuration.
         * </p>
         * 
         * @param name
         *        A name for the compute environment configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the compute environment.
         * </p>
         * 
         * @param description
         *        A description of the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The type of compute environment.
         * </p>
         * 
         * @param computeEnvironment
         *        The type of compute environment.
         * @see ComputeEnvironment
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeEnvironment
         */
        Builder computeEnvironment(String computeEnvironment);

        /**
         * <p>
         * The type of compute environment.
         * </p>
         * 
         * @param computeEnvironment
         *        The type of compute environment.
         * @see ComputeEnvironment
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeEnvironment
         */
        Builder computeEnvironment(ComputeEnvironment computeEnvironment);

        /**
         * <p>
         * The supported authentication types for the compute environment.
         * </p>
         * 
         * @param supportedAuthenticationTypes
         *        The supported authentication types for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAuthenticationTypesWithStrings(Collection<String> supportedAuthenticationTypes);

        /**
         * <p>
         * The supported authentication types for the compute environment.
         * </p>
         * 
         * @param supportedAuthenticationTypes
         *        The supported authentication types for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAuthenticationTypesWithStrings(String... supportedAuthenticationTypes);

        /**
         * <p>
         * The supported authentication types for the compute environment.
         * </p>
         * 
         * @param supportedAuthenticationTypes
         *        The supported authentication types for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAuthenticationTypes(Collection<AuthenticationType> supportedAuthenticationTypes);

        /**
         * <p>
         * The supported authentication types for the compute environment.
         * </p>
         * 
         * @param supportedAuthenticationTypes
         *        The supported authentication types for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAuthenticationTypes(AuthenticationType... supportedAuthenticationTypes);

        /**
         * <p>
         * The parameters used as connection options for the compute environment.
         * </p>
         * 
         * @param connectionOptions
         *        The parameters used as connection options for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionOptions(Map<String, Property> connectionOptions);

        /**
         * <p>
         * The connection property name overrides for the compute environment.
         * </p>
         * 
         * @param connectionPropertyNameOverrides
         *        The connection property name overrides for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionPropertyNameOverrides(Map<String, String> connectionPropertyNameOverrides);

        /**
         * <p>
         * The connection option name overrides for the compute environment.
         * </p>
         * 
         * @param connectionOptionNameOverrides
         *        The connection option name overrides for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionOptionNameOverrides(Map<String, String> connectionOptionNameOverrides);

        /**
         * <p>
         * The connection properties that are required as overrides for the compute environment.
         * </p>
         * 
         * @param connectionPropertiesRequiredOverrides
         *        The connection properties that are required as overrides for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionPropertiesRequiredOverrides(Collection<String> connectionPropertiesRequiredOverrides);

        /**
         * <p>
         * The connection properties that are required as overrides for the compute environment.
         * </p>
         * 
         * @param connectionPropertiesRequiredOverrides
         *        The connection properties that are required as overrides for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionPropertiesRequiredOverrides(String... connectionPropertiesRequiredOverrides);

        /**
         * <p>
         * Indicates whether <code>PhysicalConnectionProperties</code> are required for the compute environment.
         * </p>
         * 
         * @param physicalConnectionPropertiesRequired
         *        Indicates whether <code>PhysicalConnectionProperties</code> are required for the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalConnectionPropertiesRequired(Boolean physicalConnectionPropertiesRequired);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String computeEnvironment;

        private List<String> supportedAuthenticationTypes = DefaultSdkAutoConstructList.getInstance();

        private Map<String, Property> connectionOptions = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> connectionPropertyNameOverrides = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> connectionOptionNameOverrides = DefaultSdkAutoConstructMap.getInstance();

        private List<String> connectionPropertiesRequiredOverrides = DefaultSdkAutoConstructList.getInstance();

        private Boolean physicalConnectionPropertiesRequired;

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeEnvironmentConfiguration model) {
            name(model.name);
            description(model.description);
            computeEnvironment(model.computeEnvironment);
            supportedAuthenticationTypesWithStrings(model.supportedAuthenticationTypes);
            connectionOptions(model.connectionOptions);
            connectionPropertyNameOverrides(model.connectionPropertyNameOverrides);
            connectionOptionNameOverrides(model.connectionOptionNameOverrides);
            connectionPropertiesRequiredOverrides(model.connectionPropertiesRequiredOverrides);
            physicalConnectionPropertiesRequired(model.physicalConnectionPropertiesRequired);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getComputeEnvironment() {
            return computeEnvironment;
        }

        public final void setComputeEnvironment(String computeEnvironment) {
            this.computeEnvironment = computeEnvironment;
        }

        @Override
        public final Builder computeEnvironment(String computeEnvironment) {
            this.computeEnvironment = computeEnvironment;
            return this;
        }

        @Override
        public final Builder computeEnvironment(ComputeEnvironment computeEnvironment) {
            this.computeEnvironment(computeEnvironment == null ? null : computeEnvironment.toString());
            return this;
        }

        public final Collection<String> getSupportedAuthenticationTypes() {
            if (supportedAuthenticationTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedAuthenticationTypes;
        }

        public final void setSupportedAuthenticationTypes(Collection<String> supportedAuthenticationTypes) {
            this.supportedAuthenticationTypes = AuthenticationTypesCopier.copy(supportedAuthenticationTypes);
        }

        @Override
        public final Builder supportedAuthenticationTypesWithStrings(Collection<String> supportedAuthenticationTypes) {
            this.supportedAuthenticationTypes = AuthenticationTypesCopier.copy(supportedAuthenticationTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedAuthenticationTypesWithStrings(String... supportedAuthenticationTypes) {
            supportedAuthenticationTypesWithStrings(Arrays.asList(supportedAuthenticationTypes));
            return this;
        }

        @Override
        public final Builder supportedAuthenticationTypes(Collection<AuthenticationType> supportedAuthenticationTypes) {
            this.supportedAuthenticationTypes = AuthenticationTypesCopier.copyEnumToString(supportedAuthenticationTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedAuthenticationTypes(AuthenticationType... supportedAuthenticationTypes) {
            supportedAuthenticationTypes(Arrays.asList(supportedAuthenticationTypes));
            return this;
        }

        public final Map<String, Property.Builder> getConnectionOptions() {
            Map<String, Property.Builder> result = PropertiesMapCopier.copyToBuilder(this.connectionOptions);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setConnectionOptions(Map<String, Property.BuilderImpl> connectionOptions) {
            this.connectionOptions = PropertiesMapCopier.copyFromBuilder(connectionOptions);
        }

        @Override
        public final Builder connectionOptions(Map<String, Property> connectionOptions) {
            this.connectionOptions = PropertiesMapCopier.copy(connectionOptions);
            return this;
        }

        public final Map<String, String> getConnectionPropertyNameOverrides() {
            if (connectionPropertyNameOverrides instanceof SdkAutoConstructMap) {
                return null;
            }
            return connectionPropertyNameOverrides;
        }

        public final void setConnectionPropertyNameOverrides(Map<String, String> connectionPropertyNameOverrides) {
            this.connectionPropertyNameOverrides = PropertyNameOverridesCopier.copy(connectionPropertyNameOverrides);
        }

        @Override
        public final Builder connectionPropertyNameOverrides(Map<String, String> connectionPropertyNameOverrides) {
            this.connectionPropertyNameOverrides = PropertyNameOverridesCopier.copy(connectionPropertyNameOverrides);
            return this;
        }

        public final Map<String, String> getConnectionOptionNameOverrides() {
            if (connectionOptionNameOverrides instanceof SdkAutoConstructMap) {
                return null;
            }
            return connectionOptionNameOverrides;
        }

        public final void setConnectionOptionNameOverrides(Map<String, String> connectionOptionNameOverrides) {
            this.connectionOptionNameOverrides = PropertyNameOverridesCopier.copy(connectionOptionNameOverrides);
        }

        @Override
        public final Builder connectionOptionNameOverrides(Map<String, String> connectionOptionNameOverrides) {
            this.connectionOptionNameOverrides = PropertyNameOverridesCopier.copy(connectionOptionNameOverrides);
            return this;
        }

        public final Collection<String> getConnectionPropertiesRequiredOverrides() {
            if (connectionPropertiesRequiredOverrides instanceof SdkAutoConstructList) {
                return null;
            }
            return connectionPropertiesRequiredOverrides;
        }

        public final void setConnectionPropertiesRequiredOverrides(Collection<String> connectionPropertiesRequiredOverrides) {
            this.connectionPropertiesRequiredOverrides = ListOfStringCopier.copy(connectionPropertiesRequiredOverrides);
        }

        @Override
        public final Builder connectionPropertiesRequiredOverrides(Collection<String> connectionPropertiesRequiredOverrides) {
            this.connectionPropertiesRequiredOverrides = ListOfStringCopier.copy(connectionPropertiesRequiredOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder connectionPropertiesRequiredOverrides(String... connectionPropertiesRequiredOverrides) {
            connectionPropertiesRequiredOverrides(Arrays.asList(connectionPropertiesRequiredOverrides));
            return this;
        }

        public final Boolean getPhysicalConnectionPropertiesRequired() {
            return physicalConnectionPropertiesRequired;
        }

        public final void setPhysicalConnectionPropertiesRequired(Boolean physicalConnectionPropertiesRequired) {
            this.physicalConnectionPropertiesRequired = physicalConnectionPropertiesRequired;
        }

        @Override
        public final Builder physicalConnectionPropertiesRequired(Boolean physicalConnectionPropertiesRequired) {
            this.physicalConnectionPropertiesRequired = physicalConnectionPropertiesRequired;
            return this;
        }

        @Override
        public ComputeEnvironmentConfiguration build() {
            return new ComputeEnvironmentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
