/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.aws.transaction.lock;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.util.Properties;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.hudi.aws.credentials.HoodieAWSCredentialsProviderFactory;
import org.apache.hudi.client.transaction.lock.StorageLockClient;
import org.apache.hudi.client.transaction.lock.models.LockGetResult;
import org.apache.hudi.client.transaction.lock.models.LockUpsertResult;
import org.apache.hudi.client.transaction.lock.models.StorageLockData;
import org.apache.hudi.client.transaction.lock.models.StorageLockFile;
import org.apache.hudi.common.util.Functions;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.VisibleForTesting;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.StorageBasedLockConfig;
import org.apache.hudi.exception.HoodieLockException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.model.GetBucketLocationRequest;
import software.amazon.awssdk.services.s3.model.GetBucketLocationResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Exception;

@ThreadSafe
public class S3StorageLockClient
implements StorageLockClient {
    private static final Logger LOG = LoggerFactory.getLogger(S3StorageLockClient.class);
    private static final int PRECONDITION_FAILURE_ERROR_CODE = 412;
    private static final int NOT_FOUND_ERROR_CODE = 404;
    private static final int CONDITIONAL_REQUEST_CONFLICT_ERROR_CODE = 409;
    private static final int RATE_LIMIT_ERROR_CODE = 429;
    private static final int INTERNAL_SERVER_ERROR_CODE_MIN = 500;
    private final Logger logger;
    private final S3Client s3Client;
    private final String bucketName;
    private final String lockFilePath;
    private final String ownerId;

    public S3StorageLockClient(String ownerId, String lockFileUri, Properties props) {
        this(ownerId, lockFileUri, props, S3StorageLockClient.createDefaultS3Client(), LOG);
    }

    @VisibleForTesting
    S3StorageLockClient(String ownerId, String lockFileUri, Properties props, Functions.Function2<String, Properties, S3Client> s3ClientSupplier, Logger logger) {
        try {
            URI uri = new URI(lockFileUri);
            this.bucketName = uri.getHost();
            this.lockFilePath = uri.getPath().replaceFirst("/", "");
            this.s3Client = (S3Client)s3ClientSupplier.apply((Object)this.bucketName, (Object)props);
            if (StringUtils.isNullOrEmpty((String)this.bucketName)) {
                throw new IllegalArgumentException("LockFileUri does not contain a valid bucket name.");
            }
            if (StringUtils.isNullOrEmpty((String)this.lockFilePath)) {
                throw new IllegalArgumentException("LockFileUri does not contain a valid lock file path.");
            }
            this.ownerId = ownerId;
            this.logger = logger;
        }
        catch (URISyntaxException e) {
            throw new HoodieLockException((Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Pair<LockGetResult, Option<StorageLockFile>> readCurrentLockFile() {
        try (ResponseInputStream in = this.s3Client.getObject((GetObjectRequest)GetObjectRequest.builder().bucket(this.bucketName).key(this.lockFilePath).build());){
            String eTag = ((GetObjectResponse)in.response()).eTag();
            Pair pair = Pair.of((Object)LockGetResult.SUCCESS, (Object)Option.of((Object)StorageLockFile.createFromStream((InputStream)in, (String)eTag)));
            return pair;
        }
        catch (S3Exception e) {
            int status = e.statusCode();
            LockGetResult result = LockGetResult.UNKNOWN_ERROR;
            if (status == 404) {
                this.logger.info("OwnerId: {}, Object not found: {}", (Object)this.ownerId, (Object)this.lockFilePath);
                result = LockGetResult.NOT_EXISTS;
                return Pair.of((Object)result, (Object)Option.empty());
            }
            if (status == 409) {
                this.logger.info("OwnerId: {}, Conflicting operation has occurred: {}", (Object)this.ownerId, (Object)this.lockFilePath);
                return Pair.of((Object)result, (Object)Option.empty());
            }
            if (status == 429) {
                this.logger.warn("OwnerId: {}, Rate limit exceeded: {}", (Object)this.ownerId, (Object)this.lockFilePath);
                return Pair.of((Object)result, (Object)Option.empty());
            }
            if (status < 500) throw e;
            this.logger.warn("OwnerId: {}, S3 internal server error: {}", new Object[]{this.ownerId, this.lockFilePath, e});
            return Pair.of((Object)result, (Object)Option.empty());
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed reading lock file from S3: " + this.lockFilePath, e);
        }
    }

    public Pair<LockUpsertResult, Option<StorageLockFile>> tryUpsertLockFile(StorageLockData newLockData, Option<StorageLockFile> previousLockFile) {
        LockUpsertResult result;
        block3: {
            boolean isLockRenewal = previousLockFile.isPresent();
            String currentEtag = isLockRenewal ? ((StorageLockFile)previousLockFile.get()).getVersionId() : null;
            result = LockUpsertResult.UNKNOWN_ERROR;
            try {
                StorageLockFile updated = this.createOrUpdateLockFileInternal(newLockData, currentEtag);
                return Pair.of((Object)LockUpsertResult.SUCCESS, (Object)Option.of((Object)updated));
            }
            catch (S3Exception e) {
                result = this.handleUpsertS3Exception(e);
            }
            catch (AwsServiceException | SdkClientException e) {
                this.logger.warn("OwnerId: {}, Unexpected SDK error while writing lock file: {}", new Object[]{this.ownerId, this.lockFilePath, e});
                if (isLockRenewal) break block3;
                throw e;
            }
        }
        return Pair.of((Object)result, (Object)Option.empty());
    }

    private StorageLockFile createOrUpdateLockFileInternal(StorageLockData lockData, String expectedEtag) {
        byte[] bytes = StorageLockFile.toByteArray((StorageLockData)lockData);
        PutObjectRequest.Builder putRequestBuilder = PutObjectRequest.builder().bucket(this.bucketName).key(this.lockFilePath);
        if (expectedEtag == null) {
            putRequestBuilder.ifNoneMatch("*");
        } else {
            putRequestBuilder.ifMatch(expectedEtag);
        }
        PutObjectResponse response = this.s3Client.putObject((PutObjectRequest)putRequestBuilder.build(), RequestBody.fromBytes((byte[])bytes));
        String newEtag = response.eTag();
        return new StorageLockFile(lockData, newEtag);
    }

    private LockUpsertResult handleUpsertS3Exception(S3Exception e) {
        int status = e.statusCode();
        if (status == 412) {
            this.logger.warn("OwnerId: {}, Lockfile modified by another process: {}", (Object)this.ownerId, (Object)this.lockFilePath);
            return LockUpsertResult.ACQUIRED_BY_OTHERS;
        }
        if (status == 409) {
            this.logger.warn("OwnerId: {}, Retriable conditional request conflict error: {}", (Object)this.ownerId, (Object)this.lockFilePath);
        } else if (status == 429) {
            this.logger.warn("OwnerId: {}, Rate limit exceeded for: {}", (Object)this.ownerId, (Object)this.lockFilePath);
        } else if (status >= 500) {
            this.logger.warn("OwnerId: {}, internal server error for: {}", new Object[]{this.ownerId, this.lockFilePath, e});
        } else {
            this.logger.error("OwnerId: {}, Error writing lock file: {}", new Object[]{this.ownerId, this.lockFilePath, e});
        }
        return LockUpsertResult.UNKNOWN_ERROR;
    }

    private static Functions.Function2<String, Properties, S3Client> createDefaultS3Client() {
        return (Functions.Function2 & Serializable)(bucketName, props) -> {
            GetBucketLocationResponse bucketLocationResponse;
            String regionString;
            Region region;
            boolean requiredFallbackRegion = false;
            try {
                region = DefaultAwsRegionProviderChain.builder().build().getRegion();
            }
            catch (SdkClientException e) {
                region = Region.US_EAST_1;
                requiredFallbackRegion = true;
            }
            long validityTimeoutSecs = (Long)props.getOrDefault((Object)StorageBasedLockConfig.VALIDITY_TIMEOUT_SECONDS.key(), StorageBasedLockConfig.VALIDITY_TIMEOUT_SECONDS.defaultValue());
            long s3CallTimeoutSecs = validityTimeoutSecs / 5L;
            S3Client s3Client = S3StorageLockClient.createS3Client(region, s3CallTimeoutSecs, props);
            if (requiredFallbackRegion && !StringUtils.isNullOrEmpty((String)(regionString = (bucketLocationResponse = s3Client.getBucketLocation((GetBucketLocationRequest)GetBucketLocationRequest.builder().bucket(bucketName).build())).locationConstraintAsString()))) {
                s3Client.close();
                return S3StorageLockClient.createS3Client(Region.of((String)regionString), s3CallTimeoutSecs, props);
            }
            return s3Client;
        };
    }

    private static S3Client createS3Client(Region region, long timeoutSecs, Properties props) {
        return (S3Client)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)S3Client.builder().overrideConfiguration(b -> b.apiCallTimeout(Duration.ofSeconds(timeoutSecs)))).credentialsProvider(HoodieAWSCredentialsProviderFactory.getAwsCredentialsProvider(props))).region(region)).build();
    }

    public void close() {
        this.s3Client.close();
    }
}

