/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.table;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.avro.Schema;
import org.apache.avro.generic.IndexedRecord;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.avro.model.HoodieArchivedMetaEntry;
import org.apache.hudi.client.utils.MetadataConversionUtils;
import org.apache.hudi.common.config.HoodieMetadataConfig;
import org.apache.hudi.common.config.SerializableConfiguration;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.HoodieAvroPayload;
import org.apache.hudi.common.model.HoodieFailedWritesCleaningPolicy;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.table.view.FileSystemViewStorageConfig;
import org.apache.hudi.common.util.CollectionUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieCommitException;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.hudi.table.HoodieTable;
import org.apache.hudi.table.marker.WriteMarkers;
import org.apache.hudi.table.marker.WriteMarkersFactory;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieTimelineArchiveLog<T extends HoodieAvroPayload, I, K, O> {
    private static final Logger LOG = LogManager.getLogger(HoodieTimelineArchiveLog.class);
    private final Path archiveFilePath;
    private final HoodieWriteConfig config;
    private HoodieLogFormat.Writer writer;
    private final int maxInstantsToKeep;
    private final int minInstantsToKeep;
    private final HoodieTable<T, I, K, O> table;
    private final HoodieTableMetaClient metaClient;

    public HoodieTimelineArchiveLog(HoodieWriteConfig config, HoodieTable<T, I, K, O> table) {
        this.config = config;
        this.table = table;
        this.metaClient = table.getMetaClient();
        this.archiveFilePath = HoodieArchivedTimeline.getArchiveLogPath((String)this.metaClient.getArchivePath());
        this.maxInstantsToKeep = config.getMaxCommitsToKeep();
        this.minInstantsToKeep = config.getMinCommitsToKeep();
    }

    private HoodieLogFormat.Writer openWriter() {
        try {
            if (this.writer == null) {
                return HoodieLogFormat.newWriterBuilder().onParentPath(this.archiveFilePath.getParent()).withFileId(this.archiveFilePath.getName()).withFileExtension(".archive").withFs((FileSystem)this.metaClient.getFs()).overBaseCommit("").build();
            }
            return this.writer;
        }
        catch (IOException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", (Throwable)e);
        }
    }

    private void close() {
        try {
            if (this.writer != null) {
                this.writer.close();
            }
        }
        catch (IOException e) {
            throw new HoodieException("Unable to close HoodieLogFormat writer", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean archiveIfRequired(HoodieEngineContext context) throws IOException {
        try {
            List<HoodieInstant> instantsToArchive = this.getInstantsToArchive().collect(Collectors.toList());
            boolean success = true;
            if (!instantsToArchive.isEmpty()) {
                this.writer = this.openWriter();
                LOG.info((Object)("Archiving instants " + instantsToArchive));
                this.archive(context, instantsToArchive);
                LOG.info((Object)("Deleting archived instants " + instantsToArchive));
                success = this.deleteArchivedInstants(instantsToArchive, context);
            } else {
                LOG.info((Object)"No Instants to archive");
            }
            boolean bl = success;
            return bl;
        }
        finally {
            this.close();
        }
    }

    private Stream<HoodieInstant> getCleanInstantsToArchive() {
        HoodieTimeline cleanAndRollbackTimeline = this.table.getActiveTimeline().getTimelineOfActions(CollectionUtils.createSet((Object[])new String[]{"clean", "rollback"})).filterCompletedInstants();
        return cleanAndRollbackTimeline.getInstants().collect(Collectors.groupingBy(HoodieInstant::getAction)).values().stream().map(hoodieInstants -> {
            if (hoodieInstants.size() > this.maxInstantsToKeep) {
                return hoodieInstants.subList(0, hoodieInstants.size() - this.minInstantsToKeep);
            }
            return new ArrayList();
        }).flatMap(Collection::stream);
    }

    private Stream<HoodieInstant> getCommitInstantsToArchive() {
        HoodieTimeline commitTimeline = this.table.getCompletedCommitsTimeline();
        Option oldestPendingCompactionInstant = this.table.getActiveTimeline().filterPendingCompactionTimeline().firstInstant();
        Option oldestInflightCommitInstant = this.table.getActiveTimeline().getTimelineOfActions(CollectionUtils.createSet((Object[])new String[]{"commit", "deltacommit"})).filterInflights().firstInstant();
        Option firstSavepoint = this.table.getCompletedSavepointTimeline().firstInstant();
        if (!commitTimeline.empty() && commitTimeline.countInstants() > this.maxInstantsToKeep) {
            Stream<HoodieInstant> instantToArchiveStream = commitTimeline.getInstants().filter(s -> !firstSavepoint.isPresent() || !HoodieTimeline.compareTimestamps((String)((HoodieInstant)firstSavepoint.get()).getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN_OR_EQUALS, (String)s.getTimestamp())).filter(s -> (Boolean)oldestPendingCompactionInstant.map(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.GREATER_THAN, (String)s.getTimestamp())).orElse((Object)true));
            if (this.config.getFailedWritesCleanPolicy() == HoodieFailedWritesCleaningPolicy.LAZY) {
                instantToArchiveStream = instantToArchiveStream.filter(s -> (Boolean)oldestInflightCommitInstant.map(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.GREATER_THAN, (String)s.getTimestamp())).orElse((Object)true));
            }
            return instantToArchiveStream.limit(commitTimeline.countInstants() - this.minInstantsToKeep);
        }
        return Stream.empty();
    }

    private Stream<HoodieInstant> getInstantsToArchive() {
        Stream<Object> instants = Stream.concat(this.getCleanInstantsToArchive(), this.getCommitInstantsToArchive());
        HoodieActiveTimeline rawActiveTimeline = new HoodieActiveTimeline(this.metaClient, false);
        Map<Pair, List<HoodieInstant>> groupByTsAction = rawActiveTimeline.getInstants().collect(Collectors.groupingBy(i -> Pair.of((Object)i.getTimestamp(), (Object)HoodieInstant.getComparableAction((String)i.getAction()))));
        if (this.config.isMetadataTableEnabled()) {
            try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create((HoodieEngineContext)this.table.getContext(), (HoodieMetadataConfig)this.config.getMetadataConfig(), (String)this.config.getBasePath(), (String)((String)FileSystemViewStorageConfig.SPILLABLE_DIR.defaultValue()));){
                Option latestCompactionTime = tableMetadata.getLatestCompactionTime();
                if (!latestCompactionTime.isPresent()) {
                    LOG.info((Object)"Not archiving as there is no compaction yet on the metadata table");
                    instants = Stream.empty();
                } else {
                    LOG.info((Object)("Limiting archiving of instants to latest compaction on metadata table at " + (String)latestCompactionTime.get()));
                    instants = instants.filter(instant -> HoodieTimeline.compareTimestamps((String)instant.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN, (String)((String)latestCompactionTime.get())));
                }
            }
            catch (Exception e) {
                throw new HoodieException("Error limiting instant archival based on metadata table", (Throwable)e);
            }
        }
        return instants.flatMap(hoodieInstant -> ((List)groupByTsAction.get(Pair.of((Object)hoodieInstant.getTimestamp(), (Object)HoodieInstant.getComparableAction((String)hoodieInstant.getAction())))).stream());
    }

    private boolean deleteArchivedInstants(List<HoodieInstant> archivedInstants, HoodieEngineContext context) throws IOException {
        LOG.info((Object)("Deleting instants " + archivedInstants));
        boolean success = true;
        List instantFiles = archivedInstants.stream().map(archivedInstant -> new Path(this.metaClient.getMetaPath(), archivedInstant.getFileName())).map(Path::toString).collect(Collectors.toList());
        context.setJobStatus(this.getClass().getSimpleName(), "Delete archived instants");
        Map resultDeleteInstantFiles = FSUtils.parallelizeFilesProcess((HoodieEngineContext)context, (FileSystem)this.metaClient.getFs(), (int)this.config.getArchiveDeleteParallelism(), (FSUtils.SerializableFunction & Serializable)pairOfSubPathAndConf -> {
            Path commitFile = new Path((String)pairOfSubPathAndConf.getKey());
            try {
                FileSystem fs = commitFile.getFileSystem(((SerializableConfiguration)pairOfSubPathAndConf.getValue()).get());
                if (fs.exists(commitFile)) {
                    return fs.delete(commitFile, false);
                }
                return true;
            }
            catch (IOException e) {
                throw new HoodieIOException("Failed to delete archived instant " + commitFile, e);
            }
        }, instantFiles);
        for (Map.Entry result : resultDeleteInstantFiles.entrySet()) {
            LOG.info((Object)("Archived and deleted instant file " + (String)result.getKey() + " : " + result.getValue()));
            success &= ((Boolean)result.getValue()).booleanValue();
        }
        Option latestCommitted = Option.fromJavaOptional(archivedInstants.stream().filter(i -> i.isCompleted() && (i.getAction().equals("commit") || i.getAction().equals("deltacommit"))).max(Comparator.comparing(HoodieInstant::getTimestamp)));
        LOG.info((Object)("Latest Committed Instant=" + latestCommitted));
        if (latestCommitted.isPresent()) {
            success &= this.deleteAllInstantsOlderorEqualsInAuxMetaFolder((HoodieInstant)latestCommitted.get());
        }
        return success;
    }

    private boolean deleteAllInstantsOlderorEqualsInAuxMetaFolder(HoodieInstant thresholdInstant) throws IOException {
        List instants = null;
        boolean success = true;
        try {
            instants = this.metaClient.scanHoodieInstantsFromFileSystem(new Path(this.metaClient.getMetaAuxiliaryPath()), HoodieActiveTimeline.VALID_EXTENSIONS_IN_ACTIVE_TIMELINE, false);
        }
        catch (FileNotFoundException e) {
            LOG.warn((Object)("Aux path not found. Skipping: " + this.metaClient.getMetaAuxiliaryPath()));
            return success;
        }
        List instantsToBeDeleted = instants.stream().filter(instant1 -> HoodieTimeline.compareTimestamps((String)instant1.getTimestamp(), (BiPredicate)HoodieTimeline.LESSER_THAN_OR_EQUALS, (String)thresholdInstant.getTimestamp())).collect(Collectors.toList());
        for (HoodieInstant deleteInstant : instantsToBeDeleted) {
            LOG.info((Object)("Deleting instant " + deleteInstant + " in auxiliary meta path " + this.metaClient.getMetaAuxiliaryPath()));
            Path metaFile = new Path(this.metaClient.getMetaAuxiliaryPath(), deleteInstant.getFileName());
            if (!this.metaClient.getFs().exists(metaFile)) continue;
            success &= this.metaClient.getFs().delete(metaFile, false);
            LOG.info((Object)("Deleted instant file in auxiliary metapath : " + metaFile));
        }
        return success;
    }

    public void archive(HoodieEngineContext context, List<HoodieInstant> instants) throws HoodieCommitException {
        try {
            Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
            LOG.info((Object)("Wrapper schema " + wrapperSchema.toString()));
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (HoodieInstant hoodieInstant : instants) {
                try {
                    this.deleteAnyLeftOverMarkers(context, hoodieInstant);
                    records.add(this.convertToAvroRecord(hoodieInstant));
                    if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                    this.writeToFile(wrapperSchema, records);
                }
                catch (Exception e) {
                    LOG.error((Object)("Failed to archive commits, .commit file: " + hoodieInstant.getFileName()), (Throwable)e);
                    if (!this.config.isFailOnTimelineArchivingEnabled()) continue;
                    throw e;
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to archive commits", e);
        }
    }

    private void deleteAnyLeftOverMarkers(HoodieEngineContext context, HoodieInstant instant) {
        WriteMarkers writeMarkers = WriteMarkersFactory.get(this.config.getMarkersType(), this.table, instant.getTimestamp());
        if (writeMarkers.deleteMarkerDir(context, this.config.getMarkersDeleteParallelism())) {
            LOG.info((Object)("Cleaned up left over marker directory for instant :" + instant));
        }
    }

    private void writeToFile(Schema wrapperSchema, List<IndexedRecord> records) throws Exception {
        if (records.size() > 0) {
            HashMap<HoodieLogBlock.HeaderMetadataType, String> header = new HashMap<HoodieLogBlock.HeaderMetadataType, String>();
            header.put(HoodieLogBlock.HeaderMetadataType.SCHEMA, wrapperSchema.toString());
            String keyField = this.table.getMetaClient().getTableConfig().getRecordKeyFieldProp();
            HoodieAvroDataBlock block = new HoodieAvroDataBlock(records, header, keyField);
            this.writer.appendBlock((HoodieLogBlock)block);
            records.clear();
        }
    }

    private IndexedRecord convertToAvroRecord(HoodieInstant hoodieInstant) throws IOException {
        return MetadataConversionUtils.createMetaWrapper(hoodieInstant, this.metaClient);
    }
}

