/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.generic.IndexedRecord;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hudi.common.model.HoodieKey;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.model.HoodieRecord;
import org.apache.hudi.common.model.HoodieRecordPayload;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.log.HoodieLogFormatReader;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieCommandBlock;
import org.apache.hudi.common.table.log.block.HoodieDataBlock;
import org.apache.hudi.common.table.log.block.HoodieDeleteBlock;
import org.apache.hudi.common.table.log.block.HoodieHFileDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.util.SpillableMapUtils;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public abstract class AbstractHoodieLogRecordScanner {
    private static final Logger LOG = LogManager.getLogger(AbstractHoodieLogRecordScanner.class);
    protected final Schema readerSchema;
    private final String latestInstantTime;
    private final HoodieTableMetaClient hoodieTableMetaClient;
    private final String payloadClassFQN;
    protected final List<String> logFilePaths;
    private final boolean readBlocksLazily;
    private final boolean reverseReader;
    private final int bufferSize;
    private final FileSystem fs;
    private AtomicLong totalLogFiles = new AtomicLong(0L);
    private AtomicLong totalLogBlocks = new AtomicLong(0L);
    private AtomicLong totalLogRecords = new AtomicLong(0L);
    private AtomicLong totalRollbacks = new AtomicLong(0L);
    private AtomicLong totalCorruptBlocks = new AtomicLong(0L);
    private Deque<HoodieLogBlock> currentInstantLogBlocks = new ArrayDeque<HoodieLogBlock>();
    private float progress = 0.0f;

    public AbstractHoodieLogRecordScanner(FileSystem fs, String basePath, List<String> logFilePaths, Schema readerSchema, String latestInstantTime, boolean readBlocksLazily, boolean reverseReader, int bufferSize) {
        this.readerSchema = readerSchema;
        this.latestInstantTime = latestInstantTime;
        this.hoodieTableMetaClient = new HoodieTableMetaClient(fs.getConf(), basePath);
        this.payloadClassFQN = this.hoodieTableMetaClient.getTableConfig().getPayloadClass();
        this.totalLogFiles.addAndGet(logFilePaths.size());
        this.logFilePaths = logFilePaths;
        this.readBlocksLazily = readBlocksLazily;
        this.reverseReader = reverseReader;
        this.fs = fs;
        this.bufferSize = bufferSize;
    }

    public void scan() {
        HoodieLogFormatReader logFormatReaderWrapper = null;
        try {
            logFormatReaderWrapper = new HoodieLogFormatReader(this.fs, this.logFilePaths.stream().map(logFile -> new HoodieLogFile(new Path(logFile))).collect(Collectors.toList()), this.readerSchema, this.readBlocksLazily, this.reverseReader, this.bufferSize);
            HashSet<HoodieLogFile> scannedLogFiles = new HashSet<HoodieLogFile>();
            block19: while (logFormatReaderWrapper.hasNext()) {
                HoodieLogFile logFile2 = logFormatReaderWrapper.getLogFile();
                LOG.info((Object)("Scanning log file " + logFile2));
                scannedLogFiles.add(logFile2);
                this.totalLogFiles.set(scannedLogFiles.size());
                HoodieLogBlock r = logFormatReaderWrapper.next();
                this.totalLogBlocks.incrementAndGet();
                if (r.getBlockType() != HoodieLogBlock.HoodieLogBlockType.CORRUPT_BLOCK && !HoodieTimeline.compareTimestamps(r.getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME), HoodieTimeline.LESSER_THAN_OR_EQUALS, this.latestInstantTime)) break;
                switch (r.getBlockType()) {
                    case HFILE_DATA_BLOCK: 
                    case AVRO_DATA_BLOCK: {
                        LOG.info((Object)("Reading a data block from file " + logFile2.getPath() + " at instant " + r.getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME)));
                        if (this.isNewInstantBlock(r) && !this.readBlocksLazily) {
                            this.processQueuedBlocksForInstant(this.currentInstantLogBlocks, scannedLogFiles.size());
                        }
                        this.currentInstantLogBlocks.push(r);
                        continue block19;
                    }
                    case DELETE_BLOCK: {
                        LOG.info((Object)("Reading a delete block from file " + logFile2.getPath()));
                        if (this.isNewInstantBlock(r) && !this.readBlocksLazily) {
                            this.processQueuedBlocksForInstant(this.currentInstantLogBlocks, scannedLogFiles.size());
                        }
                        this.currentInstantLogBlocks.push(r);
                        continue block19;
                    }
                    case COMMAND_BLOCK: {
                        LOG.info((Object)("Reading a command block from file " + logFile2.getPath()));
                        HoodieCommandBlock commandBlock = (HoodieCommandBlock)r;
                        String targetInstantForCommandBlock = r.getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.TARGET_INSTANT_TIME);
                        switch (commandBlock.getType()) {
                            case ROLLBACK_PREVIOUS_BLOCK: {
                                int numBlocksRolledBack = 0;
                                this.totalRollbacks.incrementAndGet();
                                while (!this.currentInstantLogBlocks.isEmpty()) {
                                    HoodieLogBlock lastBlock = this.currentInstantLogBlocks.peek();
                                    if (lastBlock.getBlockType() == HoodieLogBlock.HoodieLogBlockType.CORRUPT_BLOCK) {
                                        LOG.info((Object)("Rolling back the last corrupted log block read in " + logFile2.getPath()));
                                        this.currentInstantLogBlocks.pop();
                                        ++numBlocksRolledBack;
                                        continue;
                                    }
                                    if (targetInstantForCommandBlock.contentEquals(lastBlock.getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME))) {
                                        LOG.info((Object)("Rolling back the last log block read in " + logFile2.getPath()));
                                        this.currentInstantLogBlocks.pop();
                                        ++numBlocksRolledBack;
                                        continue;
                                    }
                                    if (!targetInstantForCommandBlock.contentEquals(this.currentInstantLogBlocks.peek().getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME))) {
                                        LOG.warn((Object)("TargetInstantTime " + targetInstantForCommandBlock + " invalid or extra rollback command block in " + logFile2.getPath()));
                                        break;
                                    }
                                    LOG.warn((Object)("Unable to apply rollback command block in " + logFile2.getPath()));
                                }
                                LOG.info((Object)("Number of applied rollback blocks " + numBlocksRolledBack));
                                continue block19;
                            }
                        }
                        throw new UnsupportedOperationException("Command type not yet supported.");
                    }
                    case CORRUPT_BLOCK: {
                        LOG.info((Object)("Found a corrupt block in " + logFile2.getPath()));
                        this.totalCorruptBlocks.incrementAndGet();
                        this.currentInstantLogBlocks.push(r);
                        continue block19;
                    }
                }
                throw new UnsupportedOperationException("Block type not supported yet");
            }
            if (!this.currentInstantLogBlocks.isEmpty()) {
                LOG.info((Object)"Merging the final data blocks");
                this.processQueuedBlocksForInstant(this.currentInstantLogBlocks, scannedLogFiles.size());
            }
            this.progress = 1.0f;
        }
        catch (IOException e) {
            LOG.error((Object)"Got IOException when reading log file", (Throwable)e);
            throw new HoodieIOException("IOException when reading log file ", e);
        }
        catch (Exception e) {
            LOG.error((Object)"Got exception when reading log file", (Throwable)e);
            throw new HoodieException("Exception when reading log file ", e);
        }
        finally {
            try {
                if (null != logFormatReaderWrapper) {
                    logFormatReaderWrapper.close();
                }
            }
            catch (IOException ioe) {
                LOG.error((Object)"Unable to close log format reader", (Throwable)ioe);
            }
        }
    }

    private boolean isNewInstantBlock(HoodieLogBlock logBlock) {
        return this.currentInstantLogBlocks.size() > 0 && this.currentInstantLogBlocks.peek().getBlockType() != HoodieLogBlock.HoodieLogBlockType.CORRUPT_BLOCK && !logBlock.getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME).contentEquals(this.currentInstantLogBlocks.peek().getLogBlockHeader().get((Object)HoodieLogBlock.HeaderMetadataType.INSTANT_TIME));
    }

    private void processDataBlock(HoodieDataBlock dataBlock) throws Exception {
        List<IndexedRecord> recs = dataBlock.getRecords();
        this.totalLogRecords.addAndGet(recs.size());
        for (IndexedRecord rec : recs) {
            this.processNextRecord(this.createHoodieRecord(rec));
        }
    }

    protected HoodieRecord<?> createHoodieRecord(IndexedRecord rec) {
        return (HoodieRecord)SpillableMapUtils.convertToHoodieRecordPayload((GenericRecord)rec, this.payloadClassFQN);
    }

    protected abstract void processNextRecord(HoodieRecord<? extends HoodieRecordPayload> var1) throws Exception;

    protected abstract void processNextDeletedKey(HoodieKey var1);

    private void processQueuedBlocksForInstant(Deque<HoodieLogBlock> lastBlocks, int numLogFilesSeen) throws Exception {
        while (!lastBlocks.isEmpty()) {
            LOG.info((Object)("Number of remaining logblocks to merge " + lastBlocks.size()));
            HoodieLogBlock lastBlock = lastBlocks.pollLast();
            switch (lastBlock.getBlockType()) {
                case AVRO_DATA_BLOCK: {
                    this.processDataBlock((HoodieAvroDataBlock)lastBlock);
                    break;
                }
                case HFILE_DATA_BLOCK: {
                    this.processDataBlock((HoodieHFileDataBlock)lastBlock);
                    break;
                }
                case DELETE_BLOCK: {
                    Arrays.stream(((HoodieDeleteBlock)lastBlock).getKeysToDelete()).forEach(this::processNextDeletedKey);
                    break;
                }
                case CORRUPT_BLOCK: {
                    LOG.warn((Object)"Found a corrupt block which was not rolled back");
                    break;
                }
            }
        }
        this.progress = numLogFilesSeen - 1 / this.logFilePaths.size();
    }

    public float getProgress() {
        return this.progress;
    }

    public long getTotalLogFiles() {
        return this.totalLogFiles.get();
    }

    public long getTotalLogRecords() {
        return this.totalLogRecords.get();
    }

    public long getTotalLogBlocks() {
        return this.totalLogBlocks.get();
    }

    protected String getPayloadClassFQN() {
        return this.payloadClassFQN;
    }

    public long getTotalRollbacks() {
        return this.totalRollbacks.get();
    }

    public long getTotalCorruptBlocks() {
        return this.totalCorruptBlocks.get();
    }

    public static abstract class Builder {
        public abstract Builder withFileSystem(FileSystem var1);

        public abstract Builder withBasePath(String var1);

        public abstract Builder withLogFilePaths(List<String> var1);

        public abstract Builder withReaderSchema(Schema var1);

        public abstract Builder withLatestInstantTime(String var1);

        public abstract Builder withReadBlocksLazily(boolean var1);

        public abstract Builder withReverseReader(boolean var1);

        public abstract Builder withBufferSize(int var1);

        public abstract AbstractHoodieLogRecordScanner build();
    }
}

