/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.normalizer;

import io.hops.hudi.org.apache.hbase.thirdparty.org.apache.commons.collections4.CollectionUtils;
import java.time.Instant;
import java.time.Period;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.RegionMetrics;
import org.apache.hadoop.hbase.ServerMetrics;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.Size;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.MasterSwitchType;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.assignment.RegionStates;
import org.apache.hadoop.hbase.master.normalizer.MergeNormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.NormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.NormalizationTarget;
import org.apache.hadoop.hbase.master.normalizer.RegionNormalizer;
import org.apache.hadoop.hbase.master.normalizer.SplitNormalizationPlan;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"Configuration"})
class SimpleRegionNormalizer
implements RegionNormalizer,
ConfigurationObserver {
    private static final Logger LOG = LoggerFactory.getLogger(SimpleRegionNormalizer.class);
    static final String SPLIT_ENABLED_KEY = "hbase.normalizer.split.enabled";
    static final boolean DEFAULT_SPLIT_ENABLED = true;
    static final String MERGE_ENABLED_KEY = "hbase.normalizer.merge.enabled";
    static final boolean DEFAULT_MERGE_ENABLED = true;
    static final String MIN_REGION_COUNT_KEY = "hbase.normalizer.min.region.count";
    static final int DEFAULT_MIN_REGION_COUNT = 3;
    static final String MERGE_MIN_REGION_AGE_DAYS_KEY = "hbase.normalizer.merge.min_region_age.days";
    static final int DEFAULT_MERGE_MIN_REGION_AGE_DAYS = 3;
    static final String MERGE_MIN_REGION_SIZE_MB_KEY = "hbase.normalizer.merge.min_region_size.mb";
    static final int DEFAULT_MERGE_MIN_REGION_SIZE_MB = 1;
    private MasterServices masterServices = null;
    private NormalizerConfiguration normalizerConfiguration = new NormalizerConfiguration();

    public Configuration getConf() {
        return this.normalizerConfiguration.getConf();
    }

    public void setConf(Configuration conf) {
        if (conf == null) {
            return;
        }
        this.normalizerConfiguration = new NormalizerConfiguration(conf, this.normalizerConfiguration);
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        LOG.debug("Updating configuration parameters according to new configuration instance.");
        this.setConf(conf);
    }

    private static int parseMinRegionCount(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MIN_REGION_COUNT_KEY, 3);
        if (parsedValue != (settledValue = Math.max(1, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MIN_REGION_COUNT_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private static Period parseMergeMinRegionAge(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MERGE_MIN_REGION_AGE_DAYS_KEY, 3);
        if (parsedValue != (settledValue = Math.max(0, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MERGE_MIN_REGION_AGE_DAYS_KEY, parsedValue, settledValue);
        }
        return Period.ofDays(settledValue);
    }

    private static long parseMergeMinRegionSizeMb(Configuration conf) {
        long settledValue;
        long parsedValue = conf.getLong(MERGE_MIN_REGION_SIZE_MB_KEY, 1L);
        if (parsedValue != (settledValue = Math.max(0L, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MERGE_MIN_REGION_SIZE_MB_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private static <T> void warnInvalidValue(String key, T parsedValue, T settledValue) {
        LOG.warn("Configured value {}={} is invalid. Setting value to {}.", new Object[]{key, parsedValue, settledValue});
    }

    private static <T> void logConfigurationUpdated(String key, T oldValue, T newValue) {
        if (!Objects.equals(oldValue, newValue)) {
            LOG.info("Updated configuration for key '{}' from {} to {}", new Object[]{key, oldValue, newValue});
        }
    }

    public boolean isSplitEnabled() {
        return this.normalizerConfiguration.isSplitEnabled();
    }

    public boolean isMergeEnabled() {
        return this.normalizerConfiguration.isMergeEnabled();
    }

    public int getMinRegionCount() {
        return this.normalizerConfiguration.getMinRegionCount();
    }

    public Period getMergeMinRegionAge() {
        return this.normalizerConfiguration.getMergeMinRegionAge();
    }

    public long getMergeMinRegionSizeMb() {
        return this.normalizerConfiguration.getMergeMinRegionSizeMb();
    }

    @Override
    public void setMasterServices(MasterServices masterServices) {
        this.masterServices = masterServices;
    }

    @Override
    public List<NormalizationPlan> computePlansForTable(TableDescriptor tableDescriptor) {
        if (tableDescriptor == null) {
            return Collections.emptyList();
        }
        TableName table = tableDescriptor.getTableName();
        if (table.isSystemTable()) {
            LOG.debug("Normalization of system table {} isn't allowed", (Object)table);
            return Collections.emptyList();
        }
        boolean proceedWithSplitPlanning = this.proceedWithSplitPlanning(tableDescriptor);
        boolean proceedWithMergePlanning = this.proceedWithMergePlanning(tableDescriptor);
        if (!proceedWithMergePlanning && !proceedWithSplitPlanning) {
            LOG.debug("Both split and merge are disabled. Skipping normalization of table: {}", (Object)table);
            return Collections.emptyList();
        }
        NormalizeContext ctx = new NormalizeContext(tableDescriptor);
        if (CollectionUtils.isEmpty(ctx.getTableRegions())) {
            return Collections.emptyList();
        }
        LOG.debug("Computing normalization plan for table:  {}, number of regions: {}", (Object)table, (Object)ctx.getTableRegions().size());
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        int splitPlansCount = 0;
        if (proceedWithSplitPlanning) {
            List<NormalizationPlan> splitPlans = this.computeSplitNormalizationPlans(ctx);
            splitPlansCount = splitPlans.size();
            plans.addAll(splitPlans);
        }
        int mergePlansCount = 0;
        if (proceedWithMergePlanning) {
            List<NormalizationPlan> mergePlans = this.computeMergeNormalizationPlans(ctx);
            mergePlansCount = mergePlans.size();
            plans.addAll(mergePlans);
        }
        LOG.debug("Computed normalization plans for table {}. Total plans: {}, split plans: {}, merge plans: {}", new Object[]{table, plans.size(), splitPlansCount, mergePlansCount});
        return plans;
    }

    private long getRegionSizeMB(RegionInfo hri) {
        ServerName sn = this.masterServices.getAssignmentManager().getRegionStates().getRegionServerOfRegion(hri);
        if (sn == null) {
            LOG.debug("{} region was not found on any Server", (Object)hri.getRegionNameAsString());
            return -1L;
        }
        ServerMetrics serverMetrics = this.masterServices.getServerManager().getLoad(sn);
        if (serverMetrics == null) {
            LOG.debug("server {} was not found in ServerManager", (Object)sn.getServerName());
            return -1L;
        }
        RegionMetrics regionLoad = serverMetrics.getRegionMetrics().get(hri.getRegionName());
        if (regionLoad == null) {
            LOG.debug("{} was not found in RegionsLoad", (Object)hri.getRegionNameAsString());
            return -1L;
        }
        return (long)regionLoad.getStoreFileSize().get(Size.Unit.MEGABYTE);
    }

    private boolean isMasterSwitchEnabled(MasterSwitchType masterSwitchType) {
        return this.masterServices.isSplitOrMergeEnabled(masterSwitchType);
    }

    private boolean proceedWithSplitPlanning(TableDescriptor tableDescriptor) {
        String value = tableDescriptor.getValue(SPLIT_ENABLED_KEY);
        return (value == null ? this.isSplitEnabled() : Boolean.parseBoolean(value)) && this.isMasterSwitchEnabled(MasterSwitchType.SPLIT);
    }

    private boolean proceedWithMergePlanning(TableDescriptor tableDescriptor) {
        String value = tableDescriptor.getValue(MERGE_ENABLED_KEY);
        return (value == null ? this.isMergeEnabled() : Boolean.parseBoolean(value)) && this.isMasterSwitchEnabled(MasterSwitchType.MERGE);
    }

    private double getAverageRegionSizeMb(List<RegionInfo> tableRegions, TableDescriptor tableDescriptor) {
        double avgRegionSize;
        if (CollectionUtils.isEmpty(tableRegions)) {
            throw new IllegalStateException("Cannot calculate average size of a table without any regions.");
        }
        TableName table = tableDescriptor.getTableName();
        int targetRegionCount = tableDescriptor.getNormalizerTargetRegionCount();
        long targetRegionSize = tableDescriptor.getNormalizerTargetRegionSize();
        LOG.debug("Table {} configured with target region count {}, target region size {} MB", new Object[]{table, targetRegionCount, targetRegionSize});
        if (targetRegionSize > 0L) {
            avgRegionSize = targetRegionSize;
        } else {
            int regionCount = tableRegions.size();
            long totalSizeMb = tableRegions.stream().mapToLong(this::getRegionSizeMB).sum();
            avgRegionSize = targetRegionCount > 0 ? (double)totalSizeMb / (double)targetRegionCount : (double)totalSizeMb / (double)regionCount;
            LOG.debug("Table {}, total aggregated regions size: {} MB and average region size {} MB", new Object[]{table, totalSizeMb, String.format("%.3f", avgRegionSize)});
        }
        return avgRegionSize;
    }

    private boolean skipForMerge(NormalizerConfiguration normalizerConfiguration, NormalizeContext ctx, RegionInfo regionInfo) {
        RegionState state = ctx.getRegionStates().getRegionState(regionInfo);
        String name2 = regionInfo.getEncodedName();
        return SimpleRegionNormalizer.logTraceReason(() -> state == null, "skipping merge of region {} because no state information is available.", name2) || SimpleRegionNormalizer.logTraceReason(() -> !Objects.equals((Object)state.getState(), (Object)RegionState.State.OPEN), "skipping merge of region {} because it is not open.", name2) || SimpleRegionNormalizer.logTraceReason(() -> !SimpleRegionNormalizer.isOldEnoughForMerge(normalizerConfiguration, ctx, regionInfo), "skipping merge of region {} because it is not old enough.", name2) || SimpleRegionNormalizer.logTraceReason(() -> !this.isLargeEnoughForMerge(normalizerConfiguration, ctx, regionInfo), "skipping merge region {} because it is not large enough.", name2);
    }

    private List<NormalizationPlan> computeMergeNormalizationPlans(NormalizeContext ctx) {
        NormalizerConfiguration configuration = this.normalizerConfiguration;
        if (ctx.getTableRegions().size() < configuration.getMinRegionCount(ctx)) {
            LOG.debug("Table {} has {} regions, required min number of regions for normalizer to run is {}, not computing merge plans.", new Object[]{ctx.getTableName(), ctx.getTableRegions().size(), configuration.getMinRegionCount()});
            return Collections.emptyList();
        }
        long avgRegionSizeMb = (long)ctx.getAverageRegionSizeMb();
        if (avgRegionSizeMb < configuration.getMergeMinRegionSizeMb(ctx)) {
            return Collections.emptyList();
        }
        LOG.debug("Computing normalization plan for table {}. average region size: {} MB, number of regions: {}.", new Object[]{ctx.getTableName(), avgRegionSizeMb, ctx.getTableRegions().size()});
        LinkedList<NormalizationPlan> plans = new LinkedList<NormalizationPlan>();
        LinkedList<NormalizationTarget> rangeMembers = new LinkedList<NormalizationTarget>();
        int current = 0;
        int rangeStart = 0;
        while (rangeStart < ctx.getTableRegions().size() - 1 && current < ctx.getTableRegions().size()) {
            rangeMembers.clear();
            long sumRangeMembersSizeMb = 0L;
            for (current = rangeStart; current < ctx.getTableRegions().size(); ++current) {
                RegionInfo regionInfo = ctx.getTableRegions().get(current);
                long regionSizeMb = this.getRegionSizeMB(regionInfo);
                if (this.skipForMerge(configuration, ctx, regionInfo)) {
                    rangeStart = Math.max(current, rangeStart + 1);
                    break;
                }
                if (rangeMembers.isEmpty() || rangeMembers.size() == 1 && sumRangeMembersSizeMb == 0L || regionSizeMb == 0L || regionSizeMb + sumRangeMembersSizeMb <= avgRegionSizeMb) {
                    rangeMembers.add(new NormalizationTarget(regionInfo, regionSizeMb));
                    sumRangeMembersSizeMb += regionSizeMb;
                    continue;
                }
                rangeStart = Math.max(current, rangeStart + 1);
                break;
            }
            if (rangeMembers.size() <= 1) continue;
            plans.add(new MergeNormalizationPlan.Builder().setTargets(rangeMembers).build());
        }
        return plans;
    }

    private static boolean skipForSplit(RegionState state, RegionInfo regionInfo) {
        String name2 = regionInfo.getEncodedName();
        return SimpleRegionNormalizer.logTraceReason(() -> state == null, "skipping split of region {} because no state information is available.", name2) || SimpleRegionNormalizer.logTraceReason(() -> !Objects.equals((Object)state.getState(), (Object)RegionState.State.OPEN), "skipping merge of region {} because it is not open.", name2);
    }

    private List<NormalizationPlan> computeSplitNormalizationPlans(NormalizeContext ctx) {
        double avgRegionSize = ctx.getAverageRegionSizeMb();
        LOG.debug("Table {}, average region size: {} MB", (Object)ctx.getTableName(), (Object)String.format("%.3f", avgRegionSize));
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        for (RegionInfo hri : ctx.getTableRegions()) {
            long regionSizeMb;
            if (SimpleRegionNormalizer.skipForSplit(ctx.getRegionStates().getRegionState(hri), hri) || !((double)(regionSizeMb = this.getRegionSizeMB(hri)) > 2.0 * avgRegionSize)) continue;
            LOG.info("Table {}, large region {} has size {} MB, more than twice avg size {} MB, splitting", new Object[]{ctx.getTableName(), hri.getRegionNameAsString(), regionSizeMb, String.format("%.3f", avgRegionSize)});
            plans.add(new SplitNormalizationPlan(hri, regionSizeMb));
        }
        return plans;
    }

    private static boolean isOldEnoughForMerge(NormalizerConfiguration normalizerConfiguration, NormalizeContext ctx, RegionInfo regionInfo) {
        Instant currentTime = Instant.ofEpochMilli(EnvironmentEdgeManager.currentTime());
        Instant regionCreateTime = Instant.ofEpochMilli(regionInfo.getRegionId());
        return currentTime.isAfter(regionCreateTime.plus(normalizerConfiguration.getMergeMinRegionAge(ctx)));
    }

    private boolean isLargeEnoughForMerge(NormalizerConfiguration normalizerConfiguration, NormalizeContext ctx, RegionInfo regionInfo) {
        return this.getRegionSizeMB(regionInfo) >= normalizerConfiguration.getMergeMinRegionSizeMb(ctx);
    }

    private static boolean logTraceReason(BooleanSupplier predicate, String fmtWhenTrue, Object ... args2) {
        boolean value = predicate.getAsBoolean();
        if (value) {
            LOG.trace(fmtWhenTrue, args2);
        }
        return value;
    }

    private class NormalizeContext {
        private final TableName tableName;
        private final RegionStates regionStates;
        private final List<RegionInfo> tableRegions;
        private final double averageRegionSizeMb;
        private final TableDescriptor tableDescriptor;

        public NormalizeContext(TableDescriptor tableDescriptor) {
            this.tableDescriptor = tableDescriptor;
            this.tableName = tableDescriptor.getTableName();
            this.regionStates = SimpleRegionNormalizer.this.masterServices.getAssignmentManager().getRegionStates();
            this.tableRegions = this.regionStates.getRegionsOfTable(this.tableName);
            this.tableRegions.sort(RegionInfo.COMPARATOR);
            this.averageRegionSizeMb = SimpleRegionNormalizer.this.getAverageRegionSizeMb(this.tableRegions, this.tableDescriptor);
        }

        public TableName getTableName() {
            return this.tableName;
        }

        public RegionStates getRegionStates() {
            return this.regionStates;
        }

        public List<RegionInfo> getTableRegions() {
            return this.tableRegions;
        }

        public double getAverageRegionSizeMb() {
            return this.averageRegionSizeMb;
        }

        public <T> T getOrDefault(String key, Function<String, T> function, T defaultValue) {
            String value = this.tableDescriptor.getValue(key);
            if (value == null) {
                return defaultValue;
            }
            return function.apply(value);
        }
    }

    private static final class NormalizerConfiguration {
        private final Configuration conf;
        private final boolean splitEnabled;
        private final boolean mergeEnabled;
        private final int minRegionCount;
        private final Period mergeMinRegionAge;
        private final long mergeMinRegionSizeMb;

        private NormalizerConfiguration() {
            this.conf = null;
            this.splitEnabled = true;
            this.mergeEnabled = true;
            this.minRegionCount = 3;
            this.mergeMinRegionAge = Period.ofDays(3);
            this.mergeMinRegionSizeMb = 1L;
        }

        private NormalizerConfiguration(Configuration conf, NormalizerConfiguration currentConfiguration) {
            this.conf = conf;
            this.splitEnabled = conf.getBoolean(SimpleRegionNormalizer.SPLIT_ENABLED_KEY, true);
            this.mergeEnabled = conf.getBoolean(SimpleRegionNormalizer.MERGE_ENABLED_KEY, true);
            this.minRegionCount = SimpleRegionNormalizer.parseMinRegionCount(conf);
            this.mergeMinRegionAge = SimpleRegionNormalizer.parseMergeMinRegionAge(conf);
            this.mergeMinRegionSizeMb = SimpleRegionNormalizer.parseMergeMinRegionSizeMb(conf);
            SimpleRegionNormalizer.logConfigurationUpdated(SimpleRegionNormalizer.SPLIT_ENABLED_KEY, currentConfiguration.isSplitEnabled(), this.splitEnabled);
            SimpleRegionNormalizer.logConfigurationUpdated(SimpleRegionNormalizer.MERGE_ENABLED_KEY, currentConfiguration.isMergeEnabled(), this.mergeEnabled);
            SimpleRegionNormalizer.logConfigurationUpdated(SimpleRegionNormalizer.MIN_REGION_COUNT_KEY, currentConfiguration.getMinRegionCount(), this.minRegionCount);
            SimpleRegionNormalizer.logConfigurationUpdated(SimpleRegionNormalizer.MERGE_MIN_REGION_AGE_DAYS_KEY, currentConfiguration.getMergeMinRegionAge(), this.mergeMinRegionAge);
            SimpleRegionNormalizer.logConfigurationUpdated(SimpleRegionNormalizer.MERGE_MIN_REGION_SIZE_MB_KEY, currentConfiguration.getMergeMinRegionSizeMb(), this.mergeMinRegionSizeMb);
        }

        public Configuration getConf() {
            return this.conf;
        }

        public boolean isSplitEnabled() {
            return this.splitEnabled;
        }

        public boolean isMergeEnabled() {
            return this.mergeEnabled;
        }

        public int getMinRegionCount() {
            return this.minRegionCount;
        }

        public int getMinRegionCount(NormalizeContext context) {
            int minRegionCount = context.getOrDefault(SimpleRegionNormalizer.MIN_REGION_COUNT_KEY, Integer::parseInt, 0);
            if (minRegionCount <= 0) {
                minRegionCount = this.getMinRegionCount();
            }
            return minRegionCount;
        }

        public Period getMergeMinRegionAge() {
            return this.mergeMinRegionAge;
        }

        public Period getMergeMinRegionAge(NormalizeContext context) {
            int mergeMinRegionAge = context.getOrDefault(SimpleRegionNormalizer.MERGE_MIN_REGION_AGE_DAYS_KEY, Integer::parseInt, -1);
            if (mergeMinRegionAge < 0) {
                return this.getMergeMinRegionAge();
            }
            return Period.ofDays(mergeMinRegionAge);
        }

        public long getMergeMinRegionSizeMb() {
            return this.mergeMinRegionSizeMb;
        }

        public long getMergeMinRegionSizeMb(NormalizeContext context) {
            long mergeMinRegionSizeMb = context.getOrDefault(SimpleRegionNormalizer.MERGE_MIN_REGION_SIZE_MB_KEY, Long::parseLong, -1L);
            if (mergeMinRegionSizeMb < 0L) {
                mergeMinRegionSizeMb = this.getMergeMinRegionSizeMb();
            }
            return mergeMinRegionSizeMb;
        }
    }
}

