/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.security.token;

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import javax.crypto.SecretKey;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Abortable;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.security.token.AuthenticationKey;
import org.apache.hadoop.hbase.security.token.AuthenticationTokenIdentifier;
import org.apache.hadoop.hbase.security.token.ZKSecretWatcher;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.zookeeper.ZKClusterId;
import org.apache.hadoop.hbase.zookeeper.ZKLeaderManager;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class AuthenticationTokenSecretManager
extends SecretManager<AuthenticationTokenIdentifier> {
    static final String NAME_PREFIX = "SecretManager-";
    private static final Logger LOG = LoggerFactory.getLogger(AuthenticationTokenSecretManager.class);
    private long lastKeyUpdate;
    private long keyUpdateInterval;
    private long tokenMaxLifetime;
    private ZKSecretWatcher zkWatcher;
    private LeaderElector leaderElector;
    private ZKClusterId clusterId;
    private Map<Integer, AuthenticationKey> allKeys = new ConcurrentHashMap<Integer, AuthenticationKey>();
    private AuthenticationKey currentKey;
    private int idSeq;
    private AtomicLong tokenSeq = new AtomicLong();
    private String name;

    public AuthenticationTokenSecretManager(Configuration conf, ZKWatcher zk, String serverName, long keyUpdateInterval, long tokenMaxLifetime) {
        this.zkWatcher = new ZKSecretWatcher(conf, zk, this);
        this.keyUpdateInterval = keyUpdateInterval;
        this.tokenMaxLifetime = tokenMaxLifetime;
        this.leaderElector = new LeaderElector(zk, serverName);
        this.name = NAME_PREFIX + serverName;
        this.clusterId = new ZKClusterId(zk, (Abortable)zk);
    }

    public void start() {
        try {
            this.zkWatcher.start();
            this.leaderElector.start();
        }
        catch (KeeperException ke) {
            LOG.error("ZooKeeper initialization failed", (Throwable)ke);
        }
    }

    public void stop() {
        this.leaderElector.stop("SecretManager stopping");
    }

    public boolean isMaster() {
        return this.leaderElector.isMaster();
    }

    public String getName() {
        return this.name;
    }

    protected synchronized byte[] createPassword(AuthenticationTokenIdentifier identifier) {
        long now = EnvironmentEdgeManager.currentTime();
        AuthenticationKey secretKey = this.currentKey;
        identifier.setKeyId(secretKey.getKeyId());
        identifier.setIssueDate(now);
        identifier.setExpirationDate(now + this.tokenMaxLifetime);
        identifier.setSequenceNumber(this.tokenSeq.getAndIncrement());
        return AuthenticationTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)secretKey.getKey());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] retrievePassword(AuthenticationTokenIdentifier identifier) throws SecretManager.InvalidToken {
        long now = EnvironmentEdgeManager.currentTime();
        if (identifier.getExpirationDate() < now) {
            throw new SecretManager.InvalidToken("Token has expired");
        }
        AuthenticationKey masterKey = this.allKeys.get(identifier.getKeyId());
        if (masterKey == null) {
            if (this.zkWatcher.getWatcher().isAborted()) {
                LOG.error("ZKWatcher is abort");
                throw new SecretManager.InvalidToken("Token keys could not be sync from zookeeper because of ZKWatcher abort");
            }
            AuthenticationTokenSecretManager authenticationTokenSecretManager = this;
            synchronized (authenticationTokenSecretManager) {
                if (!this.leaderElector.isAlive() || this.leaderElector.isStopped()) {
                    LOG.warn("Thread leaderElector[" + this.leaderElector.getName() + ":" + this.leaderElector.getId() + "] is stopped or not alive");
                    this.leaderElector.start();
                    LOG.info("Thread leaderElector [" + this.leaderElector.getName() + ":" + this.leaderElector.getId() + "] is started");
                }
            }
            this.zkWatcher.refreshKeys();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Sync token keys from zookeeper");
            }
            masterKey = this.allKeys.get(identifier.getKeyId());
        }
        if (masterKey == null) {
            throw new SecretManager.InvalidToken("Unknown master key for token (id=" + identifier.getKeyId() + ")");
        }
        return AuthenticationTokenSecretManager.createPassword((byte[])identifier.getBytes(), (SecretKey)masterKey.getKey());
    }

    public AuthenticationTokenIdentifier createIdentifier() {
        return new AuthenticationTokenIdentifier();
    }

    public Token<AuthenticationTokenIdentifier> generateToken(String username) {
        AuthenticationTokenIdentifier ident = new AuthenticationTokenIdentifier(username);
        Token token = new Token((TokenIdentifier)ident, (SecretManager)this);
        if (this.clusterId.hasId()) {
            token.setService(new Text(this.clusterId.getId()));
        }
        return token;
    }

    public synchronized void addKey(AuthenticationKey key) throws IOException {
        if (this.leaderElector.isMaster()) {
            LOG.debug("Running as master, ignoring new key {}", (Object)key);
            return;
        }
        LOG.debug("Adding key {}", (Object)key.getKeyId());
        this.allKeys.put(key.getKeyId(), key);
        if (this.currentKey == null || key.getKeyId() > this.currentKey.getKeyId()) {
            this.currentKey = key;
        }
        if (key.getKeyId() > this.idSeq) {
            this.idSeq = key.getKeyId();
        }
    }

    synchronized boolean removeKey(Integer keyId) {
        if (this.leaderElector.isMaster()) {
            LOG.debug("Running as master, ignoring removed keyid={}", (Object)keyId);
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Removing keyid={}", (Object)keyId);
        }
        this.allKeys.remove(keyId);
        return true;
    }

    synchronized AuthenticationKey getCurrentKey() {
        return this.currentKey;
    }

    AuthenticationKey getKey(int keyId) {
        return this.allKeys.get(keyId);
    }

    synchronized void removeExpiredKeys() {
        if (!this.leaderElector.isMaster()) {
            LOG.info("Skipping removeExpiredKeys() because not running as master.");
            return;
        }
        long now = EnvironmentEdgeManager.currentTime();
        Iterator<AuthenticationKey> iter = this.allKeys.values().iterator();
        while (iter.hasNext()) {
            AuthenticationKey key = iter.next();
            if (key.getExpiration() >= now) continue;
            LOG.debug("Removing expired key {}", (Object)key);
            iter.remove();
            this.zkWatcher.removeKeyFromZK(key);
        }
    }

    synchronized boolean isCurrentKeyRolled() {
        return this.currentKey != null;
    }

    synchronized void rollCurrentKey() {
        if (!this.leaderElector.isMaster()) {
            LOG.info("Skipping rollCurrentKey() because not running as master.");
            return;
        }
        long now = EnvironmentEdgeManager.currentTime();
        AuthenticationKey prev = this.currentKey;
        AuthenticationKey newKey = new AuthenticationKey(++this.idSeq, Long.MAX_VALUE, this.generateSecret());
        this.allKeys.put(newKey.getKeyId(), newKey);
        this.currentKey = newKey;
        this.zkWatcher.addKeyToZK(newKey);
        this.lastKeyUpdate = now;
        if (prev != null) {
            prev.setExpiration(now + this.tokenMaxLifetime);
            this.allKeys.put(prev.getKeyId(), prev);
            this.zkWatcher.updateKeyInZK(prev);
        }
    }

    synchronized long getLastKeyUpdate() {
        return this.lastKeyUpdate;
    }

    public static SecretKey createSecretKey(byte[] raw) {
        return SecretManager.createSecretKey((byte[])raw);
    }

    private class LeaderElector
    extends Thread
    implements Stoppable {
        private boolean stopped = false;
        private boolean isMaster = false;
        private ZKLeaderManager zkLeader;

        public LeaderElector(ZKWatcher watcher, String serverName) {
            this.setDaemon(true);
            this.setName("ZKSecretWatcher-leaderElector");
            this.zkLeader = new ZKLeaderManager(watcher, ZNodePaths.joinZNode(AuthenticationTokenSecretManager.this.zkWatcher.getRootKeyZNode(), "keymaster"), Bytes.toBytes(serverName), (Stoppable)this);
        }

        public boolean isMaster() {
            return this.isMaster;
        }

        @Override
        public boolean isStopped() {
            return this.stopped;
        }

        @Override
        public void stop(String reason) {
            if (this.stopped) {
                return;
            }
            this.stopped = true;
            if (this.isMaster) {
                this.zkLeader.stepDownAsLeader();
            }
            this.isMaster = false;
            LOG.info("Stopping leader election, because: " + reason);
            this.interrupt();
        }

        @Override
        public void run() {
            this.zkLeader.start();
            this.zkLeader.waitToBecomeLeader();
            this.isMaster = true;
            while (!this.stopped) {
                long now = EnvironmentEdgeManager.currentTime();
                AuthenticationTokenSecretManager.this.removeExpiredKeys();
                long localLastKeyUpdate = AuthenticationTokenSecretManager.this.getLastKeyUpdate();
                if (localLastKeyUpdate + AuthenticationTokenSecretManager.this.keyUpdateInterval < now) {
                    AuthenticationTokenSecretManager.this.rollCurrentKey();
                }
                try {
                    Thread.sleep(5000L);
                }
                catch (InterruptedException ie) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Interrupted waiting for next update", (Throwable)ie);
                }
            }
        }
    }
}

