/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.fs;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hudi.common.config.SerializableConfiguration;
import org.apache.hudi.common.fs.ConsistencyGuardConfig;
import org.apache.hudi.common.fs.FailSafeConsistencyGuard;
import org.apache.hudi.common.fs.HoodieWrapperFileSystem;
import org.apache.hudi.common.fs.NoOpConsistencyGuard;
import org.apache.hudi.common.model.HoodieFileFormat;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.exception.InvalidHoodiePathException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class FSUtils {
    private static final Logger LOG = LogManager.getLogger(FSUtils.class);
    private static final Pattern LOG_FILE_PATTERN = Pattern.compile("\\.(.*)_(.*)\\.(.*)\\.([0-9]*)(_(([0-9]*)-([0-9]*)-([0-9]*)))?");
    private static final String LOG_FILE_PREFIX = ".";
    private static final int MAX_ATTEMPTS_RECOVER_LEASE = 10;
    private static final long MIN_CLEAN_TO_KEEP = 10L;
    private static final long MIN_ROLLBACK_TO_KEEP = 10L;
    private static final String HOODIE_ENV_PROPS_PREFIX = "HOODIE_ENV_";
    private static final PathFilter ALLOW_ALL_FILTER = file -> true;

    public static Configuration prepareHadoopConf(Configuration conf) {
        conf.set("fs.hdfs.impl", DistributedFileSystem.class.getName());
        conf.set("fs.file.impl", LocalFileSystem.class.getName());
        for (Map.Entry<String, String> prop : System.getenv().entrySet()) {
            if (!prop.getKey().startsWith(HOODIE_ENV_PROPS_PREFIX)) continue;
            LOG.info("Picking up value for hoodie env var :" + prop.getKey());
            conf.set(prop.getKey().replace(HOODIE_ENV_PROPS_PREFIX, "").replaceAll("_DOT_", LOG_FILE_PREFIX), prop.getValue());
        }
        return conf;
    }

    public static FileSystem getFs(String path, Configuration conf) {
        FileSystem fs;
        FSUtils.prepareHadoopConf(conf);
        try {
            fs = new Path(path).getFileSystem(conf);
        }
        catch (IOException e) {
            throw new HoodieIOException("Failed to get instance of " + FileSystem.class.getName(), e);
        }
        LOG.info(String.format("Hadoop Configuration: fs.defaultFS: [%s], Config:[%s], FileSystem: [%s]", conf.getRaw("fs.defaultFS"), conf.toString(), fs.toString()));
        return fs;
    }

    public static FileSystem getFs(String path, Configuration conf, boolean localByDefault) {
        if (localByDefault) {
            return FSUtils.getFs(FSUtils.addSchemeIfLocalPath(path).toString(), conf);
        }
        return FSUtils.getFs(path, conf);
    }

    public static Path addSchemeIfLocalPath(String path) {
        Path providedPath = new Path(path);
        File localFile = new File(path);
        if (!providedPath.isAbsolute() && localFile.exists()) {
            Path resolvedPath = new Path("file://" + localFile.getAbsolutePath());
            LOG.info("Resolving file " + path + " to be a local file.");
            return resolvedPath;
        }
        LOG.info("Resolving file " + path + "to be a remote file.");
        return providedPath;
    }

    public static String makeWriteToken(int taskPartitionId, int stageId, long taskAttemptId) {
        return String.format("%d-%d-%d", taskPartitionId, stageId, taskAttemptId);
    }

    public static String makeDataFileName(String instantTime, String writeToken, String fileId) {
        return String.format("%s_%s_%s%s", fileId, writeToken, instantTime, HoodieFileFormat.PARQUET.getFileExtension());
    }

    public static String makeDataFileName(String instantTime, String writeToken, String fileId, String fileExtension) {
        return String.format("%s_%s_%s%s", fileId, writeToken, instantTime, fileExtension);
    }

    public static String makeBootstrapIndexFileName(String instantTime, String fileId, String ext) {
        return String.format("%s_%s_%s%s", fileId, "1-0-1", instantTime, ext);
    }

    public static String maskWithoutFileId(String instantTime, int taskPartitionId) {
        return String.format("*_%s_%s%s", taskPartitionId, instantTime, HoodieFileFormat.PARQUET.getFileExtension());
    }

    public static String getCommitFromCommitFile(String commitFileName) {
        return commitFileName.split("\\.")[0];
    }

    public static String getCommitTime(String fullFileName) {
        return fullFileName.split("_")[2].split("\\.")[0];
    }

    public static long getFileSize(FileSystem fs, Path path) throws IOException {
        return fs.getFileStatus(path).getLen();
    }

    public static String getFileId(String fullFileName) {
        return fullFileName.split("_")[0];
    }

    public static List<String> getAllPartitionFoldersThreeLevelsDown(FileSystem fs, String basePath) throws IOException {
        FileStatus[] folders;
        ArrayList<String> datePartitions = new ArrayList<String>();
        PathFilter filter = FSUtils.getExcludeMetaPathFilter();
        for (FileStatus status : folders = fs.globStatus(new Path(basePath + "/*/*/*"), filter)) {
            Path path = status.getPath();
            datePartitions.add(String.format("%s/%s/%s", path.getParent().getParent().getName(), path.getParent().getName(), path.getName()));
        }
        return datePartitions;
    }

    public static String getRelativePartitionPath(Path basePath, Path fullPartitionPath) {
        basePath = Path.getPathWithoutSchemeAndAuthority((Path)basePath);
        fullPartitionPath = Path.getPathWithoutSchemeAndAuthority((Path)fullPartitionPath);
        String fullPartitionPathStr = fullPartitionPath.toString();
        int partitionStartIndex = fullPartitionPathStr.indexOf(basePath.getName(), basePath.getParent() == null ? 0 : basePath.getParent().toString().length());
        return partitionStartIndex + basePath.getName().length() == fullPartitionPathStr.length() ? "" : fullPartitionPathStr.substring(partitionStartIndex + basePath.getName().length() + 1);
    }

    public static List<String> getAllFoldersWithPartitionMetaFile(FileSystem fs, String basePathStr) throws IOException {
        Path basePath = new Path(basePathStr);
        ArrayList<String> partitions = new ArrayList<String>();
        FSUtils.processFiles(fs, basePathStr, locatedFileStatus -> {
            Path filePath = locatedFileStatus.getPath();
            if (filePath.getName().equals(".hoodie_partition_metadata")) {
                partitions.add(FSUtils.getRelativePartitionPath(basePath, filePath.getParent()));
            }
            return true;
        }, true);
        return partitions;
    }

    public static void processFiles(FileSystem fs, String basePathStr, Function<FileStatus, Boolean> consumer, boolean excludeMetaFolder) throws IOException {
        FileStatus[] topLevelStatuses;
        PathFilter pathFilter = excludeMetaFolder ? FSUtils.getExcludeMetaPathFilter() : ALLOW_ALL_FILTER;
        for (FileStatus child : topLevelStatuses = fs.listStatus(new Path(basePathStr))) {
            if (child.isFile()) {
                boolean success = consumer.apply(child);
                if (success) continue;
                throw new HoodieException("Failed to process file-status=" + child);
            }
            if (!pathFilter.accept(child.getPath())) continue;
            RemoteIterator itr = fs.listFiles(child.getPath(), true);
            while (itr.hasNext()) {
                FileStatus status = (FileStatus)itr.next();
                boolean success = consumer.apply(status);
                if (success) continue;
                throw new HoodieException("Failed to process file-status=" + status);
            }
        }
    }

    public static List<String> getAllPartitionPaths(FileSystem fs, String basePathStr, boolean assumeDatePartitioning) throws IOException {
        if (assumeDatePartitioning) {
            return FSUtils.getAllPartitionFoldersThreeLevelsDown(fs, basePathStr);
        }
        return FSUtils.getAllFoldersWithPartitionMetaFile(fs, basePathStr);
    }

    public static String getFileExtension(String fullName) {
        Objects.requireNonNull(fullName);
        String fileName = new File(fullName).getName();
        int dotIndex = fileName.lastIndexOf(46);
        return dotIndex == -1 ? "" : fileName.substring(dotIndex);
    }

    private static PathFilter getExcludeMetaPathFilter() {
        return path -> !path.toString().contains(".hoodie");
    }

    public static String createNewFileIdPfx() {
        return UUID.randomUUID().toString();
    }

    public static String getFileExtensionFromLog(Path logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(logPath, "LogFile");
        }
        return matcher.group(3);
    }

    public static String getFileIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(1);
    }

    public static String getFileIdFromFilePath(Path filePath) {
        if (FSUtils.isLogFile(filePath)) {
            return FSUtils.getFileIdFromLogPath(filePath);
        }
        return FSUtils.getFileId(filePath.getName());
    }

    public static String getBaseCommitTimeFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(2);
    }

    public static Integer getTaskPartitionIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(7);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static String getWriteTokenFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        return matcher.group(6);
    }

    public static Integer getStageIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(8);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static Integer getTaskAttemptIdFromLogPath(Path path) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(path.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(path, "LogFile");
        }
        String val = matcher.group(9);
        return val == null ? null : Integer.valueOf(Integer.parseInt(val));
    }

    public static int getFileVersionFromLog(Path logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        if (!matcher.find()) {
            throw new InvalidHoodiePathException(logPath, "LogFile");
        }
        return Integer.parseInt(matcher.group(4));
    }

    public static String makeLogFileName(String fileId, String logFileExtension, String baseCommitTime, int version, String writeToken) {
        String suffix = writeToken == null ? String.format("%s_%s%s.%d", fileId, baseCommitTime, logFileExtension, version) : String.format("%s_%s%s.%d_%s", fileId, baseCommitTime, logFileExtension, version, writeToken);
        return LOG_FILE_PREFIX + suffix;
    }

    public static boolean isLogFile(Path logPath) {
        Matcher matcher = LOG_FILE_PATTERN.matcher(logPath.getName());
        return matcher.find() && logPath.getName().contains(".log");
    }

    public static Option<HoodieLogFile> getLatestLogFile(Stream<HoodieLogFile> logFiles) {
        return Option.fromJavaOptional(logFiles.min(HoodieLogFile.getReverseLogFileComparator()));
    }

    public static Stream<HoodieLogFile> getAllLogFiles(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        return Arrays.stream(fs.listStatus(partitionPath, path -> path.getName().startsWith(LOG_FILE_PREFIX + fileId) && path.getName().contains(logFileExtension))).map(HoodieLogFile::new).filter(s -> s.getBaseCommitTime().equals(baseCommitTime));
    }

    public static Option<Pair<Integer, String>> getLatestLogVersion(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        Option<HoodieLogFile> latestLogFile = FSUtils.getLatestLogFile(FSUtils.getAllLogFiles(fs, partitionPath, fileId, logFileExtension, baseCommitTime));
        if (latestLogFile.isPresent()) {
            return Option.of(Pair.of(latestLogFile.get().getLogVersion(), FSUtils.getWriteTokenFromLogPath(latestLogFile.get().getPath())));
        }
        return Option.empty();
    }

    public static int computeNextLogVersion(FileSystem fs, Path partitionPath, String fileId, String logFileExtension, String baseCommitTime) throws IOException {
        Option<Pair<Integer, String>> currentVersionWithWriteToken = FSUtils.getLatestLogVersion(fs, partitionPath, fileId, logFileExtension, baseCommitTime);
        return currentVersionWithWriteToken.isPresent() ? currentVersionWithWriteToken.get().getKey() + 1 : HoodieLogFile.LOGFILE_BASE_VERSION;
    }

    public static int getDefaultBufferSize(FileSystem fs) {
        return fs.getConf().getInt("io.file.buffer.size", 4096);
    }

    public static Short getDefaultReplication(FileSystem fs, Path path) {
        return fs.getDefaultReplication(path);
    }

    public static boolean recoverDFSFileLease(DistributedFileSystem dfs, Path p) throws IOException, InterruptedException {
        LOG.info("Recover lease on dfs file " + p);
        boolean recovered = false;
        for (int nbAttempt = 0; nbAttempt < 10; ++nbAttempt) {
            LOG.info("Attempt " + nbAttempt + " to recover lease on dfs file " + p);
            recovered = dfs.recoverLease(p);
            if (recovered) break;
            Thread.sleep(1000L);
        }
        return recovered;
    }

    public static void deleteOlderCleanMetaFiles(FileSystem fs, String metaPath, Stream<HoodieInstant> instants) {
        instants.skip(10L).forEach(s -> {
            try {
                fs.delete(new Path(metaPath, s.getFileName()), false);
            }
            catch (IOException e) {
                throw new HoodieIOException("Could not delete clean meta files" + s.getFileName(), e);
            }
        });
    }

    public static void deleteOlderRollbackMetaFiles(FileSystem fs, String metaPath, Stream<HoodieInstant> instants) {
        instants.skip(10L).forEach(s -> {
            try {
                fs.delete(new Path(metaPath, s.getFileName()), false);
            }
            catch (IOException e) {
                throw new HoodieIOException("Could not delete rollback meta files " + s.getFileName(), e);
            }
        });
    }

    public static void deleteInstantFile(FileSystem fs, String metaPath, HoodieInstant instant) {
        try {
            LOG.warn("try to delete instant file: " + instant);
            fs.delete(new Path(metaPath, instant.getFileName()), false);
        }
        catch (IOException e) {
            throw new HoodieIOException("Could not delete instant file" + instant.getFileName(), e);
        }
    }

    public static void createPathIfNotExists(FileSystem fs, Path partitionPath) throws IOException {
        if (!fs.exists(partitionPath)) {
            fs.mkdirs(partitionPath);
        }
    }

    public static Long getSizeInMB(long sizeInBytes) {
        return sizeInBytes / 0x100000L;
    }

    public static Path getPartitionPath(String basePath, String partitionPath) {
        return FSUtils.getPartitionPath(new Path(basePath), partitionPath);
    }

    public static Path getPartitionPath(Path basePath, String partitionPath) {
        return partitionPath == null || partitionPath.isEmpty() ? basePath : new Path(basePath, partitionPath);
    }

    public static String getDFSFullPartitionPath(FileSystem fs, Path fullPartitionPath) {
        return fs.getUri() + fullPartitionPath.toUri().getRawPath();
    }

    public static boolean isGCSInputStream(FSDataInputStream inputStream) {
        return inputStream.getClass().getCanonicalName().equals("com.google.cloud.hadoop.fs.gcs.GoogleHadoopFSInputStream") || inputStream.getWrappedStream().getClass().getCanonicalName().equals("com.google.cloud.hadoop.fs.gcs.GoogleHadoopFSInputStream");
    }

    public static Configuration registerFileSystem(Path file, Configuration conf) {
        Configuration returnConf = new Configuration(conf);
        String scheme = FSUtils.getFs(file.toString(), conf).getScheme();
        returnConf.set("fs." + HoodieWrapperFileSystem.getHoodieScheme(scheme) + ".impl", HoodieWrapperFileSystem.class.getName());
        return returnConf;
    }

    public static HoodieWrapperFileSystem getFs(String path, SerializableConfiguration hadoopConf, ConsistencyGuardConfig consistencyGuardConfig) {
        FileSystem fileSystem = FSUtils.getFs(path, hadoopConf.newCopy());
        return new HoodieWrapperFileSystem(fileSystem, consistencyGuardConfig.isConsistencyCheckEnabled() ? new FailSafeConsistencyGuard(fileSystem, consistencyGuardConfig) : new NoOpConsistencyGuard());
    }

    public static List<FileStatus> getGlobStatusExcludingMetaFolder(FileSystem fs, Path globPath) throws IOException {
        FileStatus[] statuses = fs.globStatus(globPath);
        return Arrays.stream(statuses).filter(fileStatus -> !fileStatus.getPath().toString().contains(".hoodie")).collect(Collectors.toList());
    }
}

