/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.table.log;

import java.io.IOException;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.hdfs.protocol.RecoveryInProgressException;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.fs.StorageSchemes;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.HoodieLogFormatVersion;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class HoodieLogFormatWriter
implements HoodieLogFormat.Writer {
    private static final Logger LOG = LogManager.getLogger(HoodieLogFormatWriter.class);
    private HoodieLogFile logFile;
    private final FileSystem fs;
    private final long sizeThreshold;
    private final Integer bufferSize;
    private final Short replication;
    private final String logWriteToken;
    private final String rolloverLogWriteToken;
    private FSDataOutputStream output;
    private boolean closed = false;
    private static final String APPEND_UNAVAILABLE_EXCEPTION_MESSAGE = "not sufficiently replicated yet";

    HoodieLogFormatWriter(FileSystem fs, HoodieLogFile logFile, Integer bufferSize, Short replication, Long sizeThreshold, String logWriteToken, String rolloverLogWriteToken) {
        this.fs = fs;
        this.logFile = logFile;
        this.sizeThreshold = sizeThreshold;
        this.bufferSize = bufferSize;
        this.replication = replication;
        this.logWriteToken = logWriteToken;
        this.rolloverLogWriteToken = rolloverLogWriteToken;
        this.addShutDownHook();
    }

    public FileSystem getFs() {
        return this.fs;
    }

    @Override
    public HoodieLogFile getLogFile() {
        return this.logFile;
    }

    public long getSizeThreshold() {
        return this.sizeThreshold;
    }

    private FSDataOutputStream getOutputStream() throws IOException, InterruptedException {
        if (this.output == null) {
            Path path = this.logFile.getPath();
            if (this.fs.exists(path)) {
                boolean isAppendSupported = StorageSchemes.isAppendSupported(this.fs.getScheme());
                if (isAppendSupported) {
                    LOG.info(this.logFile + " exists. Appending to existing file");
                    try {
                        this.output = this.fs.append(path, this.bufferSize.intValue());
                    }
                    catch (RemoteException e) {
                        LOG.warn("Remote Exception, attempting to handle or recover lease", e);
                        this.handleAppendExceptionOrRecoverLease(path, e);
                    }
                    catch (IOException ioe) {
                        if (ioe.getMessage().toLowerCase().contains("not supported")) {
                            isAppendSupported = false;
                        }
                        this.close();
                        throw ioe;
                    }
                }
                if (!isAppendSupported) {
                    this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
                    LOG.info("Append not supported.. Rolling over to " + this.logFile);
                    this.createNewFile();
                }
            } else {
                LOG.info(this.logFile + " does not exist. Create a new file");
                this.createNewFile();
            }
        }
        return this.output;
    }

    @Override
    public HoodieLogFormat.Writer appendBlock(HoodieLogBlock block) throws IOException, InterruptedException {
        HoodieLogFormatVersion currentLogFormatVersion = new HoodieLogFormatVersion(1);
        FSDataOutputStream outputStream = this.getOutputStream();
        long currentSize = outputStream.size();
        outputStream.write(HoodieLogFormat.MAGIC);
        byte[] headerBytes = HoodieLogBlock.getLogMetadataBytes(block.getLogBlockHeader());
        byte[] content = block.getContentBytes();
        byte[] footerBytes = HoodieLogBlock.getLogMetadataBytes(block.getLogBlockFooter());
        outputStream.writeLong((long)this.getLogBlockLength(content.length, headerBytes.length, footerBytes.length));
        outputStream.writeInt(currentLogFormatVersion.getVersion());
        outputStream.writeInt(block.getBlockType().ordinal());
        outputStream.write(headerBytes);
        outputStream.writeLong((long)content.length);
        outputStream.write(content);
        outputStream.write(footerBytes);
        outputStream.writeLong((long)outputStream.size() - currentSize);
        this.flush();
        return this.rolloverIfNeeded();
    }

    private int getLogBlockLength(int contentLength, int headerLength, int footerLength) {
        return 8 + headerLength + 8 + contentLength + footerLength + 8;
    }

    private HoodieLogFormat.Writer rolloverIfNeeded() throws IOException, InterruptedException {
        if (this.getCurrentSize() > this.sizeThreshold) {
            LOG.info("CurrentSize " + this.getCurrentSize() + " has reached threshold " + this.sizeThreshold + ". Rolling over to the next version");
            HoodieLogFile newLogFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.close();
            return new HoodieLogFormatWriter(this.fs, newLogFile, this.bufferSize, this.replication, this.sizeThreshold, this.logWriteToken, this.rolloverLogWriteToken);
        }
        return this;
    }

    private void createNewFile() throws IOException {
        this.output = this.fs.create(this.logFile.getPath(), false, this.bufferSize.intValue(), this.replication.shortValue(), 0x20000000L, null);
    }

    @Override
    public void close() throws IOException {
        if (this.output != null) {
            this.flush();
            this.output.close();
            this.output = null;
            this.closed = true;
        }
    }

    private void flush() throws IOException {
        if (this.output == null) {
            return;
        }
        this.output.flush();
        this.output.hsync();
    }

    @Override
    public long getCurrentSize() throws IOException {
        if (this.closed) {
            throw new IllegalStateException("Cannot get current size as the underlying stream has been closed already");
        }
        if (this.output == null) {
            return 0L;
        }
        return this.output.getPos();
    }

    private void addShutDownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    if (HoodieLogFormatWriter.this.output != null) {
                        HoodieLogFormatWriter.this.close();
                    }
                }
                catch (Exception e) {
                    LOG.warn("unable to close output stream for log file " + HoodieLogFormatWriter.this.logFile, e);
                }
            }
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void handleAppendExceptionOrRecoverLease(Path path, RemoteException e) throws IOException, InterruptedException {
        if (e.getMessage().contains(APPEND_UNAVAILABLE_EXCEPTION_MESSAGE)) {
            LOG.warn("Failed to open an append stream to the log file. Opening a new log file..", e);
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(AlreadyBeingCreatedException.class.getName())) {
            LOG.warn("Another task executor writing to the same log file(" + this.logFile + ". Rolling over");
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
            return;
        }
        if (e.getClassName().contentEquals(RecoveryInProgressException.class.getName()) && this.fs instanceof DistributedFileSystem) {
            LOG.warn("Trying to recover log on path " + path);
            if (FSUtils.recoverDFSFileLease((DistributedFileSystem)this.fs, path)) {
                LOG.warn("Recovered lease on path " + path);
                this.output = this.fs.append(path, this.bufferSize.intValue());
                return;
            }
            LOG.warn("Failed to recover lease on path " + path);
            throw new HoodieException(e);
        }
        try {
            this.close();
            throw new HoodieIOException("Failed to append to the output stream ", (IOException)((Object)e));
        }
        catch (Exception ce) {
            LOG.warn("Failed to close the output stream for " + this.fs.getClass().getName() + " on path " + path + ". Rolling over to a new log file.");
            this.logFile = this.logFile.rollOver(this.fs, this.rolloverLogWriteToken);
            this.createNewFile();
        }
    }
}

